from velocity.misc.format import to_json
import json
import sys
import os
import traceback
from velocity.aws import DEBUG
from support.app import helpers, AlertError, enqueue
from response import Response

class LambdaHandler:
    def __init__(self, event, context):
        self.event = event
        self.context = context
        self.serve_action_default = True
        self.skip_action = False

        requestContext = event.get("requestContext") or {}
        identity = requestContext.get("identity") or {}
        headers = event.get("headers") or {}
        auth = identity.get("cognitoAuthenticationProvider")
        self.session = {
            "authentication_provider": identity.get("cognitoAuthenticationProvider"),
            "authentication_type": identity.get("cognitoAuthenticationType"),
            "cognito_user": identity.get("user"),
            "is_desktop": headers.get("CloudFront-Is-Desktop-Viewer") == "true",
            "is_mobile": headers.get("CloudFront-Is-Mobile-Viewer") == "true",
            "is_smart_tv": headers.get("CloudFront-Is-SmartTV-Viewer") == "true",
            "is_tablet": headers.get("CloudFront-Is-Tablet-Viewer") == "true",
            "origin": headers.get("origin"),
            "path": event.get("path"),
            "referer": headers.get("Referer"),
            "source_ip": identity.get("sourceIp"),
            "user_agent": identity.get("userAgent"),
            "sub": auth.split(":")[-1] if auth else None,
        }
        if self.session.get("is_mobile"):
            self.session["device_type"] = "mobile"
        elif self.session.get("is_desktop"):
            self.session["device_type"] = "desktop"
        elif self.session.get("is_tablet"):
            self.session["device_type"] = "tablet"
        elif self.session.get("is_smart_tv"):
            self.session["device_type"] = "smart_tv"
        else:
            self.session["device_type"] = "unknown"

    def log(self, tx, message, function=None):
        if not function:
            function = "<Unknown>"
            idx = 0
            while True:
                try:
                    temp = sys._getframe(idx).f_code.co_name
                except ValueError as e:
                    break
                if temp in ["x", "log", "_transaction"]:
                    idx += 1
                    continue
                function = temp
                break

        data = {
            "app_name": os.environ["ProjectName"],
            "source_ip": self.session["source_ip"],
            "referer": self.session["referer"],
            "user_agent": self.session["user_agent"],
            "device_type": self.session["device_type"],
            "function": function,
            "message": message,
        }
        if "email_address" in self.session:
            data["sys_modified_by"] = self.session["email_address"]
        tx.table("sys_log").insert(data)

    def serve(self, tx):
        response = Response()
        body = self.event.get("body")
        postdata = {}
        if isinstance(body, str) and len(body) > 0:
            try:
                postdata = json.loads(body)
            except:
                postdata = {"raw_body": body}
        elif isinstance(body, dict):
            postdata = body
        elif isinstance(body, list) and len(body) > 0:
            try:
                new = "\n".join(body)
                postdata = json.loads(new)
            except:
                postdata = {"raw_body": body}

        req_params = self.event.get("queryStringParameters") or {}
        try:
            if hasattr(self, "beforeAction"):
                self.beforeAction(args=req_params, postdata=postdata, response=response)
            actions = []
            action = postdata.get("action", req_params.get("action"))
            if action:
                actions.append(
                    f"on action {action.replace('-', ' ').replace('_', ' ')}".title().replace(
                        " ", ""
                    )
                )
            if self.serve_action_default:
                actions.append("OnActionDefault")
            for action in actions:
                if self.skip_action:
                    break
                if hasattr(self, action):
                    result = getattr(self, action)(
                        args=req_params, postdata=postdata, response=response
                    )
                    if result and result != response:
                        response.set_body(result)
                    break
            if hasattr(self, "afterAction"):
                self.afterAction(args=req_params, postdata=postdata, response=response)
        except AlertError as e:
            response.alert(e.get_payload())
        except Exception as e:
            response.exception()
            if hasattr(self, "onError"):
                self.onError(
                    args=req_params,
                    postdata=postdata,
                    response=response,
                    exc=e.__class__.__name__,
                    tb=traceback.format_exc(),
                )

        return response.render()

    def track(self, tx, data={}, user=None):
        data = data.copy()
        data.update(
            {
                "source_ip": self.session["source_ip"],
                "referer": self.session["referer"],
                "user_agent": self.session["user_agent"],
                "device_type": self.session["device_type"],
                "sys_modified_by": self.session["email_address"],
            }
        )
        tx.table(helpers.get_tracking_table(user or self.session)).insert(data)

    def OnActionDefault(self, tx, args, postdata, response):
        return {"event": self.event, "postdata": postdata}

    def OnActionTracking(self, tx, args, postdata, response):
        self.track(tx, postdata.get("payload", {}).get("data", {}))

    def enqueue(self, tx, action, payload={}):
        return enqueue(tx, action, payload, self.session["email_address"])