"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MysqlDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const triggers_1 = require("aws-cdk-lib/triggers");
const base_database_1 = require("./base-database");
const HANDLER_PATH = path.join(__dirname, "../../assets/rds/triggers/mysql");
// https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.tls
const AMAZON_ROOT_CA1 = "https://www.amazontrust.com/repository/AmazonRootCA1.pem";
/**
 * A MySQL database.
 */
class MysqlDatabase extends base_database_1.BaseDatabase {
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forCluster(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            vpc: cluster.vpc,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forClusterFromSnapshot(scope, id, cluster, options) {
        // The DatabaseClusterFromSnapshot type is equivalent to DatabaseCluster.
        return MysqlDatabase.forCluster(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessCluster(scope, id, cluster, options) {
        // The ServerlessClusterFromSnapshot type is a subset of ServerlessCluster.
        return MysqlDatabase.forServerlessClusterFromSnapshot(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessClusterFromSnapshot(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            certificateAuthoritiesUrl: AMAZON_ROOT_CA1,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstance.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstance(scope, id, instance, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || instance.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: instance,
            endpoint: instance.instanceEndpoint,
            adminSecret: clusterSecret,
            vpc: instance.vpc,
            ...props,
        });
        database.trigger.executeAfter(instance);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstanceFromSnapshot.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstanceFromSnapshot(scope, id, instance, options) {
        // The DatabaseInstanceFromSnapshot type is equivalent to DatabaseInstance.
        return MysqlDatabase.forInstance(scope, id, instance, options);
    }
    /**
     * Creates a new MysqlDatabase.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param props - The configuration properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.ownerSecrets = [];
        this.readerSecrets = [];
        this.unprivilegedSecrets = [];
        const { adminSecret, vpc, characterSet = "utf8mb4" } = props;
        const environment = {
            AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
            ADMIN_SECRET_ARN: adminSecret.secretArn,
            OWNER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.ownerSecrets.map((s) => s.secretArn)),
            }),
            READER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.readerSecrets.map((s) => s.secretArn)),
            }),
            UNPRIVILEGED_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.unprivilegedSecrets.map((s) => s.secretArn)),
            }),
            DB_NAME: this.databaseName,
            DB_CHARACTER_SET: characterSet,
        };
        if (props.collation) {
            environment.DB_COLLATION = props.collation;
        }
        if (props.certificateAuthoritiesUrl) {
            environment.CA_CERTS_URL = props.certificateAuthoritiesUrl;
        }
        this.lambdaFunction = new aws_lambda_1.Function(this, "Function", {
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            handler: "index.handler",
            description: "Creates a schema and possibly some secondary users",
            vpc,
            vpcSubnets: this.subnetSelection,
            securityGroups: [this.securityGroup],
            environment,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        adminSecret.grantRead(this.lambdaFunction);
        this.trigger = new triggers_1.Trigger(this, "Trigger", {
            handler: this.lambdaFunction,
        });
        this.trigger.executeAfter(this.lambdaFunction);
    }
    addUserAsOwner(secret) {
        this.ownerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsReader(secret) {
        this.readerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsUnprivileged(secret) {
        this.unprivilegedSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
}
exports.MysqlDatabase = MysqlDatabase;
_a = JSII_RTTI_SYMBOL_1;
MysqlDatabase[_a] = { fqn: "shady-island.MysqlDatabase", version: "0.1.37" };
//# sourceMappingURL=data:application/json;base64,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