"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionsBoundary = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const iam_generated_1 = require("./iam.generated");
/**
 * Modify the Permissions Boundaries of Users and Roles in a construct tree.
 *
 * @stability stable
 * @example
 *
 * const policy = ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess');
 * PermissionsBoundary.of(stack).apply(policy);
 */
class PermissionsBoundary {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Access the Permissions Boundaries of a construct tree.
     *
     * @stability stable
     */
    static of(scope) {
        return new PermissionsBoundary(scope);
    }
    /**
     * Apply the given policy as Permissions Boundary to all Roles and Users in the scope.
     *
     * Will override any Permissions Boundaries configured previously; in case
     * a Permission Boundary is applied in multiple scopes, the Boundary applied
     * closest to the Role wins.
     *
     * @stability stable
     */
    apply(boundaryPolicy) {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyOverride('PermissionsBoundary', boundaryPolicy.managedPolicyArn);
                }
            },
        });
    }
    /**
     * Remove previously applied Permissions Boundaries.
     *
     * @stability stable
     */
    clear() {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyDeletionOverride('PermissionsBoundary');
                }
            },
        });
    }
}
exports.PermissionsBoundary = PermissionsBoundary;
_a = JSII_RTTI_SYMBOL_1;
PermissionsBoundary[_a] = { fqn: "@aws-cdk/aws-iam.PermissionsBoundary", version: "1.122.0" };
//# sourceMappingURL=data:application/json;base64,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