from string import Template
from pathlib import Path
from typing import Optional, Union, NamedTuple
import re
import typer
import yaml
import json
import nbformat

from nbformat.notebooknode import NotebookNode  # type: ignore


NOTEBOOK_METADATA_CELL_TEMPLATE = Template(
    "# This cell is auto-generated by Garden. Don't delete it. Do keep it as the first cell.\n"
    "# It records the base image and requirements you passed to `garden-ai notebook start`.\n"
    "# That way the next time you run this notebook Garden can start it with the same libraries.\n\n"
    '"""\n'
    "NOTEBOOK_METADATA = $metadata\n"
    '"""'
)


class RequirementsData(NamedTuple):
    file_format: str
    contents: Union[dict, list]


class NotebookMetadata(NamedTuple):
    notebook_image_name: Optional[str]
    notebook_image_uri: Optional[str]
    notebook_requirements: Optional[RequirementsData]


def add_notebook_metadata_cell(
    notebook_path: Path,
):
    ntbk = _read_notebook(notebook_path)

    # Find cell with 'garden_metadata_cell' tag
    for cell in ntbk.cells:
        cell_tags = cell.get("metadata", {}).get("tags", [])
        if "garden_metadata_cell" in cell_tags:
            # if cell exists, exit here, don't need to add again.
            return

    # Was unable to find cell with garden_metadata_cell tag, add new one to top of notebook
    # notebook_image_uri can be None here since notebook start will require the user provided a base image.
    metadata_string = json.dumps(
        {"NOTEBOOK_BASE_IMAGE_NAME": None, "NOTEBOOK_REQUIREMENTS": None}, indent=2
    )
    notebook_metadata = NOTEBOOK_METADATA_CELL_TEMPLATE.substitute(
        metadata=metadata_string
    )

    # Add new cell with garden_metadata_cell tag to top of notebook
    new_cell = nbformat.v4.new_code_cell(notebook_metadata)
    new_cell["metadata"] = {
        "tags": ["garden_metadata_cell"],
        "garden_metadata": {"NOTEBOOK_BASE_IMAGE_URI": None},
    }
    del new_cell["id"]
    ntbk.cells.insert(0, new_cell)

    # Write updated notebook data to file
    nbformat.write(ntbk, notebook_path, version=nbformat.NO_CONVERT)


def get_notebook_metadata(notebook_path: Path) -> NotebookMetadata:
    notebook_metadata_cell_source = None

    ntbk = _read_notebook(notebook_path)

    # Find cell with 'garden_metadata_cell' tag and get source
    for cell in ntbk.cells:
        cell_tags = cell.get("metadata", {}).get("tags", [])
        if "garden_metadata_cell" in cell_tags:
            # Grab cell source and hidden metadata
            notebook_metadata_cell_source = cell.get("source", None)
            notebook_hidden_metadata = cell.get("metadata", {}).get(
                "garden_metadata", {}
            )
            break

    # Return empty notebook metadata dict if was unable to find cell source
    if not notebook_metadata_cell_source:
        typer.echo("Unable to find garden metadata cell.")
        return NotebookMetadata(None, None, None)

    # Grab the part of cell source with the metadata dict in it
    clean_source = notebook_metadata_cell_source.replace("\n", "")
    regex_get_metadata = r"^(.*?)\"\"\"(.*?)NOTEBOOK_METADATA(.*?)=(.*?)\"\"\"(.*?)$"

    metadata_match = re.match(regex_get_metadata, clean_source)
    if metadata_match and len(metadata_match.groups()) == 5:
        notebook_metadata_dict = json.loads(metadata_match[4].strip())

        notebook_image_name = notebook_metadata_dict.get(
            "NOTEBOOK_BASE_IMAGE_NAME", None
        )

        notebook_image_uri = notebook_hidden_metadata.get(
            "NOTEBOOK_BASE_IMAGE_URI", None
        )

        notebook_requirements_dict = notebook_metadata_dict.get(
            "NOTEBOOK_REQUIREMENTS", None
        )
        notebook_requirements = None
        if notebook_requirements_dict:
            notebook_requirements = RequirementsData(**notebook_requirements_dict)

        return NotebookMetadata(
            notebook_image_name, notebook_image_uri, notebook_requirements
        )
    else:
        typer.echo("Unable to parse garden metadata cell.")
        return NotebookMetadata(None, None, None)


def set_notebook_metadata(
    notebook_path: Path,
    base_image_name: Optional[str],
    base_image_uri: str,
    requirements_data: Optional[RequirementsData],
):
    ntbk = _read_notebook(notebook_path)

    # Find cell with 'garden_metadata_cell' tag
    for cell in ntbk.cells:
        cell_tags = cell.get("metadata", {}).get("tags", [])
        if "garden_metadata_cell" in cell_tags:
            # Replace old cell source with new metadata
            requirements_data_dict = (
                requirements_data._asdict() if requirements_data is not None else None
            )
            metadata_string = json.dumps(
                {
                    "NOTEBOOK_BASE_IMAGE_NAME": base_image_name,
                    "NOTEBOOK_REQUIREMENTS": requirements_data_dict,
                },
                indent=2,
            )

            cell["source"] = NOTEBOOK_METADATA_CELL_TEMPLATE.substitute(
                metadata=metadata_string
            )

            # Add hidden metadata
            cell["metadata"]["garden_metadata"][
                "NOTEBOOK_BASE_IMAGE_URI"
            ] = base_image_uri

    # Write updated notebook data to file
    nbformat.write(ntbk, notebook_path, version=nbformat.NO_CONVERT)


def read_requirements_data(requirements_path: Path) -> Optional[RequirementsData]:
    # For txt requirements files, contents is list of lines, format is pip
    if requirements_path.suffix in {".txt"}:
        file_format = "pip"
        with open(requirements_path, "r") as req_file:
            # read lines into list and strip any newlines
            file_contents = [line.replace("\n", "") for line in req_file.readlines()]
            req_file.close()
        return RequirementsData(file_format, file_contents)
    # For yaml requirements files, contents is safe_load dict of yaml file, format is conda
    elif requirements_path.suffix in {".yml", ".yaml"}:
        file_format = "conda"
        with open(requirements_path, "r") as req_file:
            file_contents = yaml.safe_load(req_file)
            req_file.close()
        return RequirementsData(file_format, file_contents)
    else:
        typer.echo("Invalid requirements file format.")
        return None


def save_requirements_data(
    requirements_dir_path: Path, requirements_data: RequirementsData
) -> Optional[Path]:
    # Save requirements_data to requirements file in either pip or conda format
    # Returns path to new requirements file or None if was unable to write.
    file_format = requirements_data.file_format
    contents = requirements_data.contents

    if file_format == "pip":
        # requirements file is txt
        requirements_path = requirements_dir_path / "requirements.txt"
        with open(requirements_path, "w") as req_file:
            # contents is list of requirements
            file_contents = ""
            for line in contents:
                file_contents += f"{line}\n"
            req_file.write(file_contents)
            req_file.close()
        return requirements_path

    elif file_format == "conda":
        # requirements file is yml
        requirements_path = requirements_dir_path / "requirements.yml"
        with open(requirements_path, "w") as req_file:
            # contents is dict of yaml requirements
            yaml.dump(contents, req_file, allow_unicode=True)
            req_file.close()
        return requirements_path
    else:
        typer.echo(
            f"Invalid format for requirements data, must be either pip or conda, got {file_format}. Ignoring requirements."
        )
        req_file.close()
        return None


def _read_notebook(notebook_path: Path) -> NotebookNode:
    # Read notebook contents with nbformat
    try:
        ntbk = nbformat.read(notebook_path, as_version=4)
        return ntbk
    except ValueError:
        typer.echo(f"Unable to parse notebook: {notebook_path}")
        raise typer.Exit(1)
