from typing import Annotated, Any, Literal, TypedDict

import gdsfactory as gf
import gdsfactory.pdk as gpdk
import kfactory as kf
import numpy as np
from gdsfactory.typings import ComponentSpec as ComponentSpec
from kfactory.kcell import LayerEnum as LayerEnum

Int = int | np.int_
Float = float | np.float64
Number = Int | Float
Wgu = Annotated[Int, "value in 'wg width units'"]
Dbu = Annotated[Int, "value in database units"]
Um = Annotated[Float, "value in micrometer"]
PositionUm = Annotated[tuple[Um, Um], "x, y position in micrometer"]
PositionDbu = Annotated[tuple[Dbu, Dbu], "x, y position in database units"]
PositionAny = PositionUm | PositionDbu
OrientationChar = Literal["n", "e", "s", "w", "o"]
OrientationWord = Literal["north", "east", "south", "west", "none"]
OrientationDegree = Literal[0, 90, 180, 270, -90]
PositionWithDirectionUm = Annotated[
    tuple[Um, Um, OrientationChar], "x[um] y[um] d[neswo]"
]
PositionWithDirectionDbu = Annotated[
    tuple[Dbu, Dbu, OrientationChar], "x[dbu] y[dbu] d[n|e|s|w|o]"
]
OrientationLike = (
    OrientationChar | OrientationWord | OrientationDegree | Int | Float | None
)
PortLike = PositionDbu | tuple[Dbu, Dbu, OrientationLike] | gf.Port | kf.Port
OrientationTransition = tuple[OrientationChar, OrientationChar]
LayerLike = tuple[Int, Int] | LayerEnum | str
Layer = tuple[int, int]
PointsDbu = list[PositionDbu]
PointsWgu = list[tuple[Wgu, Wgu]]
PointsUm = list[tuple[Float, Float]]
DirectivePointsDbu = list[PositionWithDirectionDbu]


class StepDbu(TypedDict):
    x: Dbu | None
    dx: Dbu | None
    y: Dbu | None
    dy: Dbu | None


def validate_layer(layer: Any) -> Layer:
    if isinstance(layer, LayerEnum):
        return (layer.layer, layer.datatype)
    elif isinstance(layer, str):
        layer_tuple = gpdk.get_layer_tuple(layer)
        return (int(layer_tuple[0]), int(layer_tuple[1]))
    else:
        layer_int = _try_int(layer)
        if layer_int is not None:
            return (layer_int, 0)
        layer_tuple = tuple(layer)
        return (int(layer_tuple[0]), int(layer_tuple[1]))


def validate_position(p: Any) -> PositionDbu:
    if isinstance(p, gf.Port | kf.Port):
        x, y = p.center
        return (x, y)
    try:
        p = tuple(p)
    except Exception as e:
        raise ValueError(f"Unable to parse {p} as position") from e
    match p:
        case (x, y):
            return (x, y)
        case (x, y, _):
            return (x, y)
    raise ValueError(f"Unable to parse {p} as position")


def validate_position_with_orientation(
    p: Any, invert_orientation=False
) -> PositionWithDirectionDbu:
    from . import util

    def maybe_invert_orientation(orientation):
        if invert_orientation:
            return util.invert_orientation(orientation)
        else:
            return orientation

    if isinstance(p, gf.Port | kf.Port):
        x, y = p.center
        o = validate_orientation(p.orientation)
        return (int(x), int(y), maybe_invert_orientation(o))
    try:
        p = tuple(p)
    except Exception as e:
        raise ValueError(f"Unable to parse {p} as position with orientation") from e
    match p:
        case (x, y):
            return (int(x), int(y), maybe_invert_orientation("o"))
        case (x, y, o):
            return (int(x), int(y), maybe_invert_orientation(validate_orientation(o)))
    raise ValueError(f"Unable to parse {p} as position with orientation")


def validate_orientation(o: Any) -> OrientationChar:
    given = o

    if given is None:
        return "o"

    if isinstance(given, str):
        o = o.lower().strip()
        if o == "none":
            return "o"
        o = o[:1]
        if o in "nesw":
            return o
        # if none of these cases match, try to validate as integer below:

    o = _try_int(given)

    if o is None:
        raise ValueError(f"Could not parse orientation from {given}.")

    o = o % 360
    match o:
        case 0:
            return "e"
        case 90:
            return "n"
        case 180:
            return "w"
        case 270:
            return "s"

    raise ValueError(f"Could not parse orientation from {given}.")


def _try_int(s: str) -> int | None:
    try:
        return int(s)
    except Exception:
        return None
