"""Basic TelemetryFlow SDK usage example.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Compatible with TFO-Collector v${tfo_collector_version} (OCB-native)
SDK Version: ${sdk_version}

Auto-generated by telemetryflow-gen.

This example demonstrates:
- TFO v2 API initialization
- Recording metrics (counters, gauges, histograms)
- Emitting logs (info, warn, error)
- Creating trace spans with events
"""

import time

from telemetry import init, init_v2_only, get_client, SpanKind


def main() -> None:
    """Main function demonstrating SDK usage with TFO v2 API."""
    # Initialize the SDK with TFO v2 API (default)
    # For v2-only mode (production), use: init_v2_only()
    init()
    client = get_client()
    print("TelemetryFlow SDK initialized with TFO v2 API!")
    print(f"Service: ${service_name} v${service_version}")

    try:
        # ============================================
        # METRICS EXAMPLES
        # ============================================
        print("\n--- Recording Metrics ---")

        # Increment a counter - great for tracking events
        client.increment_counter(
            "app.requests.total",
            attributes={"endpoint": "/api/users", "method": "GET"}
        )
        print("Recorded counter: app.requests.total")

        # Record a gauge - great for current values
        client.record_gauge(
            "app.active_connections",
            42.0,
            attributes={"pool": "default"}
        )
        print("Recorded gauge: app.active_connections = 42")

        # Record a histogram - great for distributions
        client.record_histogram(
            "app.request.duration",
            0.125,
            unit="s",
            attributes={"endpoint": "/api/users"}
        )
        print("Recorded histogram: app.request.duration = 0.125s")

        # ============================================
        # LOGS EXAMPLES
        # ============================================
        print("\n--- Emitting Logs ---")

        # Info level log with TFO v2 API metadata
        client.log_info(
            "Application started successfully",
            {
                "version": "${service_version}",
                "environment": "${environment}",
                "tfo_api_version": "v2",
            }
        )
        print("Logged: INFO - Application started")

        # Warning level log
        client.log_warn(
            "Cache miss rate is high",
            {"cache_hit_rate": 0.65, "threshold": 0.80}
        )
        print("Logged: WARN - Cache miss rate")

        # Error level log
        client.log_error(
            "Failed to connect to database",
            {"database": "users_db", "retry_count": 3}
        )
        print("Logged: ERROR - Database connection failed")

        # ============================================
        # TRACES EXAMPLES
        # ============================================
        print("\n--- Creating Traces ---")

        # Using context manager for automatic span management
        with client.span("process_request", SpanKind.SERVER) as request_span:
            print(f"Processing request (span: {request_span})")
            time.sleep(0.05)

            # Add TFO v2 API context to span events
            client.add_span_event(
                request_span,
                "request_received",
                {"tfo_api_version": "v2", "endpoint": "/api/users"}
            )

            # Nested span for database operation
            with client.span("database_query", SpanKind.CLIENT) as db_span:
                print(f"Executing database query (span: {db_span})")
                time.sleep(0.03)
                client.add_span_event(db_span, "query_executed", {"rows_returned": 10})

            # Nested span for cache operation
            with client.span("cache_lookup", SpanKind.CLIENT) as cache_span:
                print(f"Looking up cache (span: {cache_span})")
                time.sleep(0.01)
                client.add_span_event(cache_span, "cache_hit", {"key": "user:123"})

            client.add_span_event(request_span, "processing_complete")

        # ============================================
        # FINAL STATUS
        # ============================================
        print("\n--- Final Status ---")
        status = client.get_status()
        print(f"Metrics sent: {status['metrics_sent']}")
        print(f"Logs sent: {status['logs_sent']}")
        print(f"Spans sent: {status['spans_sent']}")
        print(f"Status: {status}")

    except Exception as e:
        client.log_error(f"Unexpected error: {e}")
        raise


if __name__ == "__main__":
    main()
