import pydantic
import sympy

from classiq.interface.generator import function_params
from classiq.interface.generator.arith.register_user_input import RegisterUserInput
from classiq.interface.generator.parameters import ParameterFloatType
from classiq.interface.generator.standard_gates.standard_gates import (
    DEFAULT_STANDARD_GATE_ARG_NAME,
)


class UGate(function_params.FunctionParams):
    """
    Matrix representation:

    U(gam, phi,theta, lam) =
    e^(*gam) *
    cos(theta/2) & -e^(i*lam)*sin(theta/2) \\
    e^(i*phi)*sin(theta/2) & e^(i*(phi+lam))*cos(theta/2) \\
    """

    theta: ParameterFloatType = pydantic.Field(
        description="Angle to rotate by the Y-axis."
    )

    phi: ParameterFloatType = pydantic.Field(
        description="First angle to rotate by the Z-axis."
    )

    lam: ParameterFloatType = pydantic.Field(
        description="Second angle to rotate by the Z-axis."
    )

    gam: ParameterFloatType = pydantic.Field(
        description="Angle to apply phase gate by."
    )

    _inputs = pydantic.PrivateAttr(
        default={
            DEFAULT_STANDARD_GATE_ARG_NAME: RegisterUserInput(
                name=DEFAULT_STANDARD_GATE_ARG_NAME, size=1
            )
        }
    )
    _outputs = pydantic.PrivateAttr(
        default={
            DEFAULT_STANDARD_GATE_ARG_NAME: RegisterUserInput(
                name=DEFAULT_STANDARD_GATE_ARG_NAME, size=1
            )
        }
    )

    @pydantic.validator("*", pre=True)
    def validate_parameters(cls, value: ParameterFloatType) -> ParameterFloatType:
        if isinstance(value, str):
            sympy.parse_expr(value)
        if isinstance(value, sympy.Expr):
            return str(value)
        return value

    @property
    def is_parametric(self) -> bool:
        return not all(
            isinstance(angle, (float, int))
            for angle in (self.theta, self.phi, self.lam, self.gam)
        )
