"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * A PolicyDocument is a collection of statements
 */
class PolicyDocument {
    constructor(props = {}) {
        this.statements = new Array();
        this.creationStack = cdk.captureStackTrace();
        this.autoAssignSids = !!props.assignSids;
        this.addStatements(...props.statements || []);
    }
    /**
     * Creates a new PolicyDocument based on the object provided.
     * This will accept an object created from the `.toJSON()` call
     * @param obj the PolicyDocument in object form.
     */
    static fromJson(obj) {
        var _a;
        const newPolicyDocument = new PolicyDocument();
        const statement = (_a = obj.Statement, (_a !== null && _a !== void 0 ? _a : []));
        if (statement && !Array.isArray(statement)) {
            throw new Error('Statement must be an array');
        }
        newPolicyDocument.addStatements(...obj.Statement.map((s) => policy_statement_1.PolicyStatement.fromJson(s)));
        return newPolicyDocument;
    }
    resolve(context) {
        context.registerPostProcessor(new RemoveDuplicateStatements(this.autoAssignSids));
        return this.render();
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatements(...statement) {
        this.statements.push(...statement);
    }
    /**
     * Encode the policy document as a string
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyDocument'
        });
    }
    /**
     * JSON-ify the document
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.render();
    }
    render() {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            Statement: this.statements.map(s => s.toStatementJson()),
            Version: '2012-10-17'
        };
        return doc;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Removes duplicate statements and assign Sids if necessary
 */
class RemoveDuplicateStatements {
    constructor(autoAssignSids) {
        this.autoAssignSids = autoAssignSids;
    }
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this.autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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