"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const principals_1 = require("./principals");
const util_1 = require("./util");
const ensureArrayOrUndefined = (field) => {
    if (field === undefined) {
        return undefined;
    }
    if (typeof (field) !== "string" && !Array.isArray(field)) {
        throw new Error("Fields must be either a string or an array of strings");
    }
    if (Array.isArray(field) && !!field.find((f) => typeof (f) !== "string")) {
        throw new Error("Fields must be either a string or an array of strings");
    }
    return Array.isArray(field) ? field : [field];
};
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.sid = props.sid;
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    /**
     * Creates a new PolicyStatement based on the object provided.
     * This will accept an object created from the `.toJSON()` call
     * @param obj the PolicyStatement in object form.
     */
    static fromJson(obj) {
        return new PolicyStatement({
            sid: obj.Sid,
            actions: ensureArrayOrUndefined(obj.Action),
            resources: ensureArrayOrUndefined(obj.Resource),
            conditions: obj.Condition,
            effect: obj.Effect,
            notActions: ensureArrayOrUndefined(obj.NotAction),
            notResources: ensureArrayOrUndefined(obj.NotResource),
            principals: obj.Principal ? [new JsonPrincipal(obj.Principal)] : undefined,
            notPrincipals: obj.NotPrincipal ? [new JsonPrincipal(obj.NotPrincipal)] : undefined
        });
    }
    //
    // Actions
    //
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error(`Cannot add 'Actions' to policy statement if 'NotActions' have been added`);
        }
        this.action.push(...actions);
    }
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error(`Cannot add 'NotActions' to policy statement if 'Actions' have been added`);
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    addPrincipals(...principals) {
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error(`Cannot add 'Principals' to policy statement if 'NotPrincipals' have been added`);
        }
        for (const principal of principals) {
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addNotPrincipals(...notPrincipals) {
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error(`Cannot add 'NotPrincipals' to policy statement if 'Principals' have been added`);
        }
        for (const notPrincipal of notPrincipals) {
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.Anyone());
    }
    //
    // Resources
    //
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error(`Cannot add 'Resources' to policy statement if 'NotResources' have been added`);
        }
        this.resource.push(...arns);
    }
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error(`Cannot add 'NotResources' to policy statement if 'Resources' have been added`);
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        const existingValue = this.condition[key];
        this.condition[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action),
            NotAction: _norm(this.notAction),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            NotPrincipal: _normPrincipal(this.notPrincipal),
            Resource: _norm(this.resource),
            NotResource: _norm(this.notResource),
            Sid: _norm(this.sid),
        });
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement'
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
}
exports.PolicyStatement = PolicyStatement;
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
class JsonPrincipal extends principals_1.PrincipalBase {
    constructor(json = {}) {
        super();
        // special case: if principal is a string, turn it into an "AWS" principal
        if (typeof (json) === 'string') {
            json = { AWS: json };
        }
        this.policyFragment = {
            principalJson: json,
            conditions: []
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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