"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const iam = require("../lib");
describe('automatic cross-stack references', () => {
    test('automatic exports are created when attributes are referneced across stacks', () => {
        // GIVEN
        const app = new cdk.App();
        const stackWithUser = new cdk.Stack(app, 'stack1');
        const stackWithGroup = new cdk.Stack(app, 'stack2');
        const user = new iam.User(stackWithUser, 'User');
        const group = new iam.Group(stackWithGroup, 'Group');
        // WHEN
        group.addUser(user);
        //
        // `group.addUser` adds the group to the user resource definition, so we expect
        // that an automatic export will be created for the group and the user's stack
        // to use ImportValue to import it.
        // note that order of "expect"s matters. we first need to synthesize the user's
        // stack so that the cross stack reference will be reported and only then the
        // group's stack. in the real world, App will take care of this.
        //
        // THEN
        expect(stackWithUser).toMatchTemplate({
            Resources: {
                User00B015A1: {
                    Type: "AWS::IAM::User",
                    Properties: {
                        Groups: [{ "Fn::ImportValue": "stack2:ExportsOutputRefGroupC77FDACD8CF7DD5B" }]
                    }
                }
            }
        });
        expect(stackWithGroup).toMatchTemplate({
            Outputs: {
                ExportsOutputRefGroupC77FDACD8CF7DD5B: {
                    Value: { Ref: "GroupC77FDACD" },
                    Export: { Name: "stack2:ExportsOutputRefGroupC77FDACD8CF7DD5B" }
                }
            },
            Resources: {
                GroupC77FDACD: {
                    Type: "AWS::IAM::Group"
                }
            }
        });
    });
    test('cannot reference tokens across apps', () => {
        // GIVEN
        const stack1 = new cdk.Stack();
        const stack2 = new cdk.Stack();
        const user = new iam.User(stack1, 'User');
        const group = new iam.Group(stack2, 'Group');
        group.addUser(user);
        // THEN
        expect(() => assert_1.SynthUtils.synthesize(stack1)).toThrow(/Cannot reference across apps/);
    });
});
//# sourceMappingURL=data:application/json;base64,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