"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.shouldFollow = exports.shouldExclude = void 0;
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
const options_1 = require("./options");
/**
 * Determines whether a given file should be excluded or not based on given
 * exclusion glob patterns.
 *
 * @param exclude  exclusion patterns
 * @param filePath file path to be assessed against the pattern
 *
 * @returns `true` if the file should be excluded
 */
function shouldExclude(exclude, filePath) {
    let excludeOutput = false;
    for (const pattern of exclude) {
        const negate = pattern.startsWith('!');
        const match = minimatch(filePath, pattern, { matchBase: true, flipNegate: true });
        if (!negate && match) {
            excludeOutput = true;
        }
        if (negate && match) {
            excludeOutput = false;
        }
    }
    return excludeOutput;
}
exports.shouldExclude = shouldExclude;
/**
 * Determines whether a symlink should be followed or not, based on a FollowMode.
 *
 * @param mode       the FollowMode.
 * @param sourceRoot the root of the source tree.
 * @param realPath   the real path of the target of the symlink.
 *
 * @returns true if the link should be followed.
 */
function shouldFollow(mode, sourceRoot, realPath) {
    switch (mode) {
        case options_1.SymlinkFollowMode.ALWAYS:
            return fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.EXTERNAL:
            return !_isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.BLOCK_EXTERNAL:
            return _isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.NEVER:
            return false;
        default:
            throw new Error(`Unsupported FollowMode: ${mode}`);
    }
    function _isInternal() {
        return path.resolve(realPath).startsWith(path.resolve(sourceRoot));
    }
}
exports.shouldFollow = shouldFollow;
//# sourceMappingURL=data:application/json;base64,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