"use strict";
const fs = require("fs");
const path = require("path");
const lib_1 = require("../../lib");
const util_1 = require("../util");
const TEST_HANDLER = `${__dirname}/mock-provider`;
module.exports = {
    'minimal configuration'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        lib_1.CustomResourceProvider.getOrCreate(stack, 'Custom:MyResourceType', {
            codeDirectory: TEST_HANDLER,
            runtime: lib_1.CustomResourceProviderRuntime.NODEJS_12,
        });
        // THEN
        test.ok(fs.existsSync(path.join(TEST_HANDLER, '__entrypoint__.js')), 'expecting entrypoint to be copied to the handler directory');
        const cfn = util_1.toCloudFormation(stack);
        test.deepEqual(cfn, {
            Resources: {
                CustomMyResourceTypeCustomResourceProviderRoleBD5E655F: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Version: '2012-10-17',
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: {
                                        Service: 'lambda.amazonaws.com',
                                    },
                                },
                            ],
                        },
                        ManagedPolicyArns: [
                            {
                                'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                            },
                        ],
                    },
                },
                CustomMyResourceTypeCustomResourceProviderHandler29FBDD2A: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: {
                            S3Bucket: {
                                Ref: 'AssetParameters925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226S3Bucket8B4D0E9E',
                            },
                            S3Key: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::Select': [
                                                0,
                                                {
                                                    'Fn::Split': [
                                                        '||',
                                                        {
                                                            Ref: 'AssetParameters925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226S3VersionKeyDECB34FE',
                                                        },
                                                    ],
                                                },
                                            ],
                                        },
                                        {
                                            'Fn::Select': [
                                                1,
                                                {
                                                    'Fn::Split': [
                                                        '||',
                                                        {
                                                            Ref: 'AssetParameters925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226S3VersionKeyDECB34FE',
                                                        },
                                                    ],
                                                },
                                            ],
                                        },
                                    ],
                                ],
                            },
                        },
                        Timeout: 900,
                        MemorySize: 128,
                        Handler: '__entrypoint__.handler',
                        Role: {
                            'Fn::GetAtt': [
                                'CustomMyResourceTypeCustomResourceProviderRoleBD5E655F',
                                'Arn',
                            ],
                        },
                        Runtime: 'nodejs12.x',
                    },
                    DependsOn: [
                        'CustomMyResourceTypeCustomResourceProviderRoleBD5E655F',
                    ],
                },
            },
            Parameters: {
                AssetParameters925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226S3Bucket8B4D0E9E: {
                    Type: 'String',
                    Description: 'S3 bucket for asset "925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226"',
                },
                AssetParameters925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226S3VersionKeyDECB34FE: {
                    Type: 'String',
                    Description: 'S3 key for asset version "925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226"',
                },
                AssetParameters925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226ArtifactHashEEC400F2: {
                    Type: 'String',
                    Description: 'Artifact hash for asset "925e7fbbec7bdbf0136ef5a07b8a0fbe0b1f1bb4ea50ae2154163df78aa9f226"',
                },
            },
        });
        test.done();
    },
    'policyStatements can be used to add statements to the inline policy'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        lib_1.CustomResourceProvider.getOrCreate(stack, 'Custom:MyResourceType', {
            codeDirectory: TEST_HANDLER,
            runtime: lib_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                { statement1: 123 },
                { statement2: { foo: 111 } },
            ],
        });
        // THEN
        const template = util_1.toCloudFormation(stack);
        const role = template.Resources.CustomMyResourceTypeCustomResourceProviderRoleBD5E655F;
        test.deepEqual(role.Properties.Policies, [{
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ statement1: 123 }, { statement2: { foo: 111 } }],
                },
            }]);
        test.done();
    },
    'memorySize and timeout'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        lib_1.CustomResourceProvider.getOrCreate(stack, 'Custom:MyResourceType', {
            codeDirectory: TEST_HANDLER,
            runtime: lib_1.CustomResourceProviderRuntime.NODEJS_12,
            memorySize: lib_1.Size.gibibytes(2),
            timeout: lib_1.Duration.minutes(5),
        });
        // THEN
        const template = util_1.toCloudFormation(stack);
        const lambda = template.Resources.CustomMyResourceTypeCustomResourceProviderHandler29FBDD2A;
        test.deepEqual(lambda.Properties.MemorySize, 2048);
        test.deepEqual(lambda.Properties.Timeout, 300);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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