"use strict";
const child_process = require("child_process");
const sinon = require("sinon");
const lib_1 = require("../lib");
module.exports = {
    'tearDown'(callback) {
        sinon.restore();
        callback();
    },
    'bundling with image from registry'(test) {
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        image._run({
            command: ['cool', 'command'],
            environment: {
                VAR1: 'value1',
                VAR2: 'value2',
            },
            volumes: [{ hostPath: '/host-path', containerPath: '/container-path' }],
            workingDirectory: '/working-directory',
            user: 'user:group',
        });
        test.ok(spawnSyncStub.calledWith('docker', [
            'run', '--rm',
            '-u', 'user:group',
            '-v', '/host-path:/container-path',
            '--env', 'VAR1=value1',
            '--env', 'VAR2=value2',
            '-w', '/working-directory',
            'alpine',
            'cool', 'command',
        ]));
        test.done();
    },
    'bundling with image from asset'(test) {
        const imageId = 'abcdef123456';
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from(`Successfully built ${imageId}`),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromAsset('docker-path', {
            buildArgs: {
                TEST_ARG: 'cdk-test',
            },
        });
        image._run();
        test.ok(spawnSyncStub.firstCall.calledWith('docker', [
            'build',
            '--build-arg', 'TEST_ARG=cdk-test',
            'docker-path',
        ]));
        test.ok(spawnSyncStub.secondCall.calledWith('docker', [
            'run', '--rm',
            imageId,
        ]));
        test.done();
    },
    'throws if image id cannot be extracted from build output'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        test.throws(() => lib_1.BundlingDockerImage.fromAsset('docker-path'), /Failed to extract image ID from Docker build output/);
        test.done();
    },
    'throws in case of spawnSync error'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
            error: new Error('UnknownError'),
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /UnknownError/);
        test.done();
    },
    'throws if status is not 0'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: -1,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /\[Status -1\]/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5idW5kbGluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuYnVuZGxpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLCtDQUErQztBQUUvQywrQkFBK0I7QUFDL0IsZ0NBQTZDO0FBRTdDLGlCQUFTO0lBQ1AsVUFBVSxDQUFDLFFBQWE7UUFDdEIsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ2hCLFFBQVEsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVELG1DQUFtQyxDQUFDLElBQVU7UUFDNUMsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ25FLE1BQU0sRUFBRSxDQUFDO1lBQ1QsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixHQUFHLEVBQUUsR0FBRztZQUNSLE1BQU0sRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUM7WUFDNUIsTUFBTSxFQUFFLElBQUk7U0FDYixDQUFDLENBQUM7UUFFSCxNQUFNLEtBQUssR0FBRyx5QkFBbUIsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDekQsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNULE9BQU8sRUFBRSxDQUFDLE1BQU0sRUFBRSxTQUFTLENBQUM7WUFDNUIsV0FBVyxFQUFFO2dCQUNYLElBQUksRUFBRSxRQUFRO2dCQUNkLElBQUksRUFBRSxRQUFRO2FBQ2Y7WUFDRCxPQUFPLEVBQUUsQ0FBQyxFQUFFLFFBQVEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLGlCQUFpQixFQUFFLENBQUM7WUFDdkUsZ0JBQWdCLEVBQUUsb0JBQW9CO1lBQ3RDLElBQUksRUFBRSxZQUFZO1NBQ25CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7WUFDekMsS0FBSyxFQUFFLE1BQU07WUFDYixJQUFJLEVBQUUsWUFBWTtZQUNsQixJQUFJLEVBQUUsNEJBQTRCO1lBQ2xDLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLElBQUksRUFBRSxvQkFBb0I7WUFDMUIsUUFBUTtZQUNSLE1BQU0sRUFBRSxTQUFTO1NBQ2xCLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdDQUFnQyxDQUFDLElBQVU7UUFDekMsTUFBTSxPQUFPLEdBQUcsY0FBYyxDQUFDO1FBQy9CLE1BQU0sYUFBYSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNuRSxNQUFNLEVBQUUsQ0FBQztZQUNULE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsT0FBTyxFQUFFLENBQUM7WUFDcEQsR0FBRyxFQUFFLEdBQUc7WUFDUixNQUFNLEVBQUUsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDO1lBQzVCLE1BQU0sRUFBRSxJQUFJO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsTUFBTSxLQUFLLEdBQUcseUJBQW1CLENBQUMsU0FBUyxDQUFDLGFBQWEsRUFBRTtZQUN6RCxTQUFTLEVBQUU7Z0JBQ1QsUUFBUSxFQUFFLFVBQVU7YUFDckI7U0FDRixDQUFDLENBQUM7UUFDSCxLQUFLLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFYixJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLFFBQVEsRUFBRTtZQUNuRCxPQUFPO1lBQ1AsYUFBYSxFQUFFLG1CQUFtQjtZQUNsQyxhQUFhO1NBQ2QsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLFFBQVEsRUFBRTtZQUNwRCxLQUFLLEVBQUUsTUFBTTtZQUNiLE9BQU87U0FDUixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwREFBMEQsQ0FBQyxJQUFVO1FBQ25FLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUM3QyxNQUFNLEVBQUUsQ0FBQztZQUNULE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDN0IsR0FBRyxFQUFFLEdBQUc7WUFDUixNQUFNLEVBQUUsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDO1lBQzVCLE1BQU0sRUFBRSxJQUFJO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyx5QkFBbUIsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLEVBQUUscURBQXFELENBQUMsQ0FBQztRQUN2SCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUNBQW1DLENBQUMsSUFBVTtRQUM1QyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDN0MsTUFBTSxFQUFFLENBQUM7WUFDVCxNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDN0IsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLEdBQUcsRUFBRSxHQUFHO1lBQ1IsTUFBTSxFQUFFLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQztZQUM1QixNQUFNLEVBQUUsSUFBSTtZQUNaLEtBQUssRUFBRSxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUM7U0FDakMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxLQUFLLEdBQUcseUJBQW1CLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyQkFBMkIsQ0FBQyxJQUFVO1FBQ3BDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUM3QyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1lBQ1YsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixHQUFHLEVBQUUsR0FBRztZQUNSLE1BQU0sRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUM7WUFDNUIsTUFBTSxFQUFFLElBQUk7U0FDYixDQUFDLENBQUM7UUFFSCxNQUFNLEtBQUssR0FBRyx5QkFBbUIsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDekQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjaGlsZF9wcm9jZXNzIGZyb20gJ2NoaWxkX3Byb2Nlc3MnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCAqIGFzIHNpbm9uIGZyb20gJ3Npbm9uJztcbmltcG9ydCB7IEJ1bmRsaW5nRG9ja2VySW1hZ2UgfSBmcm9tICcuLi9saWInO1xuXG5leHBvcnQgPSB7XG4gICd0ZWFyRG93bicoY2FsbGJhY2s6IGFueSkge1xuICAgIHNpbm9uLnJlc3RvcmUoKTtcbiAgICBjYWxsYmFjaygpO1xuICB9LFxuXG4gICdidW5kbGluZyB3aXRoIGltYWdlIGZyb20gcmVnaXN0cnknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzcGF3blN5bmNTdHViID0gc2lub24uc3R1YihjaGlsZF9wcm9jZXNzLCAnc3Bhd25TeW5jJykucmV0dXJucyh7XG4gICAgICBzdGF0dXM6IDAsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oJ3N0ZG91dCcpLFxuICAgICAgcGlkOiAxMjMsXG4gICAgICBvdXRwdXQ6IFsnc3Rkb3V0JywgJ3N0ZGVyciddLFxuICAgICAgc2lnbmFsOiBudWxsLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaW1hZ2UgPSBCdW5kbGluZ0RvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnYWxwaW5lJyk7XG4gICAgaW1hZ2UuX3J1bih7XG4gICAgICBjb21tYW5kOiBbJ2Nvb2wnLCAnY29tbWFuZCddLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgVkFSMTogJ3ZhbHVlMScsXG4gICAgICAgIFZBUjI6ICd2YWx1ZTInLFxuICAgICAgfSxcbiAgICAgIHZvbHVtZXM6IFt7IGhvc3RQYXRoOiAnL2hvc3QtcGF0aCcsIGNvbnRhaW5lclBhdGg6ICcvY29udGFpbmVyLXBhdGgnIH1dLFxuICAgICAgd29ya2luZ0RpcmVjdG9yeTogJy93b3JraW5nLWRpcmVjdG9yeScsXG4gICAgICB1c2VyOiAndXNlcjpncm91cCcsXG4gICAgfSk7XG5cbiAgICB0ZXN0Lm9rKHNwYXduU3luY1N0dWIuY2FsbGVkV2l0aCgnZG9ja2VyJywgW1xuICAgICAgJ3J1bicsICctLXJtJyxcbiAgICAgICctdScsICd1c2VyOmdyb3VwJyxcbiAgICAgICctdicsICcvaG9zdC1wYXRoOi9jb250YWluZXItcGF0aCcsXG4gICAgICAnLS1lbnYnLCAnVkFSMT12YWx1ZTEnLFxuICAgICAgJy0tZW52JywgJ1ZBUjI9dmFsdWUyJyxcbiAgICAgICctdycsICcvd29ya2luZy1kaXJlY3RvcnknLFxuICAgICAgJ2FscGluZScsXG4gICAgICAnY29vbCcsICdjb21tYW5kJyxcbiAgICBdKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2J1bmRsaW5nIHdpdGggaW1hZ2UgZnJvbSBhc3NldCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IGltYWdlSWQgPSAnYWJjZGVmMTIzNDU2JztcbiAgICBjb25zdCBzcGF3blN5bmNTdHViID0gc2lub24uc3R1YihjaGlsZF9wcm9jZXNzLCAnc3Bhd25TeW5jJykucmV0dXJucyh7XG4gICAgICBzdGF0dXM6IDAsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oYFN1Y2Nlc3NmdWxseSBidWlsdCAke2ltYWdlSWR9YCksXG4gICAgICBwaWQ6IDEyMyxcbiAgICAgIG91dHB1dDogWydzdGRvdXQnLCAnc3RkZXJyJ10sXG4gICAgICBzaWduYWw6IG51bGwsXG4gICAgfSk7XG5cbiAgICBjb25zdCBpbWFnZSA9IEJ1bmRsaW5nRG9ja2VySW1hZ2UuZnJvbUFzc2V0KCdkb2NrZXItcGF0aCcsIHtcbiAgICAgIGJ1aWxkQXJnczoge1xuICAgICAgICBURVNUX0FSRzogJ2Nkay10ZXN0JyxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgaW1hZ2UuX3J1bigpO1xuXG4gICAgdGVzdC5vayhzcGF3blN5bmNTdHViLmZpcnN0Q2FsbC5jYWxsZWRXaXRoKCdkb2NrZXInLCBbXG4gICAgICAnYnVpbGQnLFxuICAgICAgJy0tYnVpbGQtYXJnJywgJ1RFU1RfQVJHPWNkay10ZXN0JyxcbiAgICAgICdkb2NrZXItcGF0aCcsXG4gICAgXSkpO1xuXG4gICAgdGVzdC5vayhzcGF3blN5bmNTdHViLnNlY29uZENhbGwuY2FsbGVkV2l0aCgnZG9ja2VyJywgW1xuICAgICAgJ3J1bicsICctLXJtJyxcbiAgICAgIGltYWdlSWQsXG4gICAgXSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0aHJvd3MgaWYgaW1hZ2UgaWQgY2Fubm90IGJlIGV4dHJhY3RlZCBmcm9tIGJ1aWxkIG91dHB1dCcodGVzdDogVGVzdCkge1xuICAgIHNpbm9uLnN0dWIoY2hpbGRfcHJvY2VzcywgJ3NwYXduU3luYycpLnJldHVybnMoe1xuICAgICAgc3RhdHVzOiAwLFxuICAgICAgc3RkZXJyOiBCdWZmZXIuZnJvbSgnc3RkZXJyJyksXG4gICAgICBzdGRvdXQ6IEJ1ZmZlci5mcm9tKCdzdGRvdXQnKSxcbiAgICAgIHBpZDogMTIzLFxuICAgICAgb3V0cHV0OiBbJ3N0ZG91dCcsICdzdGRlcnInXSxcbiAgICAgIHNpZ25hbDogbnVsbCxcbiAgICB9KTtcblxuICAgIHRlc3QudGhyb3dzKCgpID0+IEJ1bmRsaW5nRG9ja2VySW1hZ2UuZnJvbUFzc2V0KCdkb2NrZXItcGF0aCcpLCAvRmFpbGVkIHRvIGV4dHJhY3QgaW1hZ2UgSUQgZnJvbSBEb2NrZXIgYnVpbGQgb3V0cHV0Lyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rocm93cyBpbiBjYXNlIG9mIHNwYXduU3luYyBlcnJvcicodGVzdDogVGVzdCkge1xuICAgIHNpbm9uLnN0dWIoY2hpbGRfcHJvY2VzcywgJ3NwYXduU3luYycpLnJldHVybnMoe1xuICAgICAgc3RhdHVzOiAwLFxuICAgICAgc3RkZXJyOiBCdWZmZXIuZnJvbSgnc3RkZXJyJyksXG4gICAgICBzdGRvdXQ6IEJ1ZmZlci5mcm9tKCdzdGRvdXQnKSxcbiAgICAgIHBpZDogMTIzLFxuICAgICAgb3V0cHV0OiBbJ3N0ZG91dCcsICdzdGRlcnInXSxcbiAgICAgIHNpZ25hbDogbnVsbCxcbiAgICAgIGVycm9yOiBuZXcgRXJyb3IoJ1Vua25vd25FcnJvcicpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaW1hZ2UgPSBCdW5kbGluZ0RvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnYWxwaW5lJyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gaW1hZ2UuX3J1bigpLCAvVW5rbm93bkVycm9yLyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rocm93cyBpZiBzdGF0dXMgaXMgbm90IDAnKHRlc3Q6IFRlc3QpIHtcbiAgICBzaW5vbi5zdHViKGNoaWxkX3Byb2Nlc3MsICdzcGF3blN5bmMnKS5yZXR1cm5zKHtcbiAgICAgIHN0YXR1czogLTEsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oJ3N0ZG91dCcpLFxuICAgICAgcGlkOiAxMjMsXG4gICAgICBvdXRwdXQ6IFsnc3Rkb3V0JywgJ3N0ZGVyciddLFxuICAgICAgc2lnbmFsOiBudWxsLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaW1hZ2UgPSBCdW5kbGluZ0RvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnYWxwaW5lJyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gaW1hZ2UuX3J1bigpLCAvXFxbU3RhdHVzIC0xXFxdLyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuIl19