"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("@aws-cdk/assets");
const ecr = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            this.node.addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.stagedPath,
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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