"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const lib_1 = require("../lib");
module.exports = {
    'test instantiating Asset Image'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'test-stack');
        // WHEN
        new lib_1.DockerImageAsset(stack, 'Image', {
            directory: path.join(__dirname, 'demo-image'),
        });
        // THEN
        const asm = app.synth();
        const artifact = asm.getStackArtifact(stack.artifactId);
        test.deepEqual(artifact.template, {}, 'template is empty');
        test.deepEqual(artifact.assets, [
            {
                repositoryName: 'aws-cdk/assets',
                imageTag: 'baa2d6eb2a17c75424df631c8c70ff39f2d5f3bee8b9e1a109ee24ca17300540',
                id: 'baa2d6eb2a17c75424df631c8c70ff39f2d5f3bee8b9e1a109ee24ca17300540',
                packaging: 'container-image',
                path: 'asset.baa2d6eb2a17c75424df631c8c70ff39f2d5f3bee8b9e1a109ee24ca17300540',
                sourceHash: 'baa2d6eb2a17c75424df631c8c70ff39f2d5f3bee8b9e1a109ee24ca17300540',
            },
        ]);
        test.done();
    },
    'with build args'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.DockerImageAsset(stack, 'Image', {
            directory: path.join(__dirname, 'demo-image'),
            buildArgs: {
                a: 'b',
            },
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.deepEqual(assetMetadata && assetMetadata.data.buildArgs, { a: 'b' });
        test.done();
    },
    'with target'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.DockerImageAsset(stack, 'Image', {
            directory: path.join(__dirname, 'demo-image'),
            buildArgs: {
                a: 'b',
            },
            target: 'a-target',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.deepEqual(assetMetadata && assetMetadata.data.target, 'a-target');
        test.done();
    },
    'with file'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const directoryPath = path.join(__dirname, 'demo-image-custom-docker-file');
        // WHEN
        new lib_1.DockerImageAsset(stack, 'Image', {
            directory: directoryPath,
            file: 'Dockerfile.Custom',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.deepEqual(assetMetadata && assetMetadata.data.file, 'Dockerfile.Custom');
        test.done();
    },
    'asset.repository.grantPull can be used to grant a principal permissions to use the image'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const user = new iam.User(stack, 'MyUser');
        const asset = new lib_1.DockerImageAsset(stack, 'Image', {
            directory: path.join(__dirname, 'demo-image'),
        });
        // WHEN
        asset.repository.grantPull(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                'Statement': [
                    {
                        'Action': [
                            'ecr:BatchCheckLayerAvailability',
                            'ecr:GetDownloadUrlForLayer',
                            'ecr:BatchGetImage',
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        'Ref': 'AWS::Partition',
                                    },
                                    ':ecr:',
                                    {
                                        'Ref': 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        'Ref': 'AWS::AccountId',
                                    },
                                    ':repository/aws-cdk/assets',
                                ],
                            ],
                        },
                    },
                    {
                        'Action': 'ecr:GetAuthorizationToken',
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'MyUserDefaultPolicy7B897426',
            'Users': [
                {
                    'Ref': 'MyUserDC45028B',
                },
            ],
        }));
        test.done();
    },
    'fails if the directory does not exist'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => {
            new lib_1.DockerImageAsset(stack, 'MyAsset', {
                directory: `/does/not/exist/${Math.floor(Math.random() * 9999)}`,
            });
        }, /Cannot find image directory at/);
        test.done();
    },
    'fails if the directory does not contain a Dockerfile'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => {
            new lib_1.DockerImageAsset(stack, 'Asset', {
                directory: __dirname,
            });
        }, /Cannot find file at/);
        test.done();
    },
    'fails if the file does not exist'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => {
            new lib_1.DockerImageAsset(stack, 'Asset', {
                directory: __dirname,
                file: 'doesnt-exist',
            });
        }, /Cannot find file at/);
        test.done();
    },
    'docker directory is staged if asset staging is enabled'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'stack');
        const image = new lib_1.DockerImageAsset(stack, 'MyAsset', {
            directory: path.join(__dirname, 'demo-image'),
        });
        const session = app.synth();
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'Dockerfile')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'index.py')));
        test.done();
    },
    'docker directory is staged without files specified in .dockerignore'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'stack');
        const image = new lib_1.DockerImageAsset(stack, 'MyAsset', {
            directory: path.join(__dirname, 'dockerignore-image'),
        });
        const session = app.synth();
        // .dockerignore itself should be included in output to be processed during docker build
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, '.dockerignore')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'Dockerfile')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'index.py')));
        test.ok(!fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'foobar.txt')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'subdirectory')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'subdirectory', 'baz.txt')));
        test.done();
    },
    'docker directory is staged without files specified in exclude option'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'stack');
        const image = new lib_1.DockerImageAsset(stack, 'MyAsset', {
            directory: path.join(__dirname, 'dockerignore-image'),
            exclude: ['subdirectory'],
        });
        const session = app.synth();
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, '.dockerignore')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'Dockerfile')));
        test.ok(fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'index.py')));
        test.ok(!fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'foobar.txt')));
        test.ok(!fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'subdirectory')));
        test.ok(!fs.existsSync(path.join(session.directory, `asset.${image.sourceHash}`, 'subdirectory', 'baz.txt')));
        test.done();
    },
    'fails if using tokens in build args keys or values'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const token = core_1.Lazy.stringValue({ produce: () => 'foo' });
        const expected = /Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment/;
        // THEN
        test.throws(() => new lib_1.DockerImageAsset(stack, 'MyAsset1', {
            directory: path.join(__dirname, 'demo-image'),
            buildArgs: { [token]: 'value' },
        }), expected);
        test.throws(() => new lib_1.DockerImageAsset(stack, 'MyAsset2', {
            directory: path.join(__dirname, 'demo-image'),
            buildArgs: { key: token },
        }), expected);
        test.done();
    },
    'fails if using token as repositoryName'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const token = core_1.Lazy.stringValue({ produce: () => 'foo' });
        // THEN
        test.throws(() => new lib_1.DockerImageAsset(stack, 'MyAsset1', {
            directory: path.join(__dirname, 'demo-image'),
            repositoryName: token,
        }), /Cannot use Token as value of 'repositoryName'/);
        test.done();
    },
    'docker build options are included in the asset id'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const directory = path.join(__dirname, 'demo-image-custom-docker-file');
        const asset1 = new lib_1.DockerImageAsset(stack, 'Asset1', { directory });
        const asset2 = new lib_1.DockerImageAsset(stack, 'Asset2', { directory, file: 'Dockerfile.Custom' });
        const asset3 = new lib_1.DockerImageAsset(stack, 'Asset3', { directory, target: 'NonDefaultTarget' });
        const asset4 = new lib_1.DockerImageAsset(stack, 'Asset4', { directory, buildArgs: { opt1: '123', opt2: 'boom' } });
        const asset5 = new lib_1.DockerImageAsset(stack, 'Asset5', { directory, file: 'Dockerfile.Custom', target: 'NonDefaultTarget' });
        const asset6 = new lib_1.DockerImageAsset(stack, 'Asset6', { directory, extraHash: 'random-extra' });
        const asset7 = new lib_1.DockerImageAsset(stack, 'Asset7', { directory, repositoryName: 'foo' });
        test.deepEqual(asset1.sourceHash, 'c555ab9f74e32ce24cd04ddeaa4d7b1b11c5740b9873a3f250e03bf73b28ce39');
        test.deepEqual(asset2.sourceHash, '273bd9a95dbe346ad5b116736d44a350e90f57e2b9ba7fd3d334b61d0420f9fd');
        test.deepEqual(asset3.sourceHash, '81a4b3fd058876c7705597500e7259ff436e521580f0bcb503a303dcac7e2a41');
        test.deepEqual(asset4.sourceHash, '10259531feb68a3967d5d25b70ec9a37a6a8e1f5b04083fada3d0a084291a698');
        test.deepEqual(asset5.sourceHash, '30e083bf51483a031759bc7fb35f69345de69fdbc511eec88bd3d1724b5ac0a9');
        test.deepEqual(asset6.sourceHash, '594ae5a5d23367d18468fefca5a4e56ca83b077d1274a1f812f55c8c9ead9eaa');
        test.deepEqual(asset7.sourceHash, 'bc007f81fe1dd0f0bbb24af898eba3f4f15edbff19b7abb3fac928439486d667');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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