"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const cdk = require("aws-cdk-lib");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test new load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
        LoadBalancerAttributes: [
            {
                Key: "deletion_protection.enabled",
                Value: "false"
            },
            {
                Key: "access_logs.s3.enabled",
                Value: "true"
            },
            {
                Key: "access_logs.s3.bucket",
                Value: {
                    Ref: "testoneE6ACFBB6"
                }
            },
            {
                Key: "access_logs.s3.prefix",
                Value: ""
            }
        ],
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test new load balancer and new lambda function for HTTP api', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::Lambda::Function', 1);
});
test('Test existing load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::Lambda::Function', 1);
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::LoadBalancer", {
        Scheme: "internet-facing",
    });
});
test('Test new load balancer and existing lambda function', () => {
    const testFunctionName = 'fixed-name';
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const lambdaFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        functionName: testFunctionName,
        vpc: testExistingVpc
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true,
        existingVpc: testExistingVpc
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testFunctionName
    });
});
test("Test existing load balancer and existing lambda function", () => {
    const testFunctionName = "fixed-name";
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: "us-east-1" },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, "test-alb", {
        vpc: testExistingVpc,
    });
    const lambdaFunction = new lambda.Function(stack, "existing-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: "index.handler",
        functionName: testFunctionName,
        vpc: testExistingVpc,
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")],
        },
        publicApi: true,
        existingVpc: testExistingVpc,
    };
    new lib_1.AlbToLambda(stack, "test-one", props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::LoadBalancer", {
        Scheme: "internal",
    });
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::Listener", {
        Protocol: "HTTP",
    });
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::Listener", {
        Protocol: "HTTPS",
    });
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::ListenerCertificate", {});
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::TargetGroup", {
        TargetType: "lambda",
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test('Test new load balancer and new lambda function', () => {
    const testFunctionName = 'fixed-name';
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            functionName: testFunctionName,
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")],
        },
        publicApi: true,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testFunctionName
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
});
test('Test HTTPS adding 2 lambda targets, second with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
});
test('Test HTTP adding 2 lambda targets, second with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
});
test('Test new load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda',
        Name: 'different-name'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test logging turned off', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        logAlbAccessLogs: false,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::S3::Bucket', 0);
});
test('Check Properties', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const target = new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(target.loadBalancer);
    expect(target.vpc);
    expect(target.lambdaFunction);
    expect(target.listener);
});
test('Test custom ALB properties', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true,
        loadBalancerProps: {
            loadBalancerName: 'custom-name',
        }
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
        Name: 'custom-name',
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test custom logging bucket', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true,
        albLoggingBucketProps: {
            bucketName: 'custom-name',
        }
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::S3::Bucket', {
        BucketName: 'custom-name'
    });
});
test('Test providing certificateArns is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificateArns: ['	arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012']
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        albLoggingBucketProps: {
            bucketName: 'should-fail'
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates/);
});
test('Test logging off with logBucketProperties provided is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        logAlbAccessLogs: false,
        albLoggingBucketProps: {
            bucketName: 'should-fail'
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/Error - If logAlbAccessLogs is false, supplying albLoggingBucketProps is invalid./);
});
test('Test certificate with HTTP is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")],
            protocol: 'HTTP',
        },
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/HTTP listeners cannot use a certificate/);
});
test('Test new ALB with no listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP/);
});
test('Test existing ALB with no listener with no listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP/);
});
test('Test existing ALB with a listener with listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        listenerProps: {
            protocol: 'HTTPS'
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    };
    // Assertion
    expect(app).toThrowError(/This load balancer already has a listener, listenerProps may not be specified/);
});
test('Test second target with no rules is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    };
    // Assertion
    expect(app).toThrowError(/When adding a second target to an existing listener, there must be rules provided/);
});
test('Test no cert for an HTTPS listener is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {},
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/A listener using HTTPS protocol requires a certificate/);
});
test('Test existingLoadBalancerObj and loadBalancerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
        loadBalancerProps: {
            internetFacing: true
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide loadBalancerProps or existingLoadBalancerObj, but not both./);
});
test('Test existingLoadBalancerObj and no existingVpc is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        publicApi: true,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/An existing ALB is already in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use./);
});
//# sourceMappingURL=data:application/json;base64,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