"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SizeRoundingBehavior = exports.Size = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const token_1 = require("./token");
/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 */
class Size {
    constructor(amount, unit) {
        if (!token_1.Token.isUnresolved(amount) && amount < 0) {
            throw new Error(`Storage amounts cannot be negative. Received: ${amount}`);
        }
        this.amount = amount;
        this.unit = unit;
    }
    /**
     * Create a Storage representing an amount kibibytes.
     * 1 KiB = 1024 bytes
     *
     * @param amount the amount of kibibytes to be represented
     *
     * @returns a new `Size` instance
     */
    static kibibytes(amount) {
        return new Size(amount, StorageUnit.Kibibytes);
    }
    /**
     * Create a Storage representing an amount mebibytes.
     * 1 MiB = 1024 KiB
     *
     * @param amount the amount of mebibytes to be represented
     *
     * @returns a new `Size` instance
     */
    static mebibytes(amount) {
        return new Size(amount, StorageUnit.Mebibytes);
    }
    /**
     * Create a Storage representing an amount gibibytes.
     * 1 GiB = 1024 MiB
     *
     * @param amount the amount of gibibytes to be represented
     *
     * @returns a new `Size` instance
     */
    static gibibytes(amount) {
        return new Size(amount, StorageUnit.Gibibytes);
    }
    /**
     * Create a Storage representing an amount tebibytes.
     * 1 TiB = 1024 GiB
     *
     * @param amount the amount of tebibytes to be represented
     *
     * @returns a new `Size` instance
     */
    static tebibytes(amount) {
        return new Size(amount, StorageUnit.Tebibytes);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     * 1 PiB = 1024 TiB
     *
     * @deprecated use `pebibytes` instead
     */
    static pebibyte(amount) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/core.Size#pebibyte", "use `pebibytes` instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pebibyte);
            }
            throw error;
        }
        return Size.pebibytes(amount);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     * 1 PiB = 1024 TiB
     *
     * @param amount the amount of pebibytes to be represented
     *
     * @returns a new `Size` instance
     */
    static pebibytes(amount) {
        return new Size(amount, StorageUnit.Pebibytes);
    }
    /**
     * Return this storage as a total number of kibibytes.
     *
     * @param opts the conversion options
     *
     * @returns the quantity of bytes expressed in kibibytes
     */
    toKibibytes(opts = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_SizeConversionOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toKibibytes);
            }
            throw error;
        }
        return convert(this.amount, this.unit, StorageUnit.Kibibytes, opts);
    }
    /**
     * Return this storage as a total number of mebibytes.
     *
     * @param opts the conversion options
     *
     * @returns the quantity of bytes expressed in mebibytes
     */
    toMebibytes(opts = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_SizeConversionOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toMebibytes);
            }
            throw error;
        }
        return convert(this.amount, this.unit, StorageUnit.Mebibytes, opts);
    }
    /**
     * Return this storage as a total number of gibibytes.
     *
     * @param opts the conversion options
     *
     * @returns the quantity of bytes expressed in gibibytes
     */
    toGibibytes(opts = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_SizeConversionOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toGibibytes);
            }
            throw error;
        }
        return convert(this.amount, this.unit, StorageUnit.Gibibytes, opts);
    }
    /**
     * Return this storage as a total number of tebibytes.
     *
     * @param opts the conversion options
     *
     * @returns the quantity of bytes expressed in tebibytes
     */
    toTebibytes(opts = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_SizeConversionOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toTebibytes);
            }
            throw error;
        }
        return convert(this.amount, this.unit, StorageUnit.Tebibytes, opts);
    }
    /**
     * Return this storage as a total number of pebibytes.
     *
     * @param opts the conversion options
     *
     * @returns the quantity of bytes expressed in pebibytes
     */
    toPebibytes(opts = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_SizeConversionOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toPebibytes);
            }
            throw error;
        }
        return convert(this.amount, this.unit, StorageUnit.Pebibytes, opts);
    }
    /**
     * Checks if size is a token or a resolvable object
     */
    isUnresolved() {
        return token_1.Token.isUnresolved(this.amount);
    }
}
exports.Size = Size;
_a = JSII_RTTI_SYMBOL_1;
Size[_a] = { fqn: "@aws-cdk/core.Size", version: "1.204.0" };
/**
 * Rounding behaviour when converting between units of `Size`.
 */
var SizeRoundingBehavior;
(function (SizeRoundingBehavior) {
    /** Fail the conversion if the result is not an integer. */
    SizeRoundingBehavior[SizeRoundingBehavior["FAIL"] = 0] = "FAIL";
    /** If the result is not an integer, round it to the closest integer less than the result */
    SizeRoundingBehavior[SizeRoundingBehavior["FLOOR"] = 1] = "FLOOR";
    /** Don't round. Return even if the result is a fraction. */
    SizeRoundingBehavior[SizeRoundingBehavior["NONE"] = 2] = "NONE";
})(SizeRoundingBehavior = exports.SizeRoundingBehavior || (exports.SizeRoundingBehavior = {}));
class StorageUnit {
    constructor(label, inKibiBytes) {
        this.label = label;
        this.inKibiBytes = inKibiBytes;
    }
    toString() {
        return this.label;
    }
}
StorageUnit.Kibibytes = new StorageUnit('kibibytes', 1);
StorageUnit.Mebibytes = new StorageUnit('mebibytes', 1024);
StorageUnit.Gibibytes = new StorageUnit('gibibytes', 1024 * 1024);
StorageUnit.Tebibytes = new StorageUnit('tebibytes', 1024 * 1024 * 1024);
StorageUnit.Pebibytes = new StorageUnit('pebibytes', 1024 * 1024 * 1024 * 1024);
function convert(amount, fromUnit, toUnit, options = {}) {
    const rounding = options.rounding ?? SizeRoundingBehavior.FAIL;
    if (fromUnit.inKibiBytes === toUnit.inKibiBytes) {
        return amount;
    }
    if (token_1.Token.isUnresolved(amount)) {
        throw new Error(`Size must be specified as 'Size.${toUnit}()' here since its value comes from a token and cannot be converted (got Size.${fromUnit})`);
    }
    const multiplier = fromUnit.inKibiBytes / toUnit.inKibiBytes;
    const value = amount * multiplier;
    switch (rounding) {
        case SizeRoundingBehavior.NONE:
            return value;
        case SizeRoundingBehavior.FLOOR:
            return Math.floor(value);
        default:
        case SizeRoundingBehavior.FAIL:
            if (!Number.isInteger(value)) {
                throw new Error(`'${amount} ${fromUnit}' cannot be converted into a whole number of ${toUnit}.`);
            }
            return value;
    }
}
//# sourceMappingURL=data:application/json;base64,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