"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToKendra = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToKendra class.
 */
class LambdaToKendra extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToKendra class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToKendraProps} props - user provided props for the construct.
     * @since 1.120.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        if (props.kendraIndexProps && props.existingKendraIndexObj) {
            throw new Error('You may not provide both kendraIndexProps and existingKendraIndexObj');
        }
        if (props.kendraIndexProps && props.kendraDataSourcesProps) {
            throw new Error('You may not provide both kendraDataSourcesProps and existingKendraIndexObj');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.KENDRA);
        }
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        this.kendraIndex = defaults.buildKendraIndex(this, id, {
            kendraIndexProps: props.kendraIndexProps,
            existingIndexObj: props.existingKendraIndexObj
        });
        this.kendraDataSources = defaults.AddMultipleKendraDataSources(this, id, this.kendraIndex, props.kendraDataSourcesProps);
        // Update Lambda function IAM policy with correct privileges to Kendra index
        const normalizedPermissions = props.indexPermissions ? defaults.normalizeKendraPermissions(props.indexPermissions) : undefined;
        if (!normalizedPermissions || normalizedPermissions.includes("READ")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Query",
                    "kendra:Describe*",
                    "kendra:Get*",
                    "kendra:BatchGet*",
                    "kendra:List*",
                    "kendra:Retrieve"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (!normalizedPermissions || normalizedPermissions.includes("SUBMITFEEDBACK")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:SubmitFeedback"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (normalizedPermissions?.includes("WRITE")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Associate*",
                    "kendra:BatchPut*",
                    "kendra:Clear",
                    "kendra:Create*",
                    "kendra:Delete*",
                    "kendra:Disassociate*",
                    "kendra:Put*",
                    "kendra:Update*",
                    "kendra:Start*",
                    "kendra:Submit*",
                    "kendra:Stop*",
                    "kendra:TagResource",
                    "kendra:UntagResource"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        // Configure environment variables
        const indexIdEnvironmentVariableName = props.indexIdEnvironmentVariableName || 'KENDRA_INDEX_ID';
        this.lambdaFunction.addEnvironment(indexIdEnvironmentVariableName, this.kendraIndex.attrId);
    }
}
exports.LambdaToKendra = LambdaToKendra;
_a = JSII_RTTI_SYMBOL_1;
LambdaToKendra[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-kendra.LambdaToKendra", version: "2.65.0" };
//# sourceMappingURL=data:application/json;base64,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