#!/usr/bin/python3
# -*- coding: utf-8 -*-

"""
Tracker
=======

This module contains several tools to work with data that comes from the
2 Flies Tracker software.

"""

from os import path
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

from oneeurofilter import OneEuroFilter


class TrackerData(object):

    """
    Tracker Data
    ============

    Use this class to work with all data contained in the CSV file
    generated by the OpenCSP '2 Flies Tracking' software.

    You can access both flies data through either the `male` or the
    `female` attribute. These are pandas DataFrame objects.

    Some features calculated by the tracker (or by the user) may be
    independent of gender. For that, you should use the attribute
    `common`, which is also a DataFrame contaning only features common
    two both genders. An example of this would be the distance between
    flies.

    :param str filename: CSV file generated by the tracking software
    :param int fbi: female blob index (must be 0 or 1)
    :param int fps: video FPS (frames per second)
    :param float pxmm: number of pixels corresponding to 1 mm
    """

    def __init__(self, filename, fbi=0, fps=None, pxmm=None):
        self._load_from_csv(filename)
        self.filename = path.basename(filename)

        if fbi not in (0, 1):
            raise ValueError("'fbi' must be either 0 or 1")
        self.female_blob_index = fbi
        self.male_blob_index = 0 if fbi == 1 else 1

        self.fps = fps
        self.pxmm = pxmm

        self.female = None
        self.male = None
        self.common = None

        self._prepare_dataframes()

    def _load_from_csv(self, csvfile):
        """Read CSV file resulting from OpenCSP 2 Flies Tracker
        (no arena) into a Pandas DataFrame object.
        """

        # current size: 22
        column_titles = ["Frame",                               # 0
                         "Time [seconds]",                      # 1
                         "Blob Index",                          # 2
                         "Position X [pixels]",                 # 3
                         "Position Y [pixels]",                 # 4
                         "Rotation [degrees]",                  # 5
                         "Position Head X [pixels]",            # 6
                         "Position Head Y [pixels]",            # 7
                         "Position Tail X [pixels]",            # 8
                         "Position Tail Y [pixels]",            # 9
                         "Velocity [pixels/frame]",             # 10
                         "Velocity [pixels/second]",            # 11
                         "Velocity X [pixels/frame]",           # 12
                         "Velocity Y [pixels/frame]",           # 13
                         "Velocity X [pixels/second]",          # 14
                         "Velocity Y [pixels/second]",          # 15
                         "Angular Velocity [degrees/frame]",    # 16
                         "Angular Velocity [degrees/second]",   # 17
                         "Translational Angle [degrees]",       # 18
                         "Angle Between Flies [degrees]",       # 19
                         "Line Angle Between Flies [degrees]",  # 20
                         "Distance Between Blobs [pixels]"]     # 21

        self._data = pd.read_csv(csvfile,
                                 skiprows=6,  # initial rows have wrong length
                                 header=None,
                                 names=column_titles,  # use custom headers
                                 engine='python',
                                 skipfooter=2)  # last lines are corrupted

    def _prepare_dataframes(self):
        """Prepare DataFrame objects for both male and female
        Here we manipulate all the data read. We separate by gender and
        we remove unnacessary columns.
        """

        # Frame numbers should be seen as ints!
        self._data["Frame"] = self._data["Frame"].astype(int)

        self._remove_unnecessary_columns()

        # Convert tracked positions to millimeters.
        if not self.pxmm:
            raise ValueError("Must provide a calibration factor to convert")
        for column_name in ("Position X [pixels]", "Position Y [pixels]", "Distance Between Blobs [pixels]"):
            new_column_name = column_name.replace("pixels", "mm")
            self._data[new_column_name] = self._data[column_name] / self.pxmm

        self._split_by_gender()

        # Convert frames to time in seconds
        if not self.fps:
            raise ValueError("Must provide an FPS value to convert")
        for atr in ('female', 'male', 'common'):
            df = getattr(self, atr)
            df.reset_index(inplace=True)
            df["Time [s]"] = df["Frame"] / self.fps
            df.set_index("Frame", inplace=True)

        # Filter positions to remove tracker jitter FIXME
        for column_name in ("Position X [mm]", "Position Y [mm]"):
            self.female[column_name] = self.apply_1eurofilter_to_series(
                self.female[column_name])
            self.male[column_name] = self.apply_1eurofilter_to_series(
                self.male[column_name])

        self._calc_speed(self.female)
        self._calc_speed(self.male)

    def _remove_unnecessary_columns(self):
        """Remove unnecessary columns from the DataFrame loaded from
        the CSV.
        """
        columns_to_remove = [
            "Time [seconds]",
            "Velocity [pixels/frame]",
            "Velocity [pixels/second]",
            "Velocity X [pixels/frame]",
            "Velocity Y [pixels/frame]",
            "Velocity X [pixels/second]",
            "Velocity Y [pixels/second]",
            "Angular Velocity [degrees/frame]",
            "Angular Velocity [degrees/second]",
            "Translational Angle [degrees]",
            "Angle Between Flies [degrees]",
            "Line Angle Between Flies [degrees]",
            #"Distance Between Blobs [pixels]"
        ]
        for column_name in columns_to_remove:
            self._data.drop(column_name, axis=1, inplace=True)

    def _split_by_gender(self):
        """Split all loaded data by gender (coded in the column
        'Blob Index').
        The splitting is followed by a resetting of the index to the
        frame number.
        """
        self.female = self._data[
            self._data['Blob Index'] == self.female_blob_index
        ]
        self.male = self._data[
            self._data['Blob Index'] == self.male_blob_index
        ]

        self.female = self.female.set_index("Frame")
        self.male = self.male.set_index("Frame")

        # Fill in the missing frames where tracker failed
        # assuming it is the same for the male...
        new_index = pd.Index(np.arange(self.female.first_valid_index(),
                                       self.female.last_valid_index(),
                                       1))
        self.female = self.female.reindex(new_index)
        self.male = self.male.reindex(new_index)

        # reindex loses the name, so we must set it again
        self.female.index.name = "Frame"
        self.male.index.name = "Frame"

        # Create a common DataFrame
        self.common = pd.DataFrame(index=new_index)
        self.common.index.name = "Frame"

    def _convert_frames_to_seconds(self):
        """Convert frame values to seconds using the FPS attribute and
        add it to the DataFrame."""
        if not self.fps:
            raise ValueError("Must provide an FPS value to convert")
        for atr in ('female', 'male', 'common'):
            df = self.getattr(atr)
            df.reset_index()
            df["Time [s]"] = df["Frame"] / self.fps
            df.set_index("Frame")

    @staticmethod
    def _calc_speed(df):
        """Calculate speed (2D, absolute value) and add it to the DataFrame."""
        pos = df[["Time [s]",
                  "Position X [mm]",
                  "Position Y [mm]"]]
        dist = pos.diff().fillna(0.0)
        df["Speed [mm/s]"] = \
            np.sqrt(dist["Position X [mm]"]**2 + dist["Position Y [mm]"]**2) /\
            dist["Time [s]"]

    @staticmethod
    def apply_1eurofilter_to_series(s):
        """Apply 1€ Filter to a pandas Series."""
        filter_config = {
            'freq': 30,        # data update rate
            'mincutoff': 0.3,  # minimum cutoff frequency
            'beta': 1.6,       # cutoff slope
            'dcutoff': 1.0     # cutoff frequency for derivative
            }
        filter = OneEuroFilter(**filter_config)
        s = s.apply(filter)
        return s


if __name__ == '__main__':

    fn = '/media/hugo/DATA/R14H09/opencsp_output/all/MA_6_mtout.csv'

    td = TrackerData(fn, fbi=0, fps=30,  pxmm=31.060)

    # td.male = td.male.loc[322*30:336*30]

    # td.male["Position Y [mm]"].plot()
    # td.male["Speed [mm/s]"].plot(color='k', ls='-')
    # plt.show()


    # DONE TODO apply 1€ filter to position
    td.male.plot(x="Time [s]", y="Position X [mm]", style="ko-", lw=2)
    filter_config = {
        'freq': 30,        # data update rate
        'mincutoff': 0.3,  # minimum cutoff frequency
        'beta': 1.6,       # cutoff slope
        'dcutoff': 1.0     # cutoff frequency for derivative
        }
    filter = OneEuroFilter(**filter_config)
    column_to_filter = "Position X [mm]"
    td.male[column_to_filter] = td.male[column_to_filter].apply(filter)
    td.male.plot(x="Time [s]", y=column_to_filter, style="rx-", lw=1)
    plt.title(td.filename)
    plt.ylabel(column_to_filter)
    plt.tight_layout()
    plt.show()

    # # TODO correct rotation angle continuity
    # m_rotation = td.male["Rotation [degrees]"]#.iloc[2500:2930]
    # m_rotation.plot(c="k", marker='o', lw=2)
    # # plt.show()

    # # Correct angles
    # for i in range(1, m_rotation.size):
    #     curr = m_rotation[m_rotation.index[i]]
    #     prev = m_rotation[m_rotation.index[i-1]]
    #     if not np.isnan(prev):
    #         last_valid = prev
    #     if np.isnan(curr):
    #         m_rotation[m_rotation.index[i]] = curr
    #         continue
    #     diff = curr - last_valid
    #     while abs(diff) > 180:
    #         curr = curr - np.sign(diff)*360
    #         diff = curr - last_valid
    #     m_rotation[m_rotation.index[i]] = curr

    # m_rotation.plot(c="r", marker='.', lw=2)
    # m_rotation.diff().plot(c="b", ls='--')

    # # diff_error = 10  # degrees
    # # for i in range(1, m_rotation.size):

    # #     curr = m_rotation[m_rotation.index[i]]
    # #     prev = m_rotation[m_rotation.index[i-1]]

    # #     if not np.isnan(prev):
    # #         last_valid = prev

    # #     if np.isnan(curr):
    # #         m_rotation[m_rotation.index[i]] = curr
    # #         continue

    # #     diff = curr - last_valid

    # #     while abs(diff) > (360 - diff_error):
    # #         curr = curr - np.sign(diff)*360
    # #         diff = curr - last_valid

    # #     # # Account for diffs ~ 180º <- head tail swaps
    # #     # if abs(diff - 180) <= diff_error:
    # #     #     print("{:5} {:5.3f}".format(i, diff))

    # #     # Assign corrected value
    # #     m_rotation[m_rotation.index[i]] = curr

    # # m_rotation.plot(c="g", marker='.')

    # plt.title(td.filename)
    # plt.ylabel("Rotation [degrees]")
    # plt.tight_layout()
    # plt.show()
