"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Policy = void 0;
const core_1 = require("@aws-cdk/core");
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * The AWS::IAM::Policy resource associates an IAM policy with IAM users, roles, or groups.
 *
 * For more information about IAM policies, see [Overview of IAM
 * Policies](http://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html)
 * in the IAM User Guide guide.
 *
 * @stability stable
 */
class Policy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.policyName ||
                // generatePolicyName will take the last 128 characters of the logical id since
                // policy names are limited to 128. the last 8 chars are a stack-unique hash, so
                // that shouod be sufficient to ensure uniqueness within a principal.
                core_1.Lazy.string({ produce: () => util_1.generatePolicyName(scope, resource.logicalId) }),
        });
        /**
         * The policy document.
         *
         * @stability stable
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.referenceTaken = false;
        const self = this;
        class CfnPolicyConditional extends iam_generated_1.CfnPolicy {
            /**
             * This function returns `true` if the CFN resource should be included in
             * the cloudformation template unless `force` is `true`, if the policy
             * document is empty, the resource will not be included.
             */
            shouldSynthesize() {
                return self.force || self.referenceTaken || (!self.document.isEmpty && self.isAttached);
            }
        }
        if (props.document) {
            this.document = props.document;
        }
        const resource = new CfnPolicyConditional(this, 'Resource', {
            policyDocument: this.document,
            policyName: this.physicalName,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        this._policyName = this.physicalName;
        this.force = props.force !== undefined ? props.force : false;
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
    }
    /**
     * Import a policy in this app based on its name.
     *
     * @stability stable
     */
    static fromPolicyName(scope, id, policyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.policyName = policyName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a statement to the policy document.
     *
     * @stability stable
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     *
     * @stability stable
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
        user.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a role.
     *
     * @stability stable
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
        role.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a group.
     *
     * @stability stable
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
        group.attachInlinePolicy(this);
    }
    /**
     * The name of this policy.
     *
     * @stability stable
     * @attribute true
     */
    get policyName() {
        this.referenceTaken = true;
        return this._policyName;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            if (this.force) {
                result.push('Policy created with force=true is empty. You must add statements to the policy');
            }
            if (!this.force && this.referenceTaken) {
                result.push('This Policy has been referenced by a resource, so it must contain at least one statement.');
            }
        }
        // validate that the policy is attached to at least one principal (role, user or group).
        if (!this.isAttached) {
            if (this.force) {
                result.push('Policy created with force=true must be attached to at least one principal: user, group or role');
            }
            if (!this.force && this.referenceTaken) {
                result.push('This Policy has been referenced by a resource, so it must be attached to at least one user, group or role.');
            }
        }
        result.push(...this.document.validateForIdentityPolicy());
        return result;
    }
    /**
     * Whether the policy resource has been attached to any identity
     */
    get isAttached() {
        return this.groups.length + this.users.length + this.roles.length > 0;
    }
}
exports.Policy = Policy;
//# sourceMappingURL=data:application/json;base64,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