"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScopedAws = exports.Aws = void 0;
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const AWS_ACCOUNTID = 'AWS::AccountId';
const AWS_URLSUFFIX = 'AWS::URLSuffix';
const AWS_NOTIFICATIONARNS = 'AWS::NotificationARNs';
const AWS_PARTITION = 'AWS::Partition';
const AWS_REGION = 'AWS::Region';
const AWS_STACKID = 'AWS::StackId';
const AWS_STACKNAME = 'AWS::StackName';
const AWS_NOVALUE = 'AWS::NoValue';
/**
 * Accessor for pseudo parameters.
 *
 * Since pseudo parameters need to be anchored to a stack somewhere in the
 * construct tree, this class takes an scope parameter; the pseudo parameter
 * values can be obtained as properties from an scoped object.
 *
 * @stability stable
 */
class Aws {
    constructor() { }
}
exports.Aws = Aws;
/**
 * @stability stable
 */
Aws.ACCOUNT_ID = pseudoString(AWS_ACCOUNTID);
/**
 * @stability stable
 */
Aws.URL_SUFFIX = pseudoString(AWS_URLSUFFIX);
/**
 * @stability stable
 */
Aws.NOTIFICATION_ARNS = token_1.Token.asList({ Ref: AWS_NOTIFICATIONARNS }, { displayHint: AWS_NOTIFICATIONARNS });
/**
 * @stability stable
 */
Aws.PARTITION = pseudoString(AWS_PARTITION);
/**
 * @stability stable
 */
Aws.REGION = pseudoString(AWS_REGION);
/**
 * @stability stable
 */
Aws.STACK_ID = pseudoString(AWS_STACKID);
/**
 * @stability stable
 */
Aws.STACK_NAME = pseudoString(AWS_STACKNAME);
/**
 * @stability stable
 */
Aws.NO_VALUE = pseudoString(AWS_NOVALUE);
/**
 * Accessor for scoped pseudo parameters.
 *
 * These pseudo parameters are anchored to a stack somewhere in the construct
 * tree, and their values will be exported automatically.
 *
 * @stability stable
 */
class ScopedAws {
    /**
     * @stability stable
     */
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * @stability stable
     */
    get accountId() {
        return this.asString(AWS_ACCOUNTID);
    }
    /**
     * @stability stable
     */
    get urlSuffix() {
        return this.asString(AWS_URLSUFFIX);
    }
    /**
     * @stability stable
     */
    get notificationArns() {
        return token_1.Token.asList(cfn_reference_1.CfnReference.forPseudo(AWS_NOTIFICATIONARNS, this.scope), {
            displayHint: AWS_NOTIFICATIONARNS,
        });
    }
    /**
     * @stability stable
     */
    get partition() {
        return this.asString(AWS_PARTITION);
    }
    /**
     * @stability stable
     */
    get region() {
        return this.asString(AWS_REGION);
    }
    /**
     * @stability stable
     */
    get stackId() {
        return this.asString(AWS_STACKID);
    }
    /**
     * @stability stable
     */
    get stackName() {
        return this.asString(AWS_STACKNAME);
    }
    asString(name) {
        return token_1.Token.asString(cfn_reference_1.CfnReference.forPseudo(name, this.scope), { displayHint: name });
    }
}
exports.ScopedAws = ScopedAws;
function pseudoString(name) {
    // we don't want any ':' in the serialized form,
    // as ':' is the ARN separator,
    // and so we don't want ARN components
    // (which these CFN references like AWS::Partition certainly can be)
    // to contain ':'s themselves
    return token_1.Token.asString({ Ref: name }, { displayHint: name.replace('::', '.') });
}
//# sourceMappingURL=data:application/json;base64,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