"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const fs = require("fs");
const path = require("path");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("../construct-compat");
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 * @experimental
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * (experimental) An AWS-Lambda backed custom resource provider.
 *
 * @experimental
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const asset = stack.addFileAsset({
            fileName: staging.relativeStagedPath(stack),
            sourceHash: staging.sourceHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        this.roleArn = token_1.Token.asString(role.getAtt('Arn'));
        const timeout = (_a = props.timeout) !== null && _a !== void 0 ? _a : duration_1.Duration.minutes(15);
        const memory = (_b = props.memorySize) !== null && _b !== void 0 ? _b : size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: 'nodejs12.x',
                Environment: this.renderEnvironmentVariables(props.environment),
            },
        });
        handler.addDependsOn(role);
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * (experimental) Returns a stack-level singleton ARN (service token) for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @experimental
     */
    static getOrCreate(scope, uniqueid, props) {
        return this.getOrCreateProvider(scope, uniqueid, props).serviceToken;
    }
    /**
     * (experimental) Returns a stack-level singleton for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @experimental
     */
    static getOrCreateProvider(scope, uniqueid, props) {
        var _a;
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = (_a = stack.node.tryFindChild(id)) !== null && _a !== void 0 ? _a : new CustomResourceProvider(stack, id, props);
        return provider;
    }
    renderEnvironmentVariables(env) {
        if (!env || Object.keys(env).length === 0) {
            return undefined;
        }
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it)
        const variables = {};
        const keys = Object.keys(env).sort();
        for (const key of keys) {
            variables[key] = env[key];
        }
        return { Variables: variables };
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,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