"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.captureStackTrace = void 0;
const debug_1 = require("./debug");
/**
 * Captures the current process' stack trace.
 *
 * Stack traces are often invaluable tools to help diagnose problems, however
 * their capture is a rather expensive operation, and the stack traces can be
 * large. Consequently, users are stronly advised to condition capturing stack
 * traces to specific user opt-in.
 *
 * If the `CDK_DISABLE_STACK_TRACE` environment variable is set (to any value,
 * except for an empty string), no stack traces will be captured, and instead
 * the literal value `['stack traces disabled']` will be returned instead. This
 * is only true if the `CDK_DEBUG` environment variable is not set to `'true'`
 * or '1', in which case stack traces are *always* captured.
 *
 * @param below an optional function starting from which stack frames will be
 *              ignored. Defaults to the `captureStackTrace` function itself.
 * @param limit and optional upper bound to the number of stack frames to be
 *              captured. If not provided, this defaults to
 *              `Number.MAX_SAFE_INTEGER`, effectively meaning "no limit".
 *
 * @returns the captured stack trace, as an array of stack frames.
 */
function captureStackTrace(below = captureStackTrace, limit = Number.MAX_SAFE_INTEGER) {
    if (process.env.CDK_DISABLE_STACK_TRACE && !debug_1.debugModeEnabled()) {
        return ['stack traces disabled'];
    }
    const object = {};
    const previousLimit = Error.stackTraceLimit;
    try {
        Error.stackTraceLimit = limit;
        Error.captureStackTrace(object, below);
    }
    finally {
        Error.stackTraceLimit = previousLimit;
    }
    if (!object.stack) {
        return [];
    }
    return object.stack.split('\n').slice(1).map(s => s.replace(/^\s*at\s+/, ''));
}
exports.captureStackTrace = captureStackTrace;
//# sourceMappingURL=data:application/json;base64,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