"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default key'(test) {
        const stack = new core_1.Stack();
        new lib_1.Key(stack, 'MyKey');
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'default with no retention'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'TestStack');
        new lib_1.Key(stack, 'MyKey', { removalPolicy: core_1.RemovalPolicy.DESTROY });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', { DeletionPolicy: "Delete", UpdateReplacePolicy: "Delete" }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default with some permission'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey');
        const p = new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        }));
        test.done();
    },
    'key with some options'(test) {
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const p = new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        key.node.applyAspect(new core_1.Tag('tag1', 'value1'));
        key.node.applyAspect(new core_1.Tag('tag2', 'value2'));
        key.node.applyAspect(new core_1.Tag('tag3', ''));
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        Enabled: false,
                        EnableKeyRotation: true,
                        Tags: [
                            {
                                Key: "tag1",
                                Value: "value1"
                            },
                            {
                                Key: "tag2",
                                Value: "value2"
                            },
                            {
                                Key: "tag3",
                                Value: ""
                            }
                        ]
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        }));
        test.done();
    },
    'addAlias creates an alias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        const alias = key.addAlias('alias/xoo');
        test.ok(alias.aliasName);
        assert_1.expect(stack).toMatch({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        EnableKeyRotation: true,
                        Enabled: false,
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                },
                MyKeyAlias1B45D9DA: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/xoo",
                        TargetKeyId: {
                            "Fn::GetAtt": [
                                "MyKey6AB29FA6",
                                "Arn"
                            ]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'can run multiple addAlias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        const alias1 = key.addAlias('alias/alias1');
        const alias2 = key.addAlias('alias/alias2');
        test.ok(alias1.aliasName);
        test.ok(alias2.aliasName);
        assert_1.expect(stack).toMatch({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        EnableKeyRotation: true,
                        Enabled: false,
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                },
                MyKeyAlias1B45D9DA: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/alias1",
                        TargetKeyId: {
                            "Fn::GetAtt": [
                                "MyKey6AB29FA6",
                                "Arn"
                            ]
                        }
                    }
                },
                MyKeyAliasaliasalias2EC56BD3E: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/alias2",
                        TargetKeyId: {
                            "Fn::GetAtt": [
                                "MyKey6AB29FA6",
                                "Arn"
                            ]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'grant decrypt on a key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'Key');
        const user = new aws_iam_1.User(stack, 'User');
        // WHEN
        key.grantDecrypt(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    // This one is there by default
                    {
                        // tslint:disable-next-line:max-line-length
                        Action: ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*", "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion", "kms:GenerateDataKey"],
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::", { Ref: "AWS::AccountId" }, ":root"]] } },
                        Resource: "*"
                    },
                    // This is the interesting one
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::GetAtt": ["User00B015A1", "Arn"] } },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Key961B73FD", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'import/export can be used to bring in an existing key'(test) {
        const stack2 = new core_1.Stack();
        const myKeyImported = lib_1.Key.fromKeyArn(stack2, 'MyKeyImported', 'arn:of:key');
        // addAlias can be called on imported keys.
        myKeyImported.addAlias('alias/hello');
        assert_1.expect(stack2).toMatch({
            Resources: {
                MyKeyImportedAliasB1C5269F: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/hello",
                        TargetKeyId: 'arn:of:key'
                    }
                }
            }
        });
        test.done();
    },
    'addToResourcePolicy allowNoOp and there is no policy': {
        'succeed if set to true (default)'(test) {
            const stack = new core_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            key.addToResourcePolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
            test.done();
        },
        'fails if set to false'(test) {
            const stack = new core_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            test.throws(() => key.addToResourcePolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }), /* allowNoOp */ false), 'Unable to add statement to IAM resource policy for KMS key: "foo/bar"');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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