"""File containing one class responsible for writing the output sketch
"""

from collections import deque
import os


class Sketch:
    """Sketch object that has a number of methods that simplify working
    with the output file

    Args:
        output (pathlib.Path): Path to the output sketch directory
        indent (int): The number of spaces contained in one indent
    """

    def __init__(self, output, indent):
        output = output.absolute()

        # Create directories if they haven't been created yet
        os.makedirs(output, exist_ok=True)

        # Output filename must be the same as parent directory
        self.file = open(output / (output.name + '.ino'), 'w')

        # Prepare variables
        self.indent = ' ' * indent
        self.payloads = set()
        self.content = []

    def __del__(self):
        """The output file should be closed anyway
        """
        self.file.close()

    def add_text(self, text):
        """Adds text (C++ source) to the current sketch written in a special
        format (description of the format can be found in the documentation
        for :class:`pparser.payloads.Payload`)

        Args:
            text (str): Text to add
        """
        if text:
            self.content.append(text)

    def add_payload(self, payload):
        """Adds a payload to the current sketch

        Args:
            payload (pparser.payloads.Payload): payload to add
        """
        if payload:
            used = {}
            payloads = deque([payload])

            while payloads:
                payload = payloads.popleft()

                if used.get(payload):
                    continue

                used[payload] = True
                for depend in payload.depends:
                    payloads.append(depend)

            for payload in used:
                self.payloads.add(payload)

    def fprint(self, string='', *args, **kwargs):
        """Writes text to a file. Is an alias of the print function
        """
        print(string, file=self.file, *args, **kwargs)

    def make_indents(self, constructions, level=0, indented=None):
        """Turns a nested list into an indented text. Works recursively for
        each list

        Args:
            constructions (List[Union[List, str]]): nested list describing
                C++ sources

        Returns:
            str: indented C++ sources
        """
        if indented is None:
            indented = []

        for construction in constructions:
            if isinstance(construction, str):
                indented.append(self.indent * level + construction)
            else:
                self.make_indents(construction, level + 1, indented)

        return indented

    def flush(self):
        """Generates the final sketch (with all the paylods and commands)
        and writes it to a file
        """

        # Write banner
        self.fprint('''//
//   Generated by PotatoParser
//   Do not forget to star the repo :)
//   If you have any question or trouble, create issue
//   https://github.com/YariKartoshe4ka/PotatoParser
//

#include "Keyboard.h"


// Payload headers''')

        # Write payload headers
        for payload in self.payloads:
            self.fprint(payload.header)

        # Write `setup` funcition with parsed commands
        self.fprint(f'''

// Arduino functions
void setup() {{
{self.indent}// Initialize keyboard library
{self.indent}Keyboard.begin();
{self.indent}delay(2500);

{self.indent}// Start of payload''')

        for line in self.make_indents(self.content):
            self.fprint(line)

        self.fprint(f'''
{self.indent}// De-initialize keyboard library
{self.indent}Keyboard.end();
}}

void loop() {{ }}


// Payload definitions''')

        # Write payloads
        for payload in self.payloads:
            for line in self.make_indents(payload.text):
                self.fprint(line)
            self.fprint()
