"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Effect = exports.PolicyStatement = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const group_1 = require("./group");
const principals_1 = require("./principals");
const util_1 = require("./util");
const ensureArrayOrUndefined = (field) => {
    if (field === undefined) {
        return undefined;
    }
    if (typeof (field) !== 'string' && !Array.isArray(field)) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    if (Array.isArray(field) && !!field.find((f) => typeof (f) !== 'string')) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    return Array.isArray(field) ? field : [field];
};
/**
 * Represents a statement in an IAM policy document.
 *
 * @stability stable
 */
class PolicyStatement {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action) && !cdk.Token.isUnresolved(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.sid = props.sid;
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    /**
     * Creates a new PolicyStatement based on the object provided.
     *
     * This will accept an object created from the `.toJSON()` call
     *
     * @param obj the PolicyStatement in object form.
     * @stability stable
     */
    static fromJson(obj) {
        const ret = new PolicyStatement({
            sid: obj.Sid,
            actions: ensureArrayOrUndefined(obj.Action),
            resources: ensureArrayOrUndefined(obj.Resource),
            conditions: obj.Condition,
            effect: obj.Effect,
            notActions: ensureArrayOrUndefined(obj.NotAction),
            notResources: ensureArrayOrUndefined(obj.NotResource),
            principals: obj.Principal ? [new JsonPrincipal(obj.Principal)] : undefined,
            notPrincipals: obj.NotPrincipal ? [new JsonPrincipal(obj.NotPrincipal)] : undefined,
        });
        // validate that the PolicyStatement has the correct shape
        const errors = ret.validateForAnyPolicy();
        if (errors.length > 0) {
            throw new Error('Incorrect Policy Statement: ' + errors.join('\n'));
        }
        return ret;
    }
    //
    // Actions
    //
    /**
     * Specify allowed actions into the "Action" section of the policy statement.
     *
     * @param actions actions that will be allowed.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html
     * @stability stable
     */
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error('Cannot add \'Actions\' to policy statement if \'NotActions\' have been added');
        }
        this.action.push(...actions);
    }
    /**
     * Explicitly allow all actions except the specified list of actions into the "NotAction" section of the policy document.
     *
     * @param notActions actions that will be denied.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html
     * @stability stable
     */
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error('Cannot add \'NotActions\' to policy statement if \'Actions\' have been added');
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     *
     * @stability stable
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    /**
     * Adds principals to the "Principal" section of a policy statement.
     *
     * @param principals IAM principals that will be added.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
     * @stability stable
     */
    addPrincipals(...principals) {
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error('Cannot add \'Principals\' to policy statement if \'NotPrincipals\' have been added');
        }
        for (const principal of principals) {
            this.validatePolicyPrincipal(principal);
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Specify principals that is not allowed or denied access to the "NotPrincipal" section of a policy statement.
     *
     * @param notPrincipals IAM principals that will be denied access.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html
     * @stability stable
     */
    addNotPrincipals(...notPrincipals) {
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error('Cannot add \'NotPrincipals\' to policy statement if \'Principals\' have been added');
        }
        for (const notPrincipal of notPrincipals) {
            this.validatePolicyPrincipal(notPrincipal);
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    validatePolicyPrincipal(principal) {
        if (principal instanceof group_1.Group) {
            throw new Error('Cannot use an IAM Group as the \'Principal\' or \'NotPrincipal\' in an IAM Policy');
        }
    }
    /**
     * Specify AWS account ID as the principal entity to the "Principal" section of a policy statement.
     *
     * @stability stable
     */
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    /**
     * Specify a principal using the ARN  identifier of the principal.
     *
     * You cannot specify IAM groups and instance profiles as principals.
     *
     * @param arn ARN identifier of AWS account, IAM user, or IAM role (i.e. arn:aws:iam::123456789012:user/user-name).
     * @stability stable
     */
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts options for adding the service principal (such as specifying a principal in a different region).
     * @stability stable
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    /**
     * Adds a federated identity provider such as Amazon Cognito to this policy statement.
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com').
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    /**
     * Adds an AWS account root user principal to this policy statement.
     *
     * @stability stable
     */
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    /**
     * Adds a canonical user ID principal to this policy document.
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     * @stability stable
     */
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    /**
     * Adds all identities in all accounts ("*") to this policy statement.
     *
     * @stability stable
     */
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.AnyPrincipal());
    }
    //
    // Resources
    //
    /**
     * Specify resources that this policy statement applies into the "Resource" section of this policy statement.
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement applies to.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_resource.html
     * @stability stable
     */
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error('Cannot add \'Resources\' to policy statement if \'NotResources\' have been added');
        }
        this.resource.push(...arns);
    }
    /**
     * Specify resources that this policy statement will not apply to in the "NotResource" section of this policy statement.
     *
     * All resources except the specified list will be matched.
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement does not apply to.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html
     * @stability stable
     */
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error('Cannot add \'NotResources\' to policy statement if \'Resources\' have been added');
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     *
     * @stability stable
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission has at least one resource associated with it.
     *
     * @stability stable
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy.
     *
     * @stability stable
     */
    addCondition(key, value) {
        const existingValue = this.condition[key];
        this.condition[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Add multiple conditions to the Policy.
     *
     * @stability stable
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account.
     *
     * @stability stable
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    /**
     * JSON-ify the policy statement.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action, { unique: true }),
            NotAction: _norm(this.notAction, { unique: true }),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            NotPrincipal: _normPrincipal(this.notPrincipal),
            Resource: _norm(this.resource, { unique: true }),
            NotResource: _norm(this.notResource, { unique: true }),
            Sid: _norm(this.sid),
        });
        function _norm(values, { unique } = { unique: false }) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return unique ? [...new Set(values)] : values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            if (util_1.LITERAL_STRING_KEY in principal) {
                return principal[util_1.LITERAL_STRING_KEY][0];
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            return result;
        }
    }
    /**
     * String representation of this policy statement.
     *
     * @stability stable
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement',
        });
    }
    /**
     * JSON-ify the statement.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON() {
        return this.toStatementJson();
    }
    /**
     * Add a principal's conditions
     *
     * For convenience, principals have been modeled as both a principal
     * and a set of conditions. This makes it possible to have a single
     * object represent e.g. an "SNS Topic" (SNS service principal + aws:SourcArn
     * condition) or an Organization member (* + aws:OrgId condition).
     *
     * However, when using multiple principals in the same policy statement,
     * they must all have the same conditions or the OR samentics
     * implied by a list of principals cannot be guaranteed (user needs to
     * add multiple statements in that case).
     */
    addPrincipalConditions(conditions) {
        // Stringifying the conditions is an easy way to do deep equality
        const theseConditions = JSON.stringify(conditions);
        if (this.principalConditionsJson === undefined) {
            // First principal, anything goes
            this.principalConditionsJson = theseConditions;
        }
        else {
            if (this.principalConditionsJson !== theseConditions) {
                throw new Error(`All principals in a PolicyStatement must have the same Conditions (got '${this.principalConditionsJson}' and '${theseConditions}'). Use multiple statements instead.`);
            }
        }
        this.addConditions(conditions);
    }
    /**
     * Validate that the policy statement satisfies base requirements for a policy.
     *
     * @stability stable
     */
    validateForAnyPolicy() {
        const errors = new Array();
        if (this.action.length === 0 && this.notAction.length === 0) {
            errors.push('A PolicyStatement must specify at least one \'action\' or \'notAction\'.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for a resource-based policy.
     *
     * @stability stable
     */
    validateForResourcePolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length === 0 && Object.keys(this.notPrincipal).length === 0) {
            errors.push('A PolicyStatement used in a resource-based policy must specify at least one IAM principal.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for an identity-based policy.
     *
     * @stability stable
     */
    validateForIdentityPolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0) {
            errors.push('A PolicyStatement used in an identity-based policy cannot specify any IAM principals.');
        }
        if (Object.keys(this.resource).length === 0 && Object.keys(this.notResource).length === 0) {
            errors.push('A PolicyStatement used in an identity-based policy must specify at least one resource.');
        }
        return errors;
    }
}
exports.PolicyStatement = PolicyStatement;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatement[_a] = { fqn: "@aws-cdk/aws-iam.PolicyStatement", version: "1.132.0" };
/**
 * The Effect element of an IAM policy.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_effect.html
 * @stability stable
 */
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
class JsonPrincipal extends principals_1.PrincipalBase {
    constructor(json = {}) {
        super();
        // special case: if principal is a string, turn it into a "LiteralString" principal,
        // so we render the exact same string back out.
        if (typeof (json) === 'string') {
            json = { [util_1.LITERAL_STRING_KEY]: [json] };
        }
        if (typeof (json) !== 'object') {
            throw new Error(`JSON IAM principal should be an object, got ${JSON.stringify(json)}`);
        }
        this.policyFragment = {
            principalJson: json,
            conditions: {},
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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