# Copyright (c) 2024 Jintao Li.
# Computational and Interpretation Group (CIG),
# University of Science and Technology of China (USTC).
# All rights reserved.
"""
Common SEG-Y file information, including header parsing details and encoding conversion dictionaries.

This module contains essential constants and information related to SEG-Y files, which are 
widely used in geophysical and seismic data storage. It provides details for parsing SEG-Y headers 
and includes conversion dictionaries for ASCII and EBCDIC encodings.
"""

# fmt: off
kBinaryHeaderHelp = {
    1: ("Job ID", 4),
    5: ("Line number", 4),
    9: ("Reel Number", 4),
    13: ("N traces per ensembel", 2),
    15: ("N auxiliary traces per ensembel", 2),
    17: ("Sample interval(dt)", 2),
    19: ("dt of original", 2),
    21: ("N samples per traces(ns)", 2),
    23: ("ns of orignal", 2),
    25: ("Data sample format code (1-IBM, 5-IEEE)", 2),
    27: ("Ensemble fold", 2),
    29: ("Trace sorting code", 2),
    31: ("vertical sum code", 2),
    33: ("Sweep freq at start(Hz)", 2),
    35: ("Sweep freq at end(HZ)", 2),
    37: ("Sweep length(ms)", 2),
    39: ("Sweep type code", 2),
    41: ("Trace number of sweep channel", 2),
    43: ("Sweep trace taper length in ms at strat", 2),
    45: ("Sweep trace taper length in ms at end", 2),
    47: ("Taper type", 2),
    49: ("Correlated data traces", 2),
    51: ("Binary gain recovered", 2),
    53: ("Amplitude recovery method", 2),
    55: ("Measurement system (units)", 2),
    57: ("Impulse signal polarity", 2),
    59: ("Vibratory polariy code", 2),
    61: ("Extended number of data traces per ensemble", 4),
    65: ("Extended number of auxiliary traces per ensemble", 4),
    69: ("Extended number of samples per data trace", 4),
    73: ("Extended sample interval, IEEE double precision (64-bit)", 9),
    81: ("Extended sample interval of original field recording, IEEE double precision (64-bit)", 9),
    89: ("Extended number of samples per data trace in original recording", 4),
    93: ("Extended ensemble fold", 4),
    97: ("The integer constant 16909060_10 (01020304_16)", 4),
    101: ("Unassigned", 200),
    301: ("Major SEG-Y Format Revision Number. This is an 8-bit unsigned value", 1),
    302: ("Minor SEG-Y Format Revision Number. This is an 8-bit unsigned value with a radix point between the first and second bytes.", 1),
    303: ("Fixed length trace flag", 2),
    305: ("Number of 3200-byte, Extended Textual File Header records following the Binary Header", 2),
    307: ("Max number of additional 240-byte trace header", 4),
    311: ("Time basis code", 2),
    313: ("number of trace header in this file, 64-bit unsigned integer value", 9),
    321: ("Byte offset of first trace relative to start of file or stream if known, otherwise zero. (64-bit unsigned integer value) ", 9),
    329: ("Number of 3200-byte data trailer stanza records following the last trace (4 byte signed integer).", 4),
    333: ("Unassigned", 68),
}

kTraceHeaderHelp = {
    1: ("Trace sequence number within line", 4),
    5: ("Trace sequence number within SEG-Y file", 4),
    9: ("Original field record number", 4),
    13: ("Trace number within the original field record", 4),
    17: ("Energy source point number", 4),
    21: ("Ensemble number", 4),
    25: ("Trace number within the ensemble", 4),
    29: ("Trace identification code", 2),
    31: ("Number of vertically summed traces yielding this trace", 2),
    33: ("Number of horizontally stacked traces yielding this trace", 2),
    35: ("Data use for", 2),
    37: ("Distance from center of the source point to the center of the receiver group", 4),
    41: ("Elevation of receiver group", 4),
    45: ("Surface elevation at source location", 4),
    49: ("Source depth below surface", 4),
    53: ("Seismic Datum elevation at receiver group", 4),
    57: ("Seismic Datum elevation at source", 4),
    61: ("Water column height at source location", 4),
    65: ("Water column height at receiver group location", 4),
    69: ("Scalar to be applied to all elevations and depths", 2),
    71: ("Scalar to be applied to all coordinates", 2),
    73: ("Source coordinate - X", 4),
    77: ("Source coordinate - Y", 4),
    81: ("Group coordinate - X", 4),
    85: ("Group coordinate - Y", 4),
    89: ("Coordinate units", 2),
    91: ("Weathering velocity", 2),
    93: ("Subweathering velocity", 2),
    95: ("Uphole time at source in ms", 2),
    97: ("Uphole time at group in ms", 2),
    99: ("Source static correction in ms", 2),
    101: ("Group static correction in ms", 2),
    103: ("Total static applied in ms", 2),
    105: ("Lag time A", 2),
    107: ("Lag Time B", 2),
    109: ("Delay recording time", 2),
    111: ("Mute time - Start time in ms", 2),
    113: ("Mute time - End time in ms", 2),
    115: ("Number of samples in this trace", 2),
    117: ("Sample interval for this trace", 2),
    119: ("Gain type of field instruments", 2),
    121: ("Instrument gain constant", 2),
    123: ("Instrument gain constant", 2),
    125: ("Correlated", 2),
    127: ("Sweep frequency at start", 2),
    129: ("Sweep frequency at end", 2),
    131: ("Sweep length in ms", 2),
    133: ("Sweep type", 2),
    135: ("Sweep trace taper length at start in ms", 2),
    137: ("Sweep trace taper length at end in ms", 2),
    139: ("Taper type", 2),
    141: ("Alias filter frequency (Hz)", 2),
    143: ("Alias filter slope (dB/octave)", 2),
    145: ("Notch filter frequency (Hz)", 2),
    147: ("Notch filter slope (dB/octave)", 2),
    149: ("Low-cut frequency (Hz)", 2),
    151: ("High-cut frequency (Hz)", 2),
    153: ("Low-cut slope (dB/octave)", 2),
    155: ("High-cut slope (dB/octave)", 2),
    157: ("Year data recorded", 2),
    159: ("Day of year", 2),
    161: ("Hour of day", 2),
    163: ("Minute of hour", 2),
    165: ("Second of minute", 2),
    167: ("Time basis code", 2),
    169: ("Trace weighting factor", 2),
    171: ("Geophone group number of roll switch position one", 2),
    173: ("Geophone group number of trace number one within original field record", 2),
    175: ("Geophone group number of last trace within original field record", 2),
    177: ("Gap size (total number of groups dropped)", 2),
    179: ("Over travel associated with taper at beginning or end of line", 2),
    181: ("X coordinate", 4),
    185: ("Y coordinate", 4),
    189: ("The in-line number", 4),
    193: ("The cross-line number", 4),
    197: ("Shotpoint number", 4),
    201: ("Scalar to be applied to the shotpoint number", 2),
    203: ("Trace value measurement unit", 2),
    205: ("Transduction Constant", 6),
    211: ("Transduction Units", 2),
    213: ("Device/Trace Identifier", 2),
    215: ("Scalar to be applied to bytes 95-114", 2),
    217: ("Source Type/Orientation", 2),
    219: ("Source Energy Direction with respect to the source orientation", 6),
    225: ("Source Measurement - Describes the source effort used to generate the trace", 6),
    231: ("Source Measurement Unit", 2),
    233: ("Either binary zeros or chars SEG00000", 8),
}

kTraceSortingHelp = {
    -1: "Other",
    0: "Unknown",
    1: "As recorded (no sorting)",
    2: "CDP ensemble",
    3: "Single fold continuous profile",
    4: "Horizontally stacked",
    5: "Common source point",
    6: "Common receiver point",
    7: "Common offset point",
    8: "Common mid-point",
    9: "Common conversion point",
}

kDataSampleFormatHelp = {
    1: "4-byte IBM floating-point",
    2: "4-byte, two's complement integer",
    3: "2-byte, two's complement integer",
    4: "4-byte fixed-point with gain (obsolete) (not implemented)",
    5: "4-byte IEEE floating-point",
    6: "8-byte IEEE floating-point (not implemented)",
    7: "3-byte two's complement integer (not implemented)",
    8: "1-byte, two's complement integer",
    9: "8-byte, two's complement integer (not implemented)",
    10: "4-byte, unsigned integer",
    11: "2-byte, unsigned integer",
    12: "8-byte, unsigned integer (not implemented)",
    15: "3-byte, unsigned integer (not implemented)" ,
    16: "1-byte, unsigned integer"
}

kEBCDICtoASCIImap = {
    64: ' ', 75: '.', 76: '<', 77: '(', 78: '+', 79: '|',
    80: '&', 90: '!', 91: '$', 92: '*', 93: ')', 94: ';',
    96: '-', 97: '/', 106: '|', 107: ',', 108: '%', 109: '_',
    110: '>', 111: '?', 121: '`', 122: ':', 123: '#', 124: '@',
    125: '\'', 126: '=', 127: '"', 129: 'a', 130: 'b', 131: 'c',
    132: 'd', 133: 'e', 134: 'f', 135: 'g', 136: 'h', 137: 'i',
    145: 'j', 146: 'k', 147: 'l', 148: 'm', 149: 'n', 150: 'o',
    151: 'p', 152: 'q', 153: 'r', 161: '~', 162: 's', 163: 't',
    164: 'u', 165: 'v', 166: 'w', 167: 'x', 168: 'y', 169: 'z',
    192: '{', 193: 'A', 194: 'B', 195: 'C', 196: 'D', 197: 'E',
    198: 'F', 199: 'G', 200: 'H', 201: 'I', 208: '}', 209: 'J',
    210: 'K', 211: 'L', 212: 'M', 213: 'N', 214: 'O', 215: 'P',
    216: 'Q', 217: 'R', 224: '\\', 226: 'S', 227: 'T', 228: 'U',
    229: 'V', 230: 'W', 231: 'X', 232: 'Y', 233: 'Z', 240: '0',
    241: '1', 242: '2', 243: '3', 244: '4', 245: '5', 246: '6',
    247: '7', 248: '8', 249: '9'
}

kASCIItoEBCDICmap = {
    ' ': 64, '.': 75, '<': 76, '(': 77, '+': 78, '|': 79,
    '&': 80, '!': 90, '$': 91, '*': 92, ')': 93, ';': 94,
    '-': 96, '/': 97, '|': 106, ',': 107, '%': 108, '_': 109,
    '>': 110, '?': 111, '`': 121, ':': 122, '#': 123, '@': 124,
    '\'': 125, '=': 126, '"': 127, 'a': 129, 'b': 130, 'c': 131,
    'd': 132, 'e': 133, 'f': 134, 'g': 135, 'h': 136, 'i': 137,
    'j': 145, 'k': 146, 'l': 147, 'm': 148, 'n': 149, 'o': 150,
    'p': 151, 'q': 152, 'r': 153, '~': 161, 's': 162, 't': 163,
    'u': 164, 'v': 165, 'w': 166, 'x': 167, 'y': 168, 'z': 169,
    '{': 192, 'A': 193, 'B': 194, 'C': 195, 'D': 196, 'E': 197,
    'F': 198, 'G': 199, 'H': 200, 'I': 201, '}': 208, 'J': 209,
    'K': 210, 'L': 211, 'M': 212, 'N': 213, 'O': 214, 'P': 215,
    'Q': 216, 'R': 217, '\\': 224, 'S': 226, 'T': 227, 'U': 228,
    'V': 229, 'W': 230, 'X': 231, 'Y': 232, 'Z': 233, '0': 240,
    '1': 241, '2': 242, '3': 243, '4': 244, '5': 245, '6': 246,
    '7': 247, '8': 248, '9': 249
}

# fmt: on
