from datetime import datetime

import feedparser

from flask import url_for

from udata.core.dataset.factories import (
    ResourceFactory, DatasetFactory, LicenseFactory, CommunityResourceFactory,
)
from udata.core.user.factories import UserFactory
from udata.core.organization.factories import OrganizationFactory
from udata.models import Follow

from udata_front.tests import GouvFrSettings
from udata_front.tests.frontend import GouvfrFrontTestCase


class DatasetBlueprintTest(GouvfrFrontTestCase):
    settings = GouvFrSettings

    def test_render_display(self):
        '''It should render the dataset page'''
        dataset = DatasetFactory()
        url = url_for('datasets.show', dataset=dataset)
        response = self.get(url)
        self.assert200(response)
        self.assertNotIn(b'<meta name="robots" content="noindex, nofollow">',
                         response.data)

    def test_json_ld(self):
        '''It should render a json-ld markup into the dataset page'''
        resource = ResourceFactory(format='png',
                                   description='* Title 1\n* Title 2',
                                   metrics={'views': 10})
        license = LicenseFactory(url='http://www.datagouv.fr/licence')
        dataset = DatasetFactory(license=license,
                                 tags=['foo', 'bar'],
                                 resources=[resource],
                                 description='a&éèëù$£<>\'"',
                                 owner=UserFactory(),
                                 extras={'foo': 'bar'})
        community_resource = CommunityResourceFactory(
            dataset=dataset,
            format='csv',
            description='* Title 1\n* Title 2',
            metrics={'views': 42})

        url = url_for('datasets.show', dataset=dataset)
        response = self.get(url)
        self.assert200(response)
        json_ld = self.get_json_ld(response)
        self.assertEqual(json_ld['@context'], 'http://schema.org')
        self.assertEqual(json_ld['@type'], 'Dataset')
        self.assertEqual(json_ld['@id'], str(dataset.id))
        self.assertEqual(json_ld['description'], 'a&amp;éèëù$£&lt;&gt;&apos;&quot;')
        self.assertEqual(json_ld['alternateName'], dataset.slug)
        self.assertEqual(json_ld['dateCreated'][:16],
                         dataset.created_at.isoformat()[:16])
        self.assertEqual(json_ld['dateModified'][:16],
                         dataset.last_modified.isoformat()[:16])
        self.assertEqual(json_ld['url'], 'http://local.test{}'.format(url))
        self.assertEqual(json_ld['name'], dataset.title)
        self.assertEqual(json_ld['keywords'], 'bar,foo')
        self.assertEqual(len(json_ld['distribution']), 1)

        json_ld_resource = json_ld['distribution'][0]
        self.assertEqual(json_ld_resource['@type'], 'DataDownload')
        self.assertEqual(json_ld_resource['@id'], str(resource.id))
        self.assertEqual(json_ld_resource['url'], resource.latest)
        self.assertEqual(json_ld_resource['name'], resource.title)
        self.assertEqual(json_ld_resource['contentUrl'], resource.url)
        self.assertEqual(json_ld_resource['dateCreated'][:16],
                         resource.created_at.isoformat()[:16])
        self.assertEqual(json_ld_resource['dateModified'][:16],
                         resource.last_modified.isoformat()[:16])
        self.assertEqual(json_ld_resource['encodingFormat'], 'png')
        self.assertEqual(json_ld_resource['contentSize'],
                         resource.filesize)
        self.assertEqual(json_ld_resource['fileFormat'], resource.mime)
        self.assertEqual(json_ld_resource['description'],
                         'Title 1 Title 2')
        self.assertEqual(json_ld_resource['interactionStatistic'],
                         {
                              '@type': 'InteractionCounter',
                              'interactionType': {
                                  '@type': 'DownloadAction',
                              },
                              'userInteractionCount': 10,
                         })

        self.assertEqual(len(json_ld['contributedDistribution']), 1)
        json_ld_resource = json_ld['contributedDistribution'][0]
        self.assertEqual(json_ld_resource['@type'], 'DataDownload')
        self.assertEqual(json_ld_resource['@id'], str(community_resource.id))
        self.assertEqual(json_ld_resource['url'], community_resource.latest)
        self.assertEqual(json_ld_resource['name'], community_resource.title)
        self.assertEqual(json_ld_resource['contentUrl'],
                         community_resource.url)
        self.assertEqual(json_ld_resource['dateCreated'][:16],
                         community_resource.created_at.isoformat()[:16])
        self.assertEqual(json_ld_resource['dateModified'][:16],
                         community_resource.last_modified.isoformat()[:16])
        self.assertEqual(json_ld_resource['encodingFormat'],
                         community_resource.format)
        self.assertEqual(json_ld_resource['contentSize'],
                         community_resource.filesize)
        self.assertEqual(json_ld_resource['fileFormat'],
                         community_resource.mime)
        self.assertEqual(json_ld_resource['description'], 'Title 1 Title 2')
        self.assertEqual(json_ld_resource['interactionStatistic'], {
            '@type': 'InteractionCounter',
            'interactionType': {
                '@type': 'DownloadAction',
            },
            'userInteractionCount': 42,
        })

        self.assertEqual(json_ld['extras'],
                         [{
                              '@type': 'http://schema.org/PropertyValue',
                              'name': 'foo',
                              'value': 'bar',
                         }])
        self.assertEqual(json_ld['license'], 'http://www.datagouv.fr/licence')
        self.assertEqual(json_ld['author']['@type'], 'Person')

    def test_json_ld_sanitize(self):
        '''Json-ld should be sanitized'''
        dataset = DatasetFactory(description='an <script>evil()</script>')
        url = url_for('datasets.show', dataset=dataset)
        response = self.get(url)
        json_ld = self.get_json_ld(response)
        self.assertEqual(json_ld['description'],
                         'an &lt;script&gt;evil()&lt;/script&gt;')

    def test_raise_404_if_private(self):
        '''It should raise a 404 if the dataset is private'''
        dataset = DatasetFactory(private=True)
        response = self.get(url_for('datasets.show', dataset=dataset))
        self.assert404(response)

    def test_raise_410_if_deleted(self):
        '''It should raise a 410 if the dataset is deleted'''
        dataset = DatasetFactory(deleted=datetime.utcnow())
        response = self.get(url_for('datasets.show', dataset=dataset))
        self.assert410(response)

    def test_200_if_deleted_but_authorized(self):
        '''It should not raise a 410 if the can view it'''
        self.login()
        dataset = DatasetFactory(deleted=datetime.utcnow(), owner=self.user)
        response = self.get(url_for('datasets.show', dataset=dataset))
        self.assert200(response)

    def test_no_index_on_archived(self):
        '''It should prevent crawlers from indexing empty datasets'''
        dataset = DatasetFactory(archived=datetime.utcnow())
        response = self.get(url_for('datasets.show', dataset=dataset))
        self.assert200(response)
        self.assertIn(b'<meta name="robots" content="noindex, nofollow"',
                      response.data)

    def test_not_found(self):
        '''It should render the dataset page'''
        response = self.get(url_for('datasets.show', dataset='not-found'))
        self.assert404(response)

    def test_resource_latest_url(self):
        '''It should redirect to the real resource URL'''
        resource = ResourceFactory()
        DatasetFactory(resources=[resource])
        response = self.get(url_for('datasets.resource',
                                    id=resource.id))
        self.assertStatus(response, 302)
        self.assertEqual(response.location, resource.url)

    def test_community_resource_latest_url(self):
        '''It should redirect to the real community resource URL'''
        resource = CommunityResourceFactory()
        response = self.get(url_for('datasets.resource',
                                    id=resource.id))
        self.assertStatus(response, 302)
        self.assertEqual(response.location, resource.url)

    def test_resource_latest_url_404(self):
        '''It should return 404 if resource does not exists'''
        resource = ResourceFactory()
        response = self.get(url_for('datasets.resource',
                                    id=resource.id))
        self.assert404(response)

    def test_resource_latest_url_stripped(self):
        '''It should return strip extras spaces from the resource URL'''
        url = 'http://www.somewhere.com/path/with/spaces/   '
        resource = ResourceFactory(url=url)
        DatasetFactory(resources=[resource])
        response = self.get(url_for('datasets.resource',
                                    id=resource.id))
        self.assertStatus(response, 302)
        self.assertEqual(response.location, url.strip())

    def test_recent_feed(self):
        datasets = [DatasetFactory(
            resources=[ResourceFactory()]) for i in range(3)]

        response = self.get(url_for('datasets.recent_feed'))

        self.assert200(response)

        feed = feedparser.parse(response.data)

        self.assertEqual(len(feed.entries), len(datasets))
        for i in range(1, len(feed.entries)):
            published_date = feed.entries[i].published_parsed
            prev_published_date = feed.entries[i - 1].published_parsed
            self.assertGreaterEqual(prev_published_date, published_date)

    def test_recent_feed_owner(self):
        owner = UserFactory()
        DatasetFactory(owner=owner, resources=[ResourceFactory()])

        response = self.get(url_for('datasets.recent_feed'))

        self.assert200(response)

        feed = feedparser.parse(response.data)

        self.assertEqual(len(feed.entries), 1)
        entry = feed.entries[0]
        self.assertEqual(len(entry.authors), 1)
        author = entry.authors[0]
        self.assertEqual(author.name, owner.fullname)
        self.assertEqual(author.href,
                         self.full_url('users.show', user=owner.id))

    def test_recent_feed_org(self):
        owner = UserFactory()
        org = OrganizationFactory()
        DatasetFactory(
            owner=owner, organization=org, resources=[ResourceFactory()])

        response = self.get(url_for('datasets.recent_feed'))

        self.assert200(response)

        feed = feedparser.parse(response.data)

        self.assertEqual(len(feed.entries), 1)
        entry = feed.entries[0]
        self.assertEqual(len(entry.authors), 1)
        author = entry.authors[0]
        self.assertEqual(author.name, org.name)
        self.assertEqual(author.href,
                         self.full_url('organizations.show', org=org.id))

    def test_dataset_followers(self):
        '''It should render the dataset followers list page'''
        dataset = DatasetFactory()
        followers = [
            Follow.objects.create(follower=UserFactory(), following=dataset)
            for _ in range(3)
        ]

        response = self.get(url_for('datasets.followers', dataset=dataset))

        self.assert200(response)
        rendered_followers = self.get_context_variable('followers')
        self.assertEqual(len(rendered_followers), len(followers))
