"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressBackend = exports.Ingress = exports.HttpIngressPathType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
/**
 * Specify how the path is matched against request paths.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/ingress/#path-types
 */
var HttpIngressPathType;
(function (HttpIngressPathType) {
    /**
     * Matches the URL path exactly.
     */
    HttpIngressPathType["PREFIX"] = "Prefix";
    /**
     * Matches based on a URL path prefix split by '/'.
     */
    HttpIngressPathType["EXACT"] = "Exact";
    /**
     * Matching is specified by the underlying IngressClass.
     */
    HttpIngressPathType["IMPLEMENTATION_SPECIFIC"] = "ImplementationSpecific";
})(HttpIngressPathType = exports.HttpIngressPathType || (exports.HttpIngressPathType = {}));
/**
 * Ingress is a collection of rules that allow inbound connections to reach the
 * endpoints defined by a backend. An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 */
class Ingress extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'ingresses';
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngress(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                defaultBackend: cdk8s_1.Lazy.any({ produce: () => this._defaultBackend?._toKube() }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...props.rules ?? []);
        if (props.tls) {
            this.addTls(props.tls);
        }
        this.node.addValidation({ validate: () => this._validate() });
    }
    _validate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress. A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name. This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match
     * @param backend The backend to route to
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific
     * HTTP path (the `Host` header matches this value).
     *
     * @param host The host name
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addHostRule(host, path, backend, pathType) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path, pathType });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addRule(path, backend, pathType) {
        this.addRules({ backend, path, pathType });
    }
    /**
     * Adds rules to this ingress.
     * @param rules The rules to add
     */
    addRules(...rules) {
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = rule.host ?? '';
            const backend = rule.backend;
            const path = rule.path ?? '/';
            const pathType = rule.pathType ?? HttpIngressPathType.PREFIX;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = this._rulesPerHost[host] ?? [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({
                backend: backend._toKube(),
                path,
                pathType,
            });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: entry.secret?.name,
            });
        }
        return tls;
    }
}
exports.Ingress = Ingress;
_a = JSII_RTTI_SYMBOL_1;
Ingress[_a] = { fqn: "cdk8s-plus-23.Ingress", version: "2.4.13" };
/**
 * The backend for an ingress path.
 */
class IngressBackend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     * @param serv The service object.
     */
    static fromService(serv, options = {}) {
        if (serv.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (serv.ports.length === 1) {
            servicePort = serv.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = serv.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${serv.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${serv.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressBackend({
            service: {
                name: serv.name,
                port: { number: servicePort },
            },
        });
    }
    /**
     * A Resource backend is an ObjectRef to another Kubernetes resource
     * within the same namespace as the Ingress object.
     * A common usage for a Resource backend is to ingress data to an object
     * storage backend with static assets.
     */
    static fromResource(resource) {
        return new IngressBackend({
            resource: {
                kind: resource.kind,
                name: resource.name,
                apiGroup: resource.apiGroup,
            },
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressBackend = IngressBackend;
_b = JSII_RTTI_SYMBOL_1;
IngressBackend[_b] = { fqn: "cdk8s-plus-23.IngressBackend", version: "2.4.13" };
function sortByPath(lhs, rhs) {
    const p1 = lhs.path ?? '';
    const p2 = rhs.path ?? '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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