# -*- coding: utf-8 -*-
from aiida_quantumespresso.calculations.namelists import NamelistsCalculation
from aiida_quantumespresso.calculations.pw import PwCalculation
from aiida.common.utils import classproperty
from aiida.orm import DataFactory

class Pw2wannier90Calculation(NamelistsCalculation):
    """
    pw2wannier90.x code of the Quantum ESPRESSO distribution, handles the 
    calculation of the Amn, Mmn, ... files to be used to compute
    Wannier functions with the Wannier90 code.

    For more information, refer to http://www.quantum-espresso.org/
    and http://www.wannier.org/
    """
    def _init_internal_params(self):
        super(Pw2wannier90Calculation, self)._init_internal_params()
        self._default_namelists = ['INPUTPP']
        self._SEEDNAME = 'aiida'
        self._blocked_keywords = [
                                  ('INPUTPP','outdir',self._OUTPUT_SUBFOLDER),
                                  ('INPUTPP','prefix',self._PREFIX),
                                  ('INPUTPP','seedname',self._SEEDNAME),
                                 ]
        self._default_parser = None
        self._internal_retrieve_list = []

        # Default Matdyn output parser provided by AiiDA
        self._default_parser = 'quantumespresso.pw2wannier90'

    @classproperty
    def _use_methods(cls):
        """
        Additional use_* methods for the ph class.
        """
        SinglefileData = DataFactory('singlefile')

        retdict = NamelistsCalculation._use_methods
        retdict.update({
            "nnkp_file": {
               'valid_types': SinglefileData,
               'additional_parameter': None,
               'linkname': 'nnkp_file',
               'docstring': ("A SinglefileData containing the .nnkp file "
                    "generated by wannier90.x -pp"),
               },
            })
        return retdict

    def use_parent_calculation(self,calc):
        """
        Set the parent calculation,
        from which it will inherit the outputsubfolder.
        The link will be created from parent RemoteData and NamelistCalculation
        """
        if not isinstance(calc,PwCalculation):
            raise ValueError("Parent calculation must be a PwCalculation")
        from aiida.common.exceptions import UniquenessError
        localdatas = [_[1] for _ in calc.get_outputs(also_labels=True)]
        if len(localdatas) == 0:
            raise UniquenessError("No output retrieved data found in the parent"
                                  "calc, probably it did not finish yet, "
                                  "or it crashed")

        localdata = [_[1] for _ in calc.get_outputs(also_labels=True)
                              if _[0] == 'remote_folder']
        try:
            localdata = localdata[0]
        except IndexError:
            raise UniquenessError("No 'remote_folder' output node found in "
                "the parent_calc, probably it did not finish yet, "
                "or it crashed")
        self.use_parent_folder(localdata)

    def _prepare_for_submission(self, tempfolder, inputdict):
      """
      Create input, but also first perform additional operation like copying
      the input .nnkp
      """
      from aiida.common.exceptions import InputValidationError
      try:
          nnkp_file = inputdict.pop(self.get_linkname('nnkp_file'))
      except KeyError:
          raise InputValidationError("No nnkp_file specified for this calculation")

      ## Here pop out from inputdict['settings'] any custom settings, if needed
      ##

      # Run the global namelist logic
      calcinfo = super(Pw2wannier90Calculation, self)._prepare_for_submission(tempfolder, inputdict)

      # Put the nnkp in the folder, with the correct filename
      calcinfo.local_copy_list.append(
          (nnkp_file.get_file_abs_path(), "{}.nnkp".format(self._SEEDNAME))
      )

      return calcinfo
