from django.shortcuts import redirect
from psu_base.services import utility_service, auth_service, message_service
from psu_base.classes.Log import Log
from psu_base.models.xss import Xss
from django.urls import reverse
from django.http import HttpResponseForbidden

log = Log()


def xss_prevention(get_response):
    def script_response(param, value, is_ajax, path):
        log.trace([param])

        # Log the suspicious parameter
        log.error(f"Potential XSS attempt in '{param}' parameter")
        log.info(f"\n{value}\n")

        # Store attempt in database.
        xss_instance = Xss(
            app_code=utility_service.get_app_code(),
            path=path,
            username=auth_service.get_user().username,
            parameter_name=param,
            parameter_value=value
        )
        xss_instance.save()

        if is_ajax:
            # Generate a "flash message" to display on the view
            message_service.post_error("Suspicious input detected. Unable to process request.")

            # Return as failure for AJAX calls
            return HttpResponseForbidden()
        else:
            # Redirect to "suspicious input" page for non-AJAX requests
            return redirect('psu:xss')

    def xss_middleware(request):

        # Gather conditions and values used later
        is_ajax = request.is_ajax()
        is_health_check = utility_service.is_health_check()
        is_flash_messages = request.path == reverse('psu:messages')
        is_xss_lock = request.path == reverse('psu:xss_lock')
        is_terminating_impersonation = request.path == reverse('psu:stop_impersonating')
        is_logging_out = request.path == reverse('psu:logout')
        username = auth_service.get_user().username

        # Do not log on health check or flash messages (prevent cluttered logs)
        if (not is_health_check) and (not is_flash_messages):
            log.trace(['AJAX' if is_ajax else 'non-AJAX'])

        # Script_response will return a redirect.  If there are multiple XSS attempts, all attempts
        # should be logged, but only one return is needed. Store it in a variable while iterating.
        script_response_value = None

        # Iterate through GET parameters
        for param, value in request.GET.items():
            if utility_service.contains_script(value):
                # If XSS attempt found, log it and get a Redirect
                script_response_value = script_response(param, value, is_ajax, request.path)

        # Iterate through POST parameters
        for param, value in request.POST.items():
            if utility_service.contains_script(value):
                # If XSS attempt found, log it and get a Redirect
                script_response_value = script_response(param, value, is_ajax, request.path)

        # If xss was found, return the Redirect to the blocking page
        if script_response_value is not None:
            return script_response_value

        # If not already loading the lock page (and not an AWS health check)
        if username is not None and not (is_xss_lock or is_health_check or is_flash_messages):
            # Locked out users may logout or stop impersonating
            if not(is_terminating_impersonation or is_logging_out):
                # Count the number of un-reviewed XSS attempts for this user
                attempts = len(Xss.objects.filter(username=username, review_username__isnull=True))
                # After 3 attempts, user is locked out of site
                if attempts >= 3:
                    return redirect('psu:xss_lock')

        # Otherwise, continue normally (and add XSS-Protection header)
        response = get_response(request)
        response['X-XSS-Protection'] = "1"
        return response

    return xss_middleware
