import os
import json
import logging
import fnmatch
import requests
import socket
import time
import functools
from datetime import datetime, timedelta
from contextlib import contextmanager

from crc32c import crc32c  # pylint: disable=no-name-in-module


environment = os.environ.get("ENVIRONMENT", "dev")
STATSD_HOST = os.environ.get("STATSD_HOST", "localhost")
STATSD_PORT = int(os.environ.get("STATSD_PORT", "8125"))
sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
logger = logging.getLogger(__name__)


def push_message(message):
    try:
        sock.sendto(message.encode("utf-8"), (STATSD_HOST, STATSD_PORT))
    except:  # pylint: disable=bare-except
        # We occasionally get a "PermissionError: [Errno 1] Operation not permitted" on the VRay worker
        # here and don't fully understand why at this point. Regardless, we want to make super sure that
        # any exception generated by the feature flags client doesn't propagate to the app using it.
        # So catching any exception and not re-throwing seems the right thing to do here.
        logger.warning(
            "[FF] Failed push metrics to statsd, STATSD_HOST = %s", STATSD_HOST
        )


logger = logging.getLogger(__name__)


def push_metrics(flag_name, identifier, enabled):
    identifier = identifier.replace(".", "_") or "empty"
    gated = 1 if enabled else 0
    message = f"feature_flags.[flag_name={flag_name},identifier={identifier},environment={environment}]gated:{gated}|c"
    push_message(message)


def push_duration(duration_ms):
    message = f"feature_flags.[environment={environment}]duration:{duration_ms}|ms"
    push_message(message)


def push_exception():
    message = f"feature_flags.[environment={environment}]exception:1|c"
    push_message(message)


def timed_cache(**timedelta_kwargs):
    def _wrapper(f):
        update_delta = timedelta(**timedelta_kwargs)
        next_update = datetime.utcnow() + update_delta
        # Apply @lru_cache to f with no cache size limit
        f = functools.lru_cache(None)(f)

        @functools.wraps(f)
        def _wrapped(*args, **kwargs):
            nonlocal next_update
            now = datetime.utcnow()
            if now >= next_update:
                f.cache_clear()
                next_update = now + update_delta
            return f(*args, **kwargs)

        return _wrapped

    return _wrapper


class FeatureFlagClient:

    # bucket and folder names
    FF_BUCKET_NAME = os.environ.get("FF_BUCKET_NAME", "stitch-feature-flags")
    FF_PRIVATE = os.environ.get("FF_PRIVATE", "feature-flags-private")
    FF_PUBLIC = os.environ.get("FF_PUBLIC", "feature-flags")

    @timed_cache(seconds=60)
    def get_private_file_content(self):
        response = requests.get(
            f"https://stitch-feature-flags.s3.eu-central-1.amazonaws.com/{self.FF_PRIVATE}/{environment}.json"
        )
        flags_file_content = response.content.decode("utf-8")
        return json.loads(flags_file_content)

    @timed_cache(seconds=60)
    def get_public_file_content(self):
        response = requests.get(
            f"https://stitch-feature-flags.s3.eu-central-1.amazonaws.com/{self.FF_PUBLIC}/{environment}.json"
        )
        flags_file_content = response.content.decode("utf-8")
        return json.loads(flags_file_content)

    def check_is_visible(self, flag, identifier):
        if not flag["active"]:
            return False

        is_whitelisted = any(
            fnmatch.fnmatch(identifier, pattern) for pattern in flag["whitelist"]
        )
        is_blacklisted = any(
            fnmatch.fnmatch(identifier, pattern) for pattern in flag["blacklist"]
        )

        if not is_whitelisted or is_blacklisted:
            return False

        if flag.get("percentage"):
            active = self.is_experiment_active(flag, identifier)
            logger.info("flag: %s - enabled: %s", flag["name"], active)
            return active

        return True

    def is_experiment_active(self, flag, identifier):
        if not identifier:
            raise ValueError("identifier is required when using percentages")
        # We generate an arbitrary number between 0-99 based on the identifier
        calculated = crc32c(identifier.encode()) % 100
        # Any number from zero to "percentage" is visible; any number from
        # "percentage" to 99 is not visible
        return calculated < flag["percentage"]

    def is_enabled(self, flag_name, identifier=""):
        flag_enabled = False

        begin = time.time()
        try:
            private_flags_list = self.get_private_file_content()
            public_flags_list = self.get_public_file_content()

            flags_list = private_flags_list + public_flags_list

            for flag in flags_list:
                if flag["name"] == flag_name:
                    flag_enabled = self.check_is_visible(flag, identifier)
                    break
        except Exception:  # pylint: disable=broad-except
            logger.exception("error in is_enabled")
            push_exception()
        finally:
            end = time.time()
            # duration in milliseconds
            push_metrics(flag_name, identifier, flag_enabled)
            push_duration(int((end - begin) * 1000))
            return flag_enabled  # pylint: disable=lost-exception


# Create Singleton client that can be imported directly, instead of forcing
# the code to instance new clients. This improves the caching (since it'll
# be shared) and makes it easier to mock for testing.
#
# User code can simply use it as:
#    from flags_be_client import feature_flags
#    feature_flags.is_enabled("myflag")
feature_flags = FeatureFlagClient()


@contextmanager
def mock_ff(enabled_flags):
    """Set some flags as enabled, for testing

    Usage:
    with mock_ff(["flag1", "flag3"]):
      feature_flags.is_enabled("flag3") -> returns True
    """
    from unittest.mock import patch

    is_enabled = lambda flag: flag in enabled_flags
    with patch.object(feature_flags, "is_enabled", new=is_enabled):
        # this is what turns this function into a context manager
        # when someone calls
        # with mock_ff(flags):
        #     some code
        # the "some code" part will run when we reach the yield
        yield
