#  ---------------------------------------------------------------------------------
#  Copyright (c) 2025 DataRobot, Inc. and its affiliates. All rights reserved.
#  Last updated 2025.
#
#  DataRobot, Inc. Confidential.
#  This is proprietary source code of DataRobot, Inc. and its affiliates.
#
#  This file and its contents are subject to DataRobot Tool and Utility Agreement.
#  For details, see
#  https://www.datarobot.com/wp-content/uploads/2021/07/DataRobot-Tool-and-Utility-Agreement.pdf.
#  ---------------------------------------------------------------------------------
from enum import Enum

__GUARD_ASSOCIATION_IDS_COLUMN_NAME__ = "datarobot_guard_association_id"

LOGGER_NAME_PREFIX = "moderations"

DEFAULT_PROMPT_COLUMN_NAME = "promptText"
DEFAULT_RESPONSE_COLUMN_NAME = "completion"

NEMO_GUARDRAILS_DIR = "nemo_guardrails"


TOKEN_COUNT_COLUMN_NAME = "token_count"
ROUGE_1_COLUMN_NAME = "rouge_1"
NEMO_GUARD_COLUMN_NAME = "nemo_guard"
COST_COLUMN_NAME = "cost"
FAITHFULLNESS_COLUMN_NAME = "faithfulness"
AGENT_GOAL_ACCURACY_COLUMN_NAME = "agent_goal_accuracy"
TASK_ADHERENCE_SCORE_COLUMN_NAME = "task_adherence_score"

CUSTOM_METRIC_DESCRIPTION_SUFFIX = "Created by DataRobot Moderation System"

# Setting timeout at 10 sec, we have 5 retries, so we approximately wait for
# 50 sec, before giving up on the guard.
DEFAULT_GUARD_PREDICTION_TIMEOUT_IN_SEC = 10

# Connect and read retries count
RETRY_COUNT = 10

MODERATION_CONFIG_FILE_NAME = "moderation_config.yaml"
DATAROBOT_SERVERLESS_PLATFORM = "datarobotServerless"

RUNTIME_PARAMETER_PREFIX = "MLOPS_RUNTIME_PARAM_"
SECRET_DEFINITION_PREFIX = "MLOPS_RUNTIME_PARAM_MODERATION"
OPENAI_SECRET_DEFINITION_SUFFIX = "OPENAI_API_KEY"
GOOGLE_SERVICE_ACCOUNT_SECRET_DEFINITION_SUFFIX = "GOOGLE_SERVICE_ACCOUNT"
AWS_ACCOUNT_SECRET_DEFINITION_SUFFIX = "AWS_ACCOUNT"

NONE_CUSTOM_PY_RESPONSE = "None (No response was generated by LLM)"

MODERATION_MODEL_NAME = "DataRobot Moderation"
CHAT_COMPLETION_OBJECT = "chat.completion"
CHAT_COMPLETION_CHUNK_OBJECT = "chat.completion.chunk"
DATAROBOT_MODERATIONS_ATTR = "datarobot_moderations"
CITATIONS_ATTR = "citations"
USAGE_ATTR = "usage"
AGENTIC_PIPELINE_INTERACTIONS_ATTR = "pipeline_interactions"
LLM_BLUEPRINT_ID_ATTR = "llm_blueprint_id"
LLM_PROVIDER_GUARDS_ATTR = "llm_provider_guards"
PROMPT_VECTOR_ATTR = "prompt_vector"

DATAROBOT_CONFIGURED_ON_PREM_ST_SAAS_URL = "http://datarobot-nginx/api/v2"
DATAROBOT_ACTUAL_ON_PREM_ST_SAAS_URL = "http://datarobot-prediction-server:80/predApi/v1.0"

DISABLE_MODERATION_RUNTIME_PARAM_NAME = "DISABLE_MODERATION"
ENABLE_LLM_GATEWAY_INFERENCE_RUNTIME_PARAM_NAME = "ENABLE_LLM_GATEWAY_INFERENCE"

LLM_CONTEXT_COLUMN_NAME = "_LLM_CONTEXT"
PROMPT_TOKEN_COUNT_COLUMN_NAME_FROM_USAGE = "prompt_token_count_from_usage"
RESPONSE_TOKEN_COUNT_COLUMN_NAME_FROM_USAGE = "response_token_count_from_usage"

SPAN_PREFIX = "datarobot.guard"


class TargetType(str, Enum):
    """Target types that may be handed to moderations from DRUM -- casing must align."""

    BINARY = "binary"
    REGRESSION = "regression"
    ANOMALY = "anomaly"
    UNSTRUCTURED = "unstructured"
    MULTICLASS = "multiclass"
    TRANSFORM = "transform"
    TEXT_GENERATION = "textgeneration"
    GEO_POINT = "geopoint"
    VECTOR_DATABASE = "vectordatabase"
    AGENTIC_WORKFLOW = "agenticworkflow"

    @staticmethod
    def guards():
        return [TargetType.TEXT_GENERATION, TargetType.AGENTIC_WORKFLOW]

    @staticmethod
    def vdb():
        return [TargetType.VECTOR_DATABASE]


class GuardType:
    OOTB = "ootb"  # Out of the Box
    MODEL = "model"  # wraps a deployed model
    NEMO_GUARDRAILS = "nemo_guardrails"  # Nemo guardrails

    ALL = [MODEL, NEMO_GUARDRAILS, OOTB]


class OOTBType:
    TOKEN_COUNT = "token_count"
    ROUGE_1 = "rouge_1"
    FAITHFULNESS = "faithfulness"
    AGENT_GOAL_ACCURACY = "agent_goal_accuracy"
    CUSTOM_METRIC = "custom_metric"
    COST = "cost"
    TASK_ADHERENCE = "task_adherence"

    ALL = [
        TOKEN_COUNT,
        ROUGE_1,
        FAITHFULNESS,
        CUSTOM_METRIC,
        COST,
        AGENT_GOAL_ACCURACY,
        TASK_ADHERENCE,
    ]


class CostCurrency:
    USD = "USD"

    ALL = [USD]


class GuardStage:
    """When can this guard operate?"""

    PROMPT = "prompt"
    RESPONSE = "response"

    ALL = [PROMPT, RESPONSE]


class GuardExecutionLocation(Enum):
    LOCAL = "local"
    REMOTE = "remote"

    ALL = [LOCAL, REMOTE]


class GuardAction:
    """
    Defines actions a guard can take.
    All guards report their decisions; 'report' means do nothing else.
    """

    BLOCK = "block"
    REPORT = "report"
    REPLACE = "replace"
    NONE = None

    ALL = [BLOCK, REPORT, REPLACE, NONE]

    @classmethod
    def possible_column_names(cls, input_column_name):
        return [
            f"{cls.BLOCK}ed_{input_column_name}",
            f"{cls.REPORT}ed_{input_column_name}",
            f"{cls.REPLACE}d_{input_column_name}",
        ]


class GuardModelTargetType:
    """
    Guards support a subset of DataRobot model target types.
    Ref: common.engine.TargetType
    """

    BINARY = "Binary"
    REGRESSION = "Regression"
    MULTICLASS = "Multiclass"
    TEXT_GENERATION = "TextGeneration"

    ALL = [BINARY, REGRESSION, MULTICLASS, TEXT_GENERATION]


class GuardOperatorType:
    """
    Defines what the guard should do with the metric or prediction.
    Applies when the parameter type is OPERATOR.
    Typically this compares against a number, or looks for a matching category or string.
    """

    GREATER_THAN = "greaterThan"
    LESS_THAN = "lessThan"
    EQUALS = "equals"
    NOT_EQUALS = "notEquals"
    IS = "is"
    IS_NOT = "isNot"
    MATCHES = "matches"
    DOES_NOT_MATCH = "doesNotMatch"
    CONTAINS = "contains"
    DOES_NOT_CONTAIN = "doesNotContain"

    ALL = [
        GREATER_THAN,
        LESS_THAN,
        EQUALS,
        NOT_EQUALS,
        IS,
        IS_NOT,
        MATCHES,
        DOES_NOT_MATCH,
        CONTAINS,
        DOES_NOT_CONTAIN,
    ]

    REQUIRES_LIST_COMPARAND = [MATCHES, DOES_NOT_MATCH, CONTAINS, DOES_NOT_CONTAIN]


class GuardLLMType:
    """LLM Types to use for guards"""

    OPENAI = "openAi"
    AZURE_OPENAI = "azureOpenAi"
    GOOGLE = "google"
    AMAZON = "amazon"
    DATAROBOT = "datarobot"
    NIM = "nim"

    ALL = [OPENAI, AZURE_OPENAI, GOOGLE, AMAZON, DATAROBOT, NIM]


class GoogleModel:
    CHAT_BISON = "chat-bison"
    GEMINI_15_FLASH = "google-gemini-1.5-flash"
    GEMINI_15_PRO = "google-gemini-1.5-pro"

    ALL = [CHAT_BISON, GEMINI_15_FLASH, GEMINI_15_PRO]


class AwsModel:
    TITAN = "amazon-titan"
    ANTHROPIC_CLAUDE_2 = "anthropic-claude-2"
    ANTHROPIC_CLAUDE_3_HAIKU = "anthropic-claude-3-haiku"
    ANTHROPIC_CLAUDE_3_SONNET = "anthropic-claude-3-sonnet"
    ANTHROPIC_CLAUDE_3_OPUS = "anthropic-claude-3-opus"
    ANTHROPIC_CLAUDE_3_5_SONNET_V1 = "anthropic-claude-3.5-sonnet-v1"
    ANTHROPIC_CLAUDE_3_5_SONNET_V2 = "anthropic-claude-3.5-sonnet-v2"
    AMAZON_NOVA_LITE = "amazon-nova-lite"
    AMAZON_NOVA_MICRO = "amazon-nova-micro"
    AMAZON_NOVA_PRO = "amazon-nova-pro"

    ALL = [
        TITAN,
        ANTHROPIC_CLAUDE_2,
        ANTHROPIC_CLAUDE_3_HAIKU,
        ANTHROPIC_CLAUDE_3_SONNET,
        ANTHROPIC_CLAUDE_3_OPUS,
        ANTHROPIC_CLAUDE_3_5_SONNET_V1,
        ANTHROPIC_CLAUDE_3_5_SONNET_V2,
        AMAZON_NOVA_LITE,
        AMAZON_NOVA_MICRO,
        AMAZON_NOVA_PRO,
    ]


class GoogleModelVersion:
    CHAT_BISON = "chat-bison"
    GEMINI_15_FLASH = "gemini-1.5-flash-002"
    GEMINI_15_PRO = "gemini-1.5-pro-002"

    ALL = [CHAT_BISON, GEMINI_15_FLASH, GEMINI_15_PRO]


class AwsModelVersion:
    TITAN = "amazon.titan-text-express-v1"
    ANTHROPIC_CLAUDE_2 = "anthropic.claude-v2:1"
    ANTHROPIC_CLAUDE_3_HAIKU = "anthropic.claude-3-haiku-20240307-v1:0"
    ANTHROPIC_CLAUDE_3_SONNET = "anthropic.claude-3-sonnet-20240229-v1:0"
    ANTHROPIC_CLAUDE_3_OPUS = "anthropic.claude-3-opus-20240229-v1:0"
    ANTHROPIC_CLAUDE_3_5_SONNET_V1 = "anthropic.claude-3-5-sonnet-20240620-v1:0"
    ANTHROPIC_CLAUDE_3_5_SONNET_V2 = "anthropic.claude-3-5-sonnet-20241022-v2:0"
    AMAZON_NOVA_LITE = "amazon.nova-lite-v1:0"
    AMAZON_NOVA_MICRO = "amazon.nova-micro-v1:0"
    AMAZON_NOVA_PRO = "amazon.nova-pro-v1:0"

    ALL = [
        TITAN,
        ANTHROPIC_CLAUDE_2,
        ANTHROPIC_CLAUDE_3_HAIKU,
        ANTHROPIC_CLAUDE_3_SONNET,
        ANTHROPIC_CLAUDE_3_OPUS,
        ANTHROPIC_CLAUDE_3_5_SONNET_V1,
        ANTHROPIC_CLAUDE_3_5_SONNET_V2,
        AMAZON_NOVA_LITE,
        AMAZON_NOVA_MICRO,
        AMAZON_NOVA_PRO,
    ]


GOOGLE_MODEL_TO_GOOGLE_MODEL_VERSION_MAP = {
    GoogleModel.CHAT_BISON: GoogleModelVersion.CHAT_BISON,
    GoogleModel.GEMINI_15_FLASH: GoogleModelVersion.GEMINI_15_FLASH,
    GoogleModel.GEMINI_15_PRO: GoogleModelVersion.GEMINI_15_PRO,
}

AWS_MODEL_TO_AWS_MODEL_VERSION_MAP = {
    AwsModel.TITAN: AwsModelVersion.TITAN,
    AwsModel.ANTHROPIC_CLAUDE_2: AwsModelVersion.ANTHROPIC_CLAUDE_2,
    AwsModel.ANTHROPIC_CLAUDE_3_HAIKU: AwsModelVersion.ANTHROPIC_CLAUDE_3_HAIKU,
    AwsModel.ANTHROPIC_CLAUDE_3_SONNET: AwsModelVersion.ANTHROPIC_CLAUDE_3_SONNET,
    AwsModel.ANTHROPIC_CLAUDE_3_OPUS: AwsModelVersion.ANTHROPIC_CLAUDE_3_OPUS,
    AwsModel.ANTHROPIC_CLAUDE_3_5_SONNET_V1: AwsModelVersion.ANTHROPIC_CLAUDE_3_5_SONNET_V1,
    AwsModel.ANTHROPIC_CLAUDE_3_5_SONNET_V2: AwsModelVersion.ANTHROPIC_CLAUDE_3_5_SONNET_V2,
    AwsModel.AMAZON_NOVA_LITE: AwsModelVersion.AMAZON_NOVA_LITE,
    AwsModel.AMAZON_NOVA_MICRO: AwsModelVersion.AMAZON_NOVA_MICRO,
    AwsModel.AMAZON_NOVA_PRO: AwsModelVersion.AMAZON_NOVA_PRO,
}


class GuardTimeoutAction:
    """Actions if the guard times out"""

    # Block the prompt / response if the guard times out
    BLOCK = "block"

    # Continue scoring prompt / returning response if the
    # guard times out
    SCORE = "score"

    ALL = [BLOCK, SCORE]


class ModerationEventTypes:
    MODERATION_METRIC_CREATION_ERROR = "moderationMetricCreationError"
    MODERATION_METRIC_REPORTING_ERROR = "moderationMetricReportingError"
    MODERATION_MODEL_CONFIG_ERROR = "moderationModelConfigError"
    MODERATION_MODEL_RUNTIME_ERROR = "moderationModelRuntimeError"
    MODERATION_MODEL_SCORING_ERROR = "moderationModelScoringError"

    # Reported in sync context
    SYNC_EVENTS = [
        MODERATION_METRIC_CREATION_ERROR,
        MODERATION_METRIC_REPORTING_ERROR,
        MODERATION_MODEL_SCORING_ERROR,
    ]
