"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const event_bridge_1 = require("./event-bridge");
/**
 * Use a Lambda function as a Lambda destination
 */
class LambdaDestination {
    constructor(fn, options = {}) {
        this.fn = fn;
        this.options = options;
    }
    /**
     * Returns a destination configuration
     */
    bind(scope, fn, options) {
        // Normal Lambda destination (full invocation record)
        if (!this.options.responseOnly) {
            // deduplicated automatically
            this.fn.grantInvoke(fn);
            return {
                destination: this.fn.functionArn,
            };
        }
        // Otherwise add rule to extract the response payload and use EventBridge
        // as destination
        if (!options) { // `options` added to bind() as optionnal to avoid breaking change
            throw new Error('Options must be defined when using `responseOnly`.');
        }
        // Match invocation result of the source function (`fn`) and use it
        // to trigger the destination function (`this.fn`).
        new events.Rule(scope, options.type, {
            eventPattern: {
                detailType: [`Lambda Function Invocation Result - ${options.type}`],
                resources: [`${fn.functionArn}:$LATEST`],
                source: ['lambda'],
            },
            targets: [
                new targets.LambdaFunction(this.fn, {
                    event: events.RuleTargetInput.fromEventPath('$.detail.responsePayload'),
                }),
            ],
        });
        const destination = new event_bridge_1.EventBridgeDestination(); // Use default event bus here
        return destination.bind(scope, fn);
    }
}
exports.LambdaDestination = LambdaDestination;
//# sourceMappingURL=data:application/json;base64,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