#!/usr/bin/env python3
# -*- coding: utf-8 -*-
from dataclasses import dataclass, field
from typing import List, Dict, Any
import json
from ..Utils.file_utils import CustomJSONEncoder


@dataclass
class APKOverview:
    """
    Represents the APK overview results generated by Androguards (Manifest) Analysis.

    Attributes:
        general_info (dict): General information about the APK file.
        components (dict): Components like activities, services, etc.
        permissions (dict): Permissions declared and used in the APK.
        certificates (dict): Signing certificates information.
        native_libs (list): List of native libaries in the APK.
        directory_listing (list): List of files in the APK.
        is_cross_platform (bool): Indicate if this a cross-platform APK
        cross_platform_framework (bool): name of the cross-platform framework
        permissions_details (dict): Detailed permissions analysis (optional).
    """
    general_info: Dict[str, Any] = field(default_factory=dict)
    components: Dict[str, Any] = field(default_factory=dict)
    permissions: Dict[str, Any] = field(default_factory=dict)
    certificates: Dict[str, Any] = field(default_factory=dict)
    native_libs: List[str] = field(default_factory=list)
    directory_listing: List[str] = field(default_factory=list)
    is_cross_platform: bool = field(default_factory=bool)
    cross_platform_framework: str = field(default_factory=str)
    permissions_details: Dict[str, Any] = field(default_factory=dict)

    def __post_init__(self):
        """
        Ensures that `permissions_details` is included only if provided during initialization.
        """
        # Remove `permissions_details` if it's empty or not provided
        if not self.permissions_details:
            delattr(self, "permissions_details")

    # Utility Methods
    def to_dict(self) -> Dict[str, Any]:
        """
        Converts the APKOverview object to a dictionary.

        Returns:
            dict: The object represented as a dictionary, excluding optional fields if not set.
        """
        data = {
            "general_info": self.general_info,
            "components": self.components,
            "permissions": self.permissions,
            "certificates": self.certificates,
            "native_libs": self.native_libs,
            "directory_listing": self.directory_listing,
            "is_cross_platform": self.is_cross_platform,
            "cross_platform_framework": self.cross_platform_framework
        }
        # Include `permissions_details` only if it exists
        if hasattr(self, "permissions_details"):
            data["permissions_details"] = self.permissions_details
        return data

    def to_json(self) -> str:
        """
        Converts the APKOverview object to a JSON string.

        Returns:
            str: The object represented as a JSON string.
        """
        return json.dumps(self.to_dict(), cls=CustomJSONEncoder, indent=4)

    def print_results(self):
        """
        Prints the APKOverview object in a readable JSON format.
        """
        print(self.to_json())

    def update_from_dict(self, updates: Dict[str, Any]):
        """
        Updates fields of the APKOverview object using a dictionary.

        Args:
            updates (dict): A dictionary containing updates for the fields.
        """
        if "general_info" in updates:
            self.general_info.update(updates["general_info"])
        if "components" in updates:
            self.components.update(updates["components"])
        if "permissions" in updates:
            self.permissions.update(updates["permissions"])
        if "certificates" in updates:
            self.certificates.update(updates["certificates"])
        if "native_libs" in updates:
            self.native_libs.extend(updates["native_libs"])
        if "directory_listing" in updates:
            self.directory_listing.extend(updates["directory_listing"])
        if "is_cross_platform" in updates:
            self.is_cross_platform.extend(updates["is_cross_platform"])
        if "cross_platform_framework" in updates:
            self.cross_platform_framework.extend(updates["cross_platform_framework"])

    
    def pretty_print(self):
        """
        Prints a formatted overview of the APK details.

        Args:
            do_in_depth_analysis (bool): Whether to include in-depth analysis in the output.
        """
        print("\n=== General Info ===")
        for key, value in self.general_info.items():
            print(f"{key.replace('_', ' ').title()}: {value}")

        print("\n=== Components ===")
        for component_type, components in self.components.items():
            print(f"{component_type.replace('_', ' ').title()}:")
            if components:
                print("\n".join(f"- {item}" for item in components))
            else:
                print("No items found.")

        print("\n=== Permissions ===")
        print("Declared Permissions:")
        declared_permissions = self.permissions.get("declared_permissions", [])
        if declared_permissions is not None or len(declared_permissions) > 0:
            if declared_permissions:
                print("\n".join(f"- {perm}" for perm in declared_permissions))
            else:
                print("No declared permissions.")

        print("Permissions:")
        permissions = self.permissions.get("permissions", [])
        if permissions:
            print("\n".join(f"- {perm}" for perm in permissions))
        else:
            print("No permissions.")

        print("\n=== Certificates ===")
        for version, certs in self.certificates.items():
            print(f"Version: {version}")
            for cert in certs:
                for field_name, value in cert.items():
                    print(f"{field_name.replace('_', ' ').title()}: {value}")
                print()

        print("\n=== Native Libraries ===")
        if self.native_libs:
            max_native_files_to_print = 5
            native_files_to_print = self.native_libs[:max_native_files_to_print]
            print("\n".join(native_files_to_print))

            if len(self.native_libs) > max_native_files_to_print:
                print(f"...and {len(self.native_libs) - max_native_files_to_print} more files. See generated json.")
        else:
            print("APK has no native libraries")
        
        print("\n=== Directory Listing ===")
        if self.directory_listing:        
            max_files_to_print = 10
            files_to_print = self.directory_listing[:max_files_to_print]

            print("\n".join(files_to_print))

            # Indicate if there are more files
            if len(self.directory_listing) > max_files_to_print:
                print(f"...and {len(self.directory_listing) - max_files_to_print} more files. See generated json.")
        else:
            print("No files found.")

        if self.is_cross_platform:
            print(f"\nCross-Platform APK: build using {self.cross_platform_framework}")

        # Print permissions_details if available
        if hasattr(self, "permissions_details"):
            print("\n=== Permissions Details ===")
            for key, value in self.permissions_details.items():
                print(f"{key.replace('_', ' ').title()}: {value}")


    
    # Property for MD5 (example of encapsulation)
    @property
    def md5(self) -> str:
        """Returns the MD5 checksum from general_info."""
        return self.general_info.get("md5", "")

    @md5.setter
    def md5(self, value: str):
        """Sets the MD5 checksum in general_info with validation."""
        if not isinstance(value, str):
            raise ValueError("MD5 must be a string.")
        self.general_info["md5"] = value
