import platform

from typing import Dict, Any


def cpu_codename() -> str:
    assert platform.system() in [
        "Linux"
    ], f"LIKWID is not supported {platform.system()}"

    import daisy_likwid_helpers

    daisy_likwid_helpers.inittopology()
    return daisy_likwid_helpers.getcpuinfo()["short_name"]


def gpu_codename() -> str:
    assert platform.system() in [
        "Linux"
    ], f"LIKWID is not supported {platform.system()}"

    import daisy_likwid_helpers

    gpu = daisy_likwid_helpers.getgputopology()[0]
    gpu_codename = "nvidia_cc_ge_7" if gpu["ccapMajor"] >= 7 else "nvidia_cc_le_7"
    return gpu_codename


def cpu_topology() -> str:
    assert platform.system() in [
        "Linux"
    ], f"LIKWID is not supported {platform.system()}"

    import daisy_likwid_helpers

    daisy_likwid_helpers.inittopology()
    return daisy_likwid_helpers.getcputopology()


def likwid_groups(device: str) -> Dict[str, Any]:
    assert platform.system() in [
        "Linux"
    ], f"LIKWID is not supported {platform.system()}"

    import daisy_likwid_helpers

    if device == "cpu":
        daisy_likwid_helpers.inittopology()

        # CPU
        cpu_codename = daisy_likwid_helpers.getcpuinfo()["short_name"]
        cpu_groups = LIKWID_GROUPS[cpu_codename]
        return cpu_groups
    elif device == "gpu":
        daisy_likwid_helpers.initgputopology()

        # Currently single GPU supported
        gpu = daisy_likwid_helpers.getgputopology()[0]
        gpu_codename = "nvidia_cc_ge_7" if gpu["ccapMajor"] >= 7 else "nvidia_cc_le_7"
        gpu_groups = LIKWID_GROUPS[gpu_codename]
        return gpu_groups
    else:
        raise ValueError(f"Invalid device type {device}")


LIKWID_GROUPS = {
    "nehalem": [
        "TLB",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "FLOPS_X87",
        "FLOPS_DP",
        "DIVIDE",
        "CACHE",
        "L3CACHE",
        "L2",
        "DATA",
        "SCHEDULER",
        "L2CACHE",
    ],
    "arm8_n1": ["TLB", "ICACHE", "MEM", "BRANCH", "L3", "CLOCK", "L2", "DATA"],
    "westmereEX": [
        "NUMA",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "UOPS",
        "FLOPS_X87",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "CACHE",
        "L3CACHE",
        "L2",
        "DATA",
        "L2CACHE",
    ],
    "phi": [
        "TLB",
        "READ_MISS_RATIO",
        "MEM",
        "VPU_WRITE_MISS_RATIO",
        "MEM6",
        "VPU_READ_MISS_RATIO",
        "PAIRING",
        "VPU_FILL_RATIO_DBL",
        "CPI",
        "MEM2",
        "MEM3",
        "MEM1",
        "WRITE_MISS_RATIO",
        "MEM4",
        "CACHE",
        "MEM_READ",
        "TLB_L2",
        "MEM_WRITE",
        "VPU_PAIRING",
        "VECTOR",
        "MEM5",
        "VECTOR2",
        "TLB_L1",
        "COMPUTE_TO_DATA_RATIO",
    ],
    "westmere": [
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "UOPS",
        "FLOPS_X87",
        "VIEW",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "CACHE",
        "CLOCK",
        "L3CACHE",
        "L2",
        "DATA",
        "MEM_DP",
        "L2CACHE",
    ],
    "broadwell": [
        "PORT_USAGE",
        "ICACHE",
        "FLOPS_AVX",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "CYCLE_STALLS",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "DATA",
        "TMA",
        "L2CACHE",
    ],
    "ivybridge": [
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "FLOPS_AVX",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "CYCLE_STALLS",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "atom": ["TLB", "MEM", "BRANCH", "FLOPS_SP", "FLOPS_X87", "FLOPS_DP", "DATA"],
    "haswell": [
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "L3",
        "TLB_INSTR",
        "CYCLE_STALLS",
        "UOPS",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "core2": [
        "TLB",
        "UOPS_RETIRE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "UOPS",
        "FLOPS_X87",
        "FLOPS_DP",
        "DIVIDE",
        "CACHE",
        "CLOCK",
        "L2",
        "DATA",
        "L2CACHE",
    ],
    "RKL": [
        "FLOPS_AVX",
        "BRANCH",
        "FLOPS_SP",
        "FLOPS_DP",
        "DIVIDE",
        "ENERGY",
        "DATA",
        "TMA",
    ],
    "knl": [
        "ICACHE",
        "HBM_CACHE",
        "FRONTEND_STALLS",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "TLB_INSTR",
        "HBM_OFFCORE",
        "UOPS_STALLS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "CLOCK",
        "ENERGY",
        "L2",
        "DATA",
        "HBM",
        "L2CACHE",
    ],
    "broadwellEP": [
        "NUMA",
        "PORT_USAGE",
        "ICACHE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "QPI",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "HA",
        "L3CACHE",
        "ENERGY",
        "L2",
        "DATA",
        "TMA",
        "MEM_DP",
        "L2CACHE",
    ],
    "ivybridgeEP": [
        "UNCORECLOCK",
        "NUMA",
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "QPI",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "CBOX",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "MEM_DP",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "zen": [
        "NUMA",
        "TLB",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "MEM_SP",
        "CPI",
        "FLOPS_DP",
        "DIVIDE",
        "CACHE",
        "CLOCK",
        "ENERGY",
        "L2",
        "DATA",
        "MEM_DP",
    ],
    "arm8": ["ICACHE", "MEM", "BRANCH", "L2", "DATA"],
    "ICX": [
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "MEM_SP",
        "CYCLE_STALLS",
        "FLOPS_DP",
        "DIVIDE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "MEM_FREERUN",
        "ENERGY",
        "L2",
        "DATA",
        "TMA",
        "UPI",
        "MEM_DP",
        "L2CACHE",
    ],
    "arm64fx": [
        "ICACHE",
        "FLOPS_HP",
        "MEM",
        "BRANCH",
        "TOFU",
        "FLOPS_SP",
        "MEM_SP",
        "FP_PIPE",
        "FLOPS_DP",
        "MEM_HP",
        "L2",
        "PCI",
        "DATA",
        "MEM_DP",
    ],
    "silvermont": [
        "ICACHE",
        "MEM",
        "BRANCH",
        "TLB_INSTR",
        "DIVIDE",
        "TLB_DATA",
        "CLOCK",
        "ENERGY",
        "DATA",
        "L2CACHE",
    ],
    "k8": ["ICACHE", "BRANCH", "CPI", "CACHE", "L2"],
    "broadwellD": [
        "PORT_USAGE",
        "ICACHE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "HA",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "DATA",
        "TMA",
        "MEM_DP",
        "L2CACHE",
    ],
    "k10": [
        "TLB",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FPU_EXCEPTION",
        "FLOPS_SP",
        "NUMA_4_7",
        "FLOPS_X87",
        "CPI",
        "NUMA_0_3",
        "FLOPS_DP",
        "CACHE",
        "L2",
        "L2CACHE",
    ],
    "skylakeX": [
        "UOPS_RETIRE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "UPI",
        "MEM_DP",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "zen3": [
        "NUMA",
        "TLB",
        "ICACHE",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "CPI",
        "MEM2",
        "MEM1",
        "FLOPS_DP",
        "DIVIDE",
        "CACHE",
        "CLOCK",
        "L3CACHE",
        "ENERGY",
        "L2",
        "DATA",
        "L2CACHE",
    ],
    "nehalemEX": [
        "TLB",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "FLOPS_X87",
        "FLOPS_DP",
        "DIVIDE",
        "CACHE",
        "L3CACHE",
        "L2",
        "DATA",
        "SCHEDULER",
        "L2CACHE",
    ],
    "zen2": [
        "NUMA",
        "TLB",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "MEM_SP",
        "CPI",
        "FLOPS_DP",
        "DIVIDE",
        "CACHE",
        "CLOCK",
        "ENERGY",
        "L2",
        "DATA",
        "MEM_DP",
    ],
    "ICL": [
        "ICACHE",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "MEM_SP",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "CLOCK",
        "L3CACHE",
        "ENERGY",
        "L2",
        "DATA",
        "TMA",
        "MEM_DP",
        "L2CACHE",
    ],
    "power8": [
        "NUMA",
        "USEFUL",
        "ICACHE",
        "STALLS2",
        "L1",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "FLOPS_DP2",
        "CPISTACK1",
        "FLOPS_DP",
        "FLOPS_VSU1",
        "STALLS1",
        "FLOPS_VSX",
        "TLB_DATA",
        "FLOPS_4_8",
        "FLOPS_1_2",
        "L2",
        "DATA",
        "FLOPS_FMA",
        "L2CACHE",
        "FLOPS_VSU0",
    ],
    "interlagos": [
        "NUMA",
        "ICACHE",
        "MEM",
        "LINKS",
        "BRANCH",
        "FPU_EXCEPTION",
        "FLOPS_SP",
        "NUMA_4_7",
        "L3",
        "CPI",
        "NUMA_0_3",
        "FLOPS_DP",
        "CACHE",
        "L3CACHE",
        "L2",
        "DATA",
        "L2CACHE",
    ],
    "skylake": [
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "PERF",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "MEM_DP",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "TGL": ["FLOPS_AVX", "BRANCH", "FLOPS_SP", "FLOPS_DP", "DIVIDE", "ENERGY", "DATA"],
    "pentiumm": ["BRANCH", "FLOPS_SP", "L3", "CPI", "FLOPS_DP"],
    "haswellEP": [
        "NUMA",
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "L3",
        "TLB_INSTR",
        "CYCLE_STALLS",
        "QPI",
        "UOPS",
        "DIVIDE",
        "CBOX",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "HA",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "SBOX",
        "TMA",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "CLX": [
        "UOPS_RETIRE",
        "PMM",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "UPI",
        "MEM_DP",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "power9": [
        "USEFUL",
        "ICACHE",
        "MEM",
        "BRANCH",
        "L3",
        "TLB_INSTR",
        "L2STORE",
        "FLOPS_VSX",
        "TLB_DATA",
        "L2LOAD",
        "DATA",
        "FLOPS",
        "FLOPS_FMA",
        "L2CACHE",
    ],
    "sandybridge": [
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "FLOPS_AVX",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "CYCLE_STALLS",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "arm8_tx2": [
        "ICACHE",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "SPEC",
        "FLOPS_DP",
        "TLB_DATA",
        "L2",
        "DATA",
        "L2CACHE",
    ],
    "kabini": [
        "TLB",
        "ICACHE",
        "MEM",
        "BRANCH",
        "FPU_EXCEPTION",
        "FLOPS_SP",
        "NUMA_4_7",
        "CPI",
        "NUMA_0_3",
        "FLOPS_DP",
        "CACHE",
        "L2",
        "DATA",
    ],
    "sandybridgeEP": [
        "NUMA",
        "PORT_USAGE",
        "UOPS_RETIRE",
        "ICACHE",
        "CACHES",
        "FLOPS_AVX",
        "MEM",
        "BRANCH",
        "FLOPS_SP",
        "L3",
        "TLB_INSTR",
        "MEM_SP",
        "CYCLE_STALLS",
        "QPI",
        "UOPS",
        "FLOPS_DP",
        "DIVIDE",
        "TLB_DATA",
        "FALSE_SHARE",
        "CLOCK",
        "CYCLE_ACTIVITY",
        "L3CACHE",
        "RECOVERY",
        "ENERGY",
        "L2",
        "UOPS_EXEC",
        "DATA",
        "TMA",
        "MEM_DP",
        "UOPS_ISSUE",
        "L2CACHE",
    ],
    "goldmont": [
        "ICACHE",
        "BRANCH",
        "TLB_INSTR",
        "DIVIDE",
        "TLB_DATA",
        "CLOCK",
        "ENERGY",
        "DATA",
        "L2CACHE",
    ],
    "nvidia_cc_ge_7": [
        "GPU_BRANCH",
        "GPU_DRAM",
        "GPU_FLOP_DP",
        "GPU_FLOP_FP",
        "GPU_FLOP_HP",
        "GPU_GLD",
        "GPU_INST1",
        "GPU_INST2",
        "GPU_INST3",
        "GPU_INST4",
        "GPU_INST5",
        "GPU_INST6",
        "GPU_INST7",
        "GPU_INST8",
        "GPU_INST9",
        "GPU_INST10",
        "GPU_INST11",
        "GPU_INST12",
        "GPU_L1TEX1",
        "GPU_L1TEX2",
        "GPU_L1TEX3",
        "GPU_L1TEX4",
        "GPU_L1TEX5",
        "GPU_L1TEX6",
        "GPU_L1TEX7",
        "GPU_L1TEX8",
        "GPU_L1TEX9",
        "GPU_L1TEX10",
        "GPU_L1TEX11",
        "GPU_LOCAL1",
        "GPU_LOCAL2",
        "GPU_LOCAL4",
        "GPU_LOCAL5",
        "GPU_LTS1",
        "GPU_LTS2",
        "GPU_LTS3",
        "GPU_MEM_SHARED1",
        "GPU_MEM_SHARED2",
        "GPU_MEM_SHARED3",
        "GPU_PCIE",
        "GPU_SM",
        "GPU_STALL1",
        "GPU_STALL2",
        "GPU_SURFACE1",
        "GPU_SURFACE2",
        "GPU_SYSMEM",
        "GPU_WARPS",
    ],
}
