"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToLambda = void 0;
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
class S3ToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        if (!props.existingBucketObj) {
            [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
                bucketProps: props.bucketProps
            });
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        // Create S3 trigger to invoke lambda function
        this.lambdaFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(bucket, defaults.S3EventSourceProps(props.s3EventSourceProps)));
        this.addCfnNagSuppress();
    }
    addCfnNagSuppress() {
        const root = core_1.Stack.of(this);
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        const notificationsResourceHandler = root.node.tryFindChild(logicalId);
        const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
        const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
        // Extract the CfnFunction from the Function
        const fnResource = notificationsResourceHandler.node.findChild('Resource');
        fnResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W58',
                        reason: `Lambda function has permission to write CloudWatch Logs via AWSLambdaBasicExecutionRole policy attached to the lambda role`
                    }]
            }
        };
        // Extract the CfnPolicy from the iam.Policy
        const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
        policyResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                    }]
            }
        };
    }
}
exports.S3ToLambda = S3ToLambda;
//# sourceMappingURL=data:application/json;base64,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