import re
from os import PathLike
from typing import Dict, List

from sw_ut_report.template_manager import get_local_template
from sw_ut_report.utils import remove_excess_space


def format_scenario_cover(line: str) -> str:
    line = re.sub(r"Covers:\s*", "", line).strip()
    result = re.findall(r"\[([^\]]+)\]", line)
    return ", ".join(result)


def format_txt_file(file_content: str) -> List[Dict]:
    lines = file_content.splitlines()

    if lines and lines[0].strip().lower().startswith("test case:"):
        scenarios = []
        current_scenario = {}
        current_steps = []
        current_step = {}

        for line in lines:
            line_lower = line.strip().lower()

            if line_lower.startswith("test case:"):
                # Save the current scenario if there is one
                if current_scenario.get("test_case"):
                    if current_step:  # Save any pending step
                        current_steps.append(current_step)
                        current_step = {}
                    current_scenario["steps"] = current_steps
                    scenarios.append(current_scenario)  # Save the complete scenario
                    current_scenario = {}  # Reset the scenario
                    current_steps = []  # Reset the steps

                # Start a new structured scenario
                current_scenario["test_case"] = remove_excess_space(
                    re.sub(r"Test case:\s*", "", line).strip()
                )

            elif line_lower.startswith("covers:"):
                current_scenario["covers"] = format_scenario_cover(line)

            elif "given:" in line_lower:
                if current_step:  # If there's an existing step, append it
                    current_steps.append(current_step)
                current_step = {
                    "given": remove_excess_space(line.split(": ", 1)[1].strip())
                }

            elif "when:" in line_lower:
                current_step["when"] = remove_excess_space(
                    line.split(": ", 1)[1].strip()
                )

            elif "then:" in line_lower:
                current_step["then"] = remove_excess_space(
                    line.split(": ", 1)[1].strip()
                )
                current_steps.append(current_step)  # Complete the step and add to steps
                current_step = {}  # Reset for next step

        # Add the last scenario if it's complete (i.e., has a 'test_case')
        if current_scenario.get("test_case"):
            if current_step:  # Add any remaining step
                current_steps.append(current_step)
            current_scenario["steps"] = current_steps
            scenarios.append(current_scenario)

    else:
        # If no "Test case:" is found, the file does not respect any formalism
        # Store the cleaned lines under "raw_lines"
        raw_lines = []
        for line in lines:
            if line.strip():
                cleaned_line = remove_excess_space(line)
                if cleaned_line.lower().startswith("covers:"):
                    cleaned_line = f"**Covers:** {format_scenario_cover(cleaned_line)}"
                raw_lines.append(f"{cleaned_line}\n")
            else:
                raw_lines.append("")
        scenarios = [{"raw_lines": raw_lines}]

    return scenarios


def generate_markdown(parsed_scenarios: List[Dict], output_file: PathLike):
    template = get_local_template("test_report_txt_to_markdown.j2")

    markdown_content = template.render(scenarios=parsed_scenarios)

    # Write the rendered markdown content to the output file
    with open(output_file, "w", encoding="utf-8") as f:
        f.write(markdown_content)


def convert_markdown_from_txt(input_file: PathLike, output_file: PathLike):
    with open(input_file, "r", encoding="utf-8") as f:
        file_content = f.read()

    parsed_scenarios = format_txt_file(file_content)
    generate_markdown(parsed_scenarios, output_file)
