"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const cx_api_1 = require("@aws-cdk/cx-api");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const assets_1 = require("./assets");
const construct_1 = require("./construct");
const context_provider_1 = require("./context-provider");
const asset_parameters_1 = require("./private/asset-parameters");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually a Program instance.
     * @param name The name of the CloudFormation stack. Defaults to "Stack".
     * @param props Stack properties.
     */
    constructor(scope, name, props = {}) {
        // For unit test convenience parents are optional, so bypass the type check when calling the parent.
        super(scope, name);
        /**
         * Options for CloudFormation template (like version, transform, description).
         */
        this.templateOptions = {};
        /**
         * Other stacks this stack depends on
         */
        this._stackDependencies = new Set();
        /**
         * Lists all missing contextual information.
         * This is returned when the stack is synthesized under the 'missing' attribute
         * and allows tooling to obtain the context and re-synthesize.
         */
        this._missingContext = new Array();
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = props.stackName !== undefined ? props.stackName : this.calculateStackName();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
        this.templateFile = `${this.stackName}.template.json`;
        this.templateUrl = lazy_1.Lazy.stringValue({ produce: () => this._templateUrl || '<unresolved>' });
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct) {
        return _lookup(construct);
        function _lookup(c) {
            if (Stack.isStack(c)) {
                return c;
            }
            if (!c.node.scope) {
                throw new Error(`No stack could be identified for the construct at path ${construct.node.path}`);
            }
            return _lookup(c.node.scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report) {
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `createNamingScheme` method.
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack
     */
    addDependency(stack, reason) {
        if (stack === this) {
            return;
        } // Can ignore a dependency on self
        reason = reason || 'dependency added using stack.addDependency()';
        const dep = stack.stackDependencyReasons(this);
        if (dep !== undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`'${stack.node.path}' depends on '${this.node.path}' (${dep.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        this._stackDependencies.add({ stack, reason });
        if (process.env.CDK_DEBUG_DEPS) {
            // tslint:disable-next-line:no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${stack.node.path}" because: ${reason}`);
        }
    }
    /**
     * Return the stacks this stack depends on
     */
    get dependencies() {
        return Array.from(this._stackDependencies.values()).map(d => d.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested() {
        return this.parentStack !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returnst the list of AZs that are availability in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs())
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxapi.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxapi.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    addFileAsset(asset) {
        // assets are always added at the top-level stack
        if (this.parentStack) {
            return this.parentStack.addFileAsset(asset);
        }
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.node.addMetadata(cxapi.ASSET_METADATA, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const s3Url = `https://s3.${this.region}.${this.urlSuffix}/${bucketName}/${objectKey}`;
        return { bucketName, objectKey, s3Url };
    }
    addDockerImageAsset(asset) {
        if (this.parentStack) {
            return this.parentStack.addDockerImageAsset(asset);
        }
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.DockerImageAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                id: this.node.uniqueId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                imageNameParameter: params.imageNameParameter.logicalId,
                repositoryName: asset.repositoryName,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget
            };
            this.node.addMetadata(cxapi.ASSET_METADATA, metadata);
        }
        // Parse repository name and tag from the parameter (<REPO_NAME>@sha256:<TAG>)
        // Example: cdk/cdkexampleimageb2d7f504@sha256:72c4f956379a43b5623d529ddd969f6826dde944d6221f445ff3e7add9875500
        const components = cfn_fn_1.Fn.split('@sha256:', params.imageNameParameter.valueAsString);
        const repositoryName = cfn_fn_1.Fn.select(0, components).toString();
        const imageSha = cfn_fn_1.Fn.select(1, components).toString();
        const imageUri = `${this.account}.dkr.ecr.${this.region}.${this.urlSuffix}/${repositoryName}@sha256:${imageSha}`;
        return {
            imageUri, repositoryName
        };
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Prepare stack
     *
     * Find all CloudFormation references and tell them we're consuming them.
     *
     * Find all dependencies as well and add the appropriate DependsOn fields.
     */
    prepare() {
        const tokens = this.findTokens();
        // References (originating from this stack)
        for (const reference of tokens) {
            // skip if this is not a CfnReference
            if (!cfn_reference_1.CfnReference.isCfnReference(reference)) {
                continue;
            }
            const targetStack = Stack.of(reference.target);
            // skip if this is not a cross-stack reference
            if (targetStack === this) {
                continue;
            }
            // determine which stack should create the cross reference
            const factory = this.determineCrossReferenceFactory(targetStack);
            // if one side is a nested stack (has "parentStack"), we let it create the reference
            // since it has more knowledge about the world.
            const consumedValue = factory.prepareCrossReference(this, reference);
            // if the reference has already been assigned a value for the consuming stack, carry on.
            if (!reference.hasValueForStack(this)) {
                reference.assignValueForStack(this, consumedValue);
            }
        }
        // Resource dependencies
        for (const dependency of this.node.dependencies) {
            const theirStack = Stack.of(dependency.target);
            if (theirStack !== undefined && theirStack !== this && Stack.of(dependency.source) === this) {
                this.addDependency(theirStack, `"${dependency.source.node.path}" depends on "${dependency.target.node.path}"`);
            }
            else {
                for (const target of findResources([dependency.target])) {
                    for (const source of findResources([dependency.source])) {
                        source.addDependsOn(target);
                    }
                }
            }
        }
        if (this.tags.hasTags()) {
            this.node.addMetadata(cxapi.STACK_TAGS_METADATA_KEY, this.tags.renderTags());
        }
        if (this.parentStack) {
            // add the nested stack template as an asset
            const cfn = JSON.stringify(this._toCloudFormation());
            const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
            const parent = this.parentStack;
            const templateLocation = parent.addFileAsset({
                packaging: assets_1.FileAssetPackaging.FILE,
                sourceHash: templateHash,
                fileName: this.templateFile
            });
            // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
            // be resolved as cross-stack references like any other (see "multi" tests).
            this._templateUrl = `https://s3.${parent.region}.${parent.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        }
    }
    synthesize(session) {
        const builder = session.assembly;
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        const text = JSON.stringify(this._toCloudFormation(), undefined, 2);
        fs.writeFileSync(outPath, text);
        // if this is a nested stack, do not emit it as a cloud assembly artifact (it will be registered as an s3 asset instead)
        if (this.nested) {
            return;
        }
        const deps = this.dependencies.map(s => s.stackName);
        const meta = this.collectMetadata();
        const properties = {
            templateFile: this.templateFile
        };
        // add an artifact that represents this stack
        builder.addArtifact(this.stackName, {
            type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
            environment: this.environment,
            properties,
            dependencies: deps.length > 0 ? deps : undefined,
            metadata: Object.keys(meta).length > 0 ? meta : undefined,
        });
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        if (this.templateOptions.transform) {
            // tslint:disable-next-line: max-line-length
            this.node.addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `templateOptions.transforms`.');
            if (!this.templateOptions.transforms) {
                this.templateOptions.transforms = [];
            }
            if (this.templateOptions.transforms.indexOf(this.templateOptions.transform) === -1) {
                this.templateOptions.transforms.unshift(this.templateOptions.transform);
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: extractSingleValue(this.templateOptions.transforms),
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * Exports a resolvable value for use in another stack.
     *
     * @returns a token that can be used to reference the value from the producing stack.
     */
    prepareCrossReference(sourceStack, reference) {
        const targetStack = Stack.of(reference.target);
        // Ensure a singleton "Exports" scoping Construct
        // This mostly exists to trigger LogicalID munging, which would be
        // disabled if we parented constructs directly under Stack.
        // Also it nicely prevents likely construct name clashes
        const exportsScope = targetStack.getCreateExportsScope();
        // Ensure a singleton CfnOutput for this value
        const resolved = targetStack.resolve(reference);
        const id = 'Output' + JSON.stringify(resolved);
        const exportName = targetStack.generateExportName(exportsScope, id);
        const output = exportsScope.node.tryFindChild(id);
        if (!output) {
            new cfn_output_1.CfnOutput(exportsScope, id, { value: token_1.Token.asString(reference), exportName });
        }
        // add a dependency on the producing stack - it has to be deployed before this stack can consume the exported value
        // if the producing stack is a nested stack (i.e. has a parent), the dependency is taken on the parent.
        const producerDependency = targetStack.parentStack ? targetStack.parentStack : targetStack;
        const consumerDependency = sourceStack.parentStack ? sourceStack.parentStack : sourceStack;
        consumerDependency.addDependency(producerDependency, `${sourceStack.node.path} -> ${reference.target.node.path}.${reference.displayName}`);
        // We want to return an actual FnImportValue Token here, but Fn.importValue() returns a 'string',
        // so construct one in-place.
        return new intrinsic_1.Intrinsic({ 'Fn::ImportValue': exportName });
    }
    getCreateExportsScope() {
        const exportsName = 'Exports';
        let stackExports = this.node.tryFindChild(exportsName);
        if (stackExports === undefined) {
            stackExports = new construct_1.Construct(this, exportsName);
        }
        return stackExports;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region but
        // they do not need to be scoped, the only situation in which
        // export/fn::importvalue would work if { Ref: "AWS::AccountId" } is the
        // same for provider and consumer anyway.
        const account = env.account || cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = env.region || cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account, region,
            environment: cx_api_1.EnvironmentUtils.format(envAccount, envRegion)
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of this._stackDependencies) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [dep.reason].concat(ret);
            }
        }
        return undefined;
    }
    collectMetadata() {
        const output = {};
        const stack = this;
        visit(this);
        return output;
        function visit(node) {
            // break off if we reached a node that is not a child of this stack
            const parent = findParentStack(node);
            if (parent !== stack) {
                return;
            }
            if (node.node.metadata.length > 0) {
                // Make the path absolute
                output[construct_1.ConstructNode.PATH_SEP + node.node.path] = node.node.metadata.map(md => stack.resolve(md));
            }
            for (const child of node.node.children) {
                visit(child);
            }
        }
        function findParentStack(node) {
            if (node instanceof Stack && node.parentStack === undefined) {
                return node;
            }
            if (!node.node.scope) {
                return undefined;
            }
            return findParentStack(node.node.scope);
        }
    }
    /**
     * Calculcate the stack name based on the construct path
     */
    calculateStackName() {
        // In tests, it's possible for this stack to be the root object, in which case
        // we need to use it as part of the root path.
        const rootPath = this.node.scope !== undefined ? this.node.scopes.slice(1) : [this];
        const ids = rootPath.map(c => c.node.id);
        // Special case, if rootPath is length 1 then just use ID (backwards compatibility)
        // otherwise use a unique stack name (including hash). This logic is already
        // in makeUniqueId, *however* makeUniqueId will also strip dashes from the name,
        // which *are* allowed and also used, so we short-circuit it.
        if (ids.length === 1) {
            // Could be empty in a unit test, so just pretend it's named "Stack" then
            return ids[0] || 'Stack';
        }
        return uniqueid_1.makeUniqueId(ids);
    }
    generateExportName(stackExports, id) {
        const stack = Stack.of(stackExports);
        const components = [...stackExports.node.scopes.slice(2).map(c => c.node.id), id];
        const prefix = stack.stackName ? stack.stackName + ':' : '';
        const exportName = prefix + uniqueid_1.makeUniqueId(components);
        return exportName;
    }
    get assetParameters() {
        if (!this._assetParameters) {
            this._assetParameters = new construct_1.Construct(this, 'AssetParameters');
        }
        return this._assetParameters;
    }
    determineCrossReferenceFactory(target) {
        // unsupported: stacks from different apps
        if (target.node.root !== this.node.root) {
            throw new Error(`Cannot reference across apps. ` +
                `Consuming and producing stacks must be defined within the same CDK app.`);
        }
        // unsupported: stacks are not in the same environment
        if (target.environment !== this.environment) {
            throw new Error(`Stack "${this.node.path}" cannot consume a cross reference from stack "${target.node.path}". ` +
                `Cross stack references are only supported for stacks deployed to the same environment or between nested stacks and their parent stack`);
        }
        // if one of the stacks is a nested stack, go ahead and give it the right to make the cross reference
        if (target.nested) {
            return target;
        }
        if (this.nested) {
            return this;
        }
        // both stacks are top-level (non-nested), the taret (producing stack) gets to make the reference
        return target;
    }
    /**
     * Returns all the tokens used within the scope of the current stack.
     */
    findTokens() {
        const tokens = new Array();
        for (const element of cfnElements(this)) {
            try {
                tokens.push(...resolve_1.findTokens(element, () => element._toCloudFormation()));
            }
            catch (e) {
                // Note: it might be that the properties of the CFN object aren't valid.
                // This will usually be preventatively caught in a construct's validate()
                // and turned into a nicely descriptive error, but we're running prepare()
                // before validate(). Swallow errors that occur because the CFN layer
                // doesn't validate completely.
                //
                // This does make the assumption that the error will not be rectified,
                // but the error will be thrown later on anyway. If the error doesn't
                // get thrown down the line, we may miss references.
                if (e.type === 'CfnSynthesisError') {
                    continue;
                }
                throw e;
            }
        }
        return tokens;
    }
}
exports.Stack = Stack;
function merge(template, part) {
    for (const section of Object.keys(part)) {
        const src = part[section];
        // create top-level section if it doesn't exist
        let dest = template[section];
        if (!dest) {
            template[section] = dest = src;
        }
        else {
            // add all entities from source section to destination section
            for (const id of Object.keys(src)) {
                if (id in dest) {
                    throw new Error(`section '${section}' already contains '${id}'`);
                }
                dest[id] = src[id];
            }
        }
    }
}
/**
 * Collect all CfnElements from a Stack
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
// These imports have to be at the end to prevent circular imports
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_output_1 = require("./cfn-output");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const intrinsic_1 = require("./private/intrinsic");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
/**
 * Find all resources in a set of constructs
 */
function findResources(roots) {
    const ret = new Array();
    for (const root of roots) {
        ret.push(...root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource));
    }
    return ret;
}
function extractSingleValue(array) {
    if (array && array.length === 1) {
        return array[0];
    }
    return array;
}
//# sourceMappingURL=data:application/json;base64,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