# -*- coding: utf-8 -*-
# Copyright (C) 2018-2023, earthobservations developers.
# Distributed under the MIT License. See LICENSE for more info.
import datetime as dt
import logging
from enum import Enum
from io import StringIO
from typing import Dict, Iterator, List, Optional, Union
from urllib.error import HTTPError
from urllib.parse import urljoin

import numpy as np
import pandas as pd
import polars as pl
from backports.datetime_fromisoformat import MonkeyPatch

from wetterdienst import Kind, Parameter, Period, Provider, Settings
from wetterdienst.core.timeseries.request import TimeseriesRequest
from wetterdienst.core.timeseries.result import StationsResult, ValuesResult
from wetterdienst.core.timeseries.values import TimeseriesValues
from wetterdienst.exceptions import InvalidParameter
from wetterdienst.metadata.columns import Columns
from wetterdienst.metadata.datarange import DataRange
from wetterdienst.metadata.period import PeriodType
from wetterdienst.metadata.resolution import Resolution, ResolutionType
from wetterdienst.metadata.timezone import Timezone
from wetterdienst.metadata.unit import OriginUnit, SIUnit, UnitEnum
from wetterdienst.provider.dwd.mosmix.access import KMLReader
from wetterdienst.util.cache import CacheExpiry
from wetterdienst.util.enumeration import parse_enumeration_from_template
from wetterdienst.util.network import download_file, list_remote_files_fsspec
from wetterdienst.util.parameter import DatasetTreeCore
from wetterdienst.util.python import to_list

MonkeyPatch.patch_fromisoformat()
log = logging.getLogger(__name__)


class DwdDmoPeriod(Enum):
    FUTURE = Period.FUTURE.value


class DwdDmoDataset(Enum):
    ICON_EU = "icon-eu"
    ICON = "icon"


class DwdDmoType(Enum):
    ICON_EU = "icon-eu"
    ICON = "icon"


class DwdForecastDate(Enum):
    """
    Enumeration for pointing to different mosmix dates.
    """

    LATEST = "latest"


class DwdDmoParameter(DatasetTreeCore):
    # https://opendata.dwd.de/weather/lib/MetElementDefinition.xml
    class ICON_EU(DatasetTreeCore):
        class ICON_EU(Enum):
            CLOUD_COVER_ABOVE_7_KM = "nh"
            CLOUD_COVER_BELOW_500_FT = "n05"
            CLOUD_COVER_BELOW_1000_FT = "nl"
            CLOUD_COVER_BETWEEN_2_TO_7_KM = "nm"
            CLOUD_COVER_EFFECTIVE = "neff"
            CLOUD_COVER_TOTAL = "n"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = "rr1c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = "rr3c"
            PRESSURE_AIR_SITE_REDUCED = "pppp"
            PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H = "fxh25"
            PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H = "fxh40"
            PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H = "fxh55"
            PROBABILITY_FOG_LAST_1H = "wwm"
            PROBABILITY_FOG_LAST_6H = "wwm6"
            PROBABILITY_FOG_LAST_12H = "wwmh"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H = "rh00"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H = "r602"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H = "rh02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H = "rd02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H = "rh10"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H = "r650"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H = "rh50"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H = "rd50"
            RADIATION_GLOBAL = "rad1h"
            SUNSHINE_DURATION = "sund1"
            TEMPERATURE_AIR_MAX_200 = "tx"
            TEMPERATURE_AIR_MEAN_005 = "t5cm"
            TEMPERATURE_AIR_MEAN_200 = "ttt"
            TEMPERATURE_AIR_MIN_200 = "tn"
            TEMPERATURE_DEW_POINT_MEAN_200 = "td"
            VISIBILITY_RANGE = "vv"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = "rrs1c"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = "rrs3c"
            WEATHER_LAST_6H = "w1w2"
            WEATHER_SIGNIFICANT = "ww"
            WIND_DIRECTION = "dd"
            WIND_GUST_MAX_LAST_1H = "fx1"
            WIND_GUST_MAX_LAST_3H = "fx3"
            WIND_GUST_MAX_LAST_12H = "fxh"
            WIND_SPEED = "ff"

        CLOUD_COVER_ABOVE_7_KM = ICON_EU.CLOUD_COVER_ABOVE_7_KM
        CLOUD_COVER_BELOW_500_FT = ICON_EU.CLOUD_COVER_BELOW_500_FT
        CLOUD_COVER_BELOW_1000_FT = ICON_EU.CLOUD_COVER_BELOW_1000_FT
        CLOUD_COVER_BETWEEN_2_TO_7_KM = ICON_EU.CLOUD_COVER_BETWEEN_2_TO_7_KM
        CLOUD_COVER_EFFECTIVE = ICON_EU.CLOUD_COVER_EFFECTIVE
        CLOUD_COVER_TOTAL = ICON_EU.CLOUD_COVER_TOTAL
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = ICON_EU.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = ICON_EU.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H
        PRESSURE_AIR_SITE_REDUCED = ICON_EU.PRESSURE_AIR_SITE_REDUCED
        PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H = ICON_EU.PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H = ICON_EU.PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H = ICON_EU.PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H
        PROBABILITY_FOG_LAST_1H = ICON_EU.PROBABILITY_FOG_LAST_1H
        PROBABILITY_FOG_LAST_6H = ICON_EU.PROBABILITY_FOG_LAST_6H
        PROBABILITY_FOG_LAST_12H = ICON_EU.PROBABILITY_FOG_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H = (
            ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H
        )
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H = ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H = (
            ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H
        )
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H = (
            ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H
        )
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H = (
            ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H
        )
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H = ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H = (
            ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H
        )
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H = (
            ICON_EU.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H
        )
        RADIATION_GLOBAL = ICON_EU.RADIATION_GLOBAL
        SUNSHINE_DURATION = ICON_EU.SUNSHINE_DURATION
        TEMPERATURE_AIR_MAX_200 = ICON_EU.TEMPERATURE_AIR_MAX_200
        TEMPERATURE_AIR_MEAN_005 = ICON_EU.TEMPERATURE_AIR_MEAN_005
        TEMPERATURE_AIR_MEAN_200 = ICON_EU.TEMPERATURE_AIR_MEAN_200
        TEMPERATURE_AIR_MIN_200 = ICON_EU.TEMPERATURE_AIR_MIN_200
        TEMPERATURE_DEW_POINT_MEAN_200 = ICON_EU.TEMPERATURE_DEW_POINT_MEAN_200
        VISIBILITY_RANGE = ICON_EU.VISIBILITY_RANGE
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = ICON_EU.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = ICON_EU.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H
        WEATHER_LAST_6H = ICON_EU.WEATHER_LAST_6H
        WEATHER_SIGNIFICANT = ICON_EU.WEATHER_SIGNIFICANT
        WIND_DIRECTION = ICON_EU.WIND_DIRECTION
        WIND_GUST_MAX_LAST_1H = ICON_EU.WIND_GUST_MAX_LAST_1H
        WIND_GUST_MAX_LAST_3H = ICON_EU.WIND_GUST_MAX_LAST_3H
        WIND_GUST_MAX_LAST_12H = ICON_EU.WIND_GUST_MAX_LAST_12H
        WIND_SPEED = ICON_EU.WIND_SPEED

    class ICON(DatasetTreeCore):
        class ICON(Enum):
            CLOUD_BASE_CONVECTIVE = "h_bsc"
            CLOUD_COVER_ABOVE_7_KM = "nh"
            CLOUD_COVER_BELOW_500_FT = "n05"
            CLOUD_COVER_BELOW_1000_FT = "nl"
            CLOUD_COVER_BELOW_7_KM = "nlm"
            CLOUD_COVER_BETWEEN_2_TO_7_KM = "nm"
            CLOUD_COVER_EFFECTIVE = "neff"
            CLOUD_COVER_TOTAL = "n"
            ERROR_ABSOLUTE_PRESSURE_AIR_SITE = "e_ppp"
            ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_200 = "e_ttt"
            ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_200 = "e_td"
            ERROR_ABSOLUTE_WIND_DIRECTION = "e_dd"
            ERROR_ABSOLUTE_WIND_SPEED = "e_ff"
            EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = "pevap"
            PRECIPITATION_DURATION = "drr1"
            PRECIPITATION_HEIGHT_LAST_1H = "rr1"
            PRECIPITATION_HEIGHT_LAST_3H = "rr3"
            PRECIPITATION_HEIGHT_LAST_6H = "rr6"
            PRECIPITATION_HEIGHT_LAST_12H = "rrh"
            PRECIPITATION_HEIGHT_LAST_24H = "rrd"
            PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = "rrl1c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = "rr1c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = "rr3c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = "rr6c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = "rrhc"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = "rrdc"
            PRESSURE_AIR_SITE_REDUCED = "pppp"
            PROBABILITY_DRIZZLE_LAST_1H = "wwz"
            PROBABILITY_DRIZZLE_LAST_6H = "wwz6"
            PROBABILITY_DRIZZLE_LAST_12H = "wwzh"
            PROBABILITY_FOG_LAST_1H = "wwm"
            PROBABILITY_FOG_LAST_6H = "wwm6"
            PROBABILITY_FOG_LAST_12H = "wwmh"
            PROBABILITY_FOG_LAST_24H = "wwmd"
            PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = "wwf"
            PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = "wwf6"
            PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = "wwfh"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H = "rh00"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H = "r602"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H = "rh02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H = "rd02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H = "rh10"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H = "r650"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H = "rh50"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H = "rd50"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_6H = "r600"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_24H = "rd00"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1_MM_LAST_1H = "r101"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_1H = "r102"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3_MM_LAST_1H = "r103"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5_MM_LAST_1H = "r105"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7_MM_LAST_1H = "r107"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_1H = "r110"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_6H = "r610"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_24H = "rd10"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_2_0_MM_LAST_1H = "r120"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_3_0_MM_LAST_1H = "r130"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_1H = "r150"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_10_0_MM_LAST_1H = "rr1o1"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_15_0_MM_LAST_1H = "rr1w1"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_25_0_MM_LAST_1H = "rr1u1"
            PROBABILITY_PRECIPITATION_LAST_1H = "wwp"
            PROBABILITY_PRECIPITATION_LAST_6H = "wwp6"
            PROBABILITY_PRECIPITATION_LAST_12H = "wwph"
            PROBABILITY_PRECIPITATION_LAST_24H = "wwpd"
            PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = "wwl"
            PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = "wwl6"
            PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = "wwlh"
            PROBABILITY_PRECIPITATION_SOLID_LAST_1H = "wws"
            PROBABILITY_PRECIPITATION_SOLID_LAST_6H = "wws6"
            PROBABILITY_PRECIPITATION_SOLID_LAST_12H = "wwsh"
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = "wwd"
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = "wwd6"
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = "wwdh"
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = "wwc"
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = "wwc6"
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = "wwch"
            PROBABILITY_RADIATION_GLOBAL_LAST_1H = "rrad1"
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0_PCT_LAST_24H = "psd00"
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30_PCT_LAST_24H = "psd30"
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60_PCT_LAST_24H = "psd60"
            PROBABILITY_THUNDER_LAST_1H = "wwt"
            PROBABILITY_THUNDER_LAST_6H = "wwt6"
            PROBABILITY_THUNDER_LAST_12H = "wwth"
            PROBABILITY_THUNDER_LAST_24H = "wwtd"
            PROBABILITY_VISIBILITY_BELOW_1000_M = "vv10"
            PROBABILITY_WIND_GUST_GE_25_KN_LAST_6H = "fx625"
            PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H = "fxh25"
            PROBABILITY_WIND_GUST_GE_40_KN_LAST_6H = "fx640"
            PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H = "fxh40"
            PROBABILITY_WIND_GUST_GE_55_KN_LAST_6H = "fx655"
            PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H = "fxh55"
            RADIATION_GLOBAL = "rad1h"
            RADIATION_GLOBAL_LAST_3H = "rads3"
            RADIATION_SKY_LONG_WAVE_LAST_3H = "radl3"
            SUNSHINE_DURATION = "sund1"
            SUNSHINE_DURATION_LAST_3H = "sund3"
            SUNSHINE_DURATION_RELATIVE_LAST_24H = "rsund"
            SUNSHINE_DURATION_YESTERDAY = "sund"
            TEMPERATURE_AIR_MAX_200 = "tx"
            TEMPERATURE_AIR_MEAN_005 = "t5cm"
            TEMPERATURE_AIR_MEAN_200 = "ttt"
            TEMPERATURE_AIR_MEAN_200_LAST_24H = "tm"
            TEMPERATURE_AIR_MIN_005_LAST_12H = "tg"
            TEMPERATURE_AIR_MIN_200 = "tn"
            TEMPERATURE_DEW_POINT_MEAN_200 = "td"
            VISIBILITY_RANGE = "vv"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = "rrs1c"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = "rrs3c"
            WEATHER_LAST_6H = "w1w2"
            WEATHER_SIGNIFICANT = "ww"
            WEATHER_SIGNIFICANT_LAST_3H = "ww3"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = "wpc11"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = "wpc31"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = "wpc61"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = "wpch1"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = "wpcd1"
            WIND_DIRECTION = "dd"
            WIND_GUST_MAX_LAST_1H = "fx1"
            WIND_GUST_MAX_LAST_3H = "fx3"
            WIND_GUST_MAX_LAST_12H = "fxh"
            WIND_SPEED = "ff"

        CLOUD_BASE_CONVECTIVE = ICON.CLOUD_BASE_CONVECTIVE
        CLOUD_COVER_ABOVE_7_KM = ICON.CLOUD_COVER_ABOVE_7_KM
        CLOUD_COVER_BELOW_500_FT = ICON.CLOUD_COVER_BELOW_500_FT
        CLOUD_COVER_BELOW_1000_FT = ICON.CLOUD_COVER_BELOW_1000_FT
        CLOUD_COVER_BELOW_7_KM = ICON.CLOUD_COVER_BELOW_7_KM
        CLOUD_COVER_BETWEEN_2_TO_7_KM = ICON.CLOUD_COVER_BETWEEN_2_TO_7_KM
        CLOUD_COVER_EFFECTIVE = ICON.CLOUD_COVER_EFFECTIVE
        CLOUD_COVER_TOTAL = ICON.CLOUD_COVER_TOTAL
        ERROR_ABSOLUTE_PRESSURE_AIR_SITE = ICON.ERROR_ABSOLUTE_PRESSURE_AIR_SITE
        ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_200 = ICON.ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_200
        ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_200 = ICON.ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_200
        ERROR_ABSOLUTE_WIND_DIRECTION = ICON.ERROR_ABSOLUTE_WIND_DIRECTION
        ERROR_ABSOLUTE_WIND_SPEED = ICON.ERROR_ABSOLUTE_WIND_SPEED
        EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = ICON.EVAPOTRANSPIRATION_POTENTIAL_LAST_24H
        PRECIPITATION_DURATION = ICON.PRECIPITATION_DURATION
        PRECIPITATION_HEIGHT_LAST_1H = ICON.PRECIPITATION_HEIGHT_LAST_1H
        PRECIPITATION_HEIGHT_LAST_3H = ICON.PRECIPITATION_HEIGHT_LAST_3H
        PRECIPITATION_HEIGHT_LAST_6H = ICON.PRECIPITATION_HEIGHT_LAST_6H
        PRECIPITATION_HEIGHT_LAST_12H = ICON.PRECIPITATION_HEIGHT_LAST_12H
        PRECIPITATION_HEIGHT_LAST_24H = ICON.PRECIPITATION_HEIGHT_LAST_24H
        PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = (
            ICON.PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H
        )
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = ICON.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = ICON.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = ICON.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = ICON.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = ICON.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H
        PRESSURE_AIR_SITE_REDUCED = ICON.PRESSURE_AIR_SITE_REDUCED
        PROBABILITY_DRIZZLE_LAST_1H = ICON.PROBABILITY_DRIZZLE_LAST_1H
        PROBABILITY_DRIZZLE_LAST_6H = ICON.PROBABILITY_DRIZZLE_LAST_6H
        PROBABILITY_DRIZZLE_LAST_12H = ICON.PROBABILITY_DRIZZLE_LAST_12H
        PROBABILITY_FOG_LAST_1H = ICON.PROBABILITY_FOG_LAST_1H
        PROBABILITY_FOG_LAST_6H = ICON.PROBABILITY_FOG_LAST_6H
        PROBABILITY_FOG_LAST_12H = ICON.PROBABILITY_FOG_LAST_12H
        PROBABILITY_FOG_LAST_24H = ICON.PROBABILITY_FOG_LAST_24H
        PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = ICON.PROBABILITY_PRECIPITATION_FREEZING_LAST_1H
        PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = ICON.PROBABILITY_PRECIPITATION_FREEZING_LAST_6H
        PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = ICON.PROBABILITY_PRECIPITATION_FREEZING_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_6H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_24H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_6H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_24H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_2_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_2_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_3_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_3_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_10_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_10_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_15_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_15_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_25_0_MM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_HEIGHT_GT_25_0_MM_LAST_1H
        PROBABILITY_PRECIPITATION_LAST_1H = ICON.PROBABILITY_PRECIPITATION_LAST_1H
        PROBABILITY_PRECIPITATION_LAST_6H = ICON.PROBABILITY_PRECIPITATION_LAST_6H
        PROBABILITY_PRECIPITATION_LAST_12H = ICON.PROBABILITY_PRECIPITATION_LAST_12H
        PROBABILITY_PRECIPITATION_LAST_24H = ICON.PROBABILITY_PRECIPITATION_LAST_24H
        PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = ICON.PROBABILITY_PRECIPITATION_LIQUID_LAST_1H
        PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = ICON.PROBABILITY_PRECIPITATION_LIQUID_LAST_6H
        PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = ICON.PROBABILITY_PRECIPITATION_LIQUID_LAST_12H
        PROBABILITY_PRECIPITATION_SOLID_LAST_1H = ICON.PROBABILITY_PRECIPITATION_SOLID_LAST_1H
        PROBABILITY_PRECIPITATION_SOLID_LAST_6H = ICON.PROBABILITY_PRECIPITATION_SOLID_LAST_6H
        PROBABILITY_PRECIPITATION_SOLID_LAST_12H = ICON.PROBABILITY_PRECIPITATION_SOLID_LAST_12H
        PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = ICON.PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H
        PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = ICON.PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H
        PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = ICON.PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H
        PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = ICON.PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H
        PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = ICON.PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H
        PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = ICON.PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H
        PROBABILITY_RADIATION_GLOBAL_LAST_1H = ICON.PROBABILITY_RADIATION_GLOBAL_LAST_1H
        PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0_PCT_LAST_24H = (
            ICON.PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0_PCT_LAST_24H
        )
        PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30_PCT_LAST_24H = (
            ICON.PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30_PCT_LAST_24H
        )
        PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60_PCT_LAST_24H = (
            ICON.PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60_PCT_LAST_24H
        )
        PROBABILITY_THUNDER_LAST_1H = ICON.PROBABILITY_THUNDER_LAST_1H
        PROBABILITY_THUNDER_LAST_6H = ICON.PROBABILITY_THUNDER_LAST_6H
        PROBABILITY_THUNDER_LAST_12H = ICON.PROBABILITY_THUNDER_LAST_12H
        PROBABILITY_THUNDER_LAST_24H = ICON.PROBABILITY_THUNDER_LAST_24H
        PROBABILITY_VISIBILITY_BELOW_1000_M = ICON.PROBABILITY_VISIBILITY_BELOW_1000_M
        PROBABILITY_WIND_GUST_GE_25_KN_LAST_6H = ICON.PROBABILITY_WIND_GUST_GE_25_KN_LAST_6H
        PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H = ICON.PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_40_KN_LAST_6H = ICON.PROBABILITY_WIND_GUST_GE_40_KN_LAST_6H
        PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H = ICON.PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_55_KN_LAST_6H = ICON.PROBABILITY_WIND_GUST_GE_55_KN_LAST_6H
        PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H = ICON.PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H
        RADIATION_GLOBAL = ICON.RADIATION_GLOBAL
        RADIATION_GLOBAL_LAST_3H = ICON.RADIATION_GLOBAL_LAST_3H
        RADIATION_SKY_LONG_WAVE_LAST_3H = ICON.RADIATION_SKY_LONG_WAVE_LAST_3H
        SUNSHINE_DURATION = ICON.SUNSHINE_DURATION
        SUNSHINE_DURATION_LAST_3H = ICON.SUNSHINE_DURATION_LAST_3H
        SUNSHINE_DURATION_RELATIVE_LAST_24H = ICON.SUNSHINE_DURATION_RELATIVE_LAST_24H
        SUNSHINE_DURATION_YESTERDAY = ICON.SUNSHINE_DURATION_YESTERDAY
        TEMPERATURE_AIR_MAX_200 = ICON.TEMPERATURE_AIR_MAX_200
        TEMPERATURE_AIR_MEAN_005 = ICON.TEMPERATURE_AIR_MEAN_005
        TEMPERATURE_AIR_MEAN_200 = ICON.TEMPERATURE_AIR_MEAN_200
        TEMPERATURE_AIR_MEAN_200_LAST_24H = ICON.TEMPERATURE_AIR_MEAN_200_LAST_24H
        TEMPERATURE_AIR_MIN_005_LAST_12H = ICON.TEMPERATURE_AIR_MIN_005_LAST_12H
        TEMPERATURE_AIR_MIN_200 = ICON.TEMPERATURE_AIR_MIN_200
        TEMPERATURE_DEW_POINT_MEAN_200 = ICON.TEMPERATURE_DEW_POINT_MEAN_200
        VISIBILITY_RANGE = ICON.VISIBILITY_RANGE
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = ICON.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = ICON.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H
        WEATHER_LAST_6H = ICON.WEATHER_LAST_6H
        WEATHER_SIGNIFICANT = ICON.WEATHER_SIGNIFICANT
        WEATHER_SIGNIFICANT_LAST_3H = ICON.WEATHER_SIGNIFICANT_LAST_3H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = ICON.WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = ICON.WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = ICON.WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = ICON.WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = ICON.WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H
        WIND_DIRECTION = ICON.WIND_DIRECTION
        WIND_GUST_MAX_LAST_1H = ICON.WIND_GUST_MAX_LAST_1H
        WIND_GUST_MAX_LAST_3H = ICON.WIND_GUST_MAX_LAST_3H
        WIND_GUST_MAX_LAST_12H = ICON.WIND_GUST_MAX_LAST_12H
        WIND_SPEED = ICON.WIND_SPEED


class DwdDmoUnit(DatasetTreeCore):
    class ICON_EU(DatasetTreeCore):
        class ICON_EU(UnitEnum):
            TEMPERATURE_AIR_MEAN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_DEW_POINT_MEAN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MAX_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MIN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            WIND_DIRECTION = (
                OriginUnit.WIND_DIRECTION.value,
                SIUnit.WIND_DIRECTION.value,
            )
            WIND_SPEED = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_1H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_3H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_12H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WEATHER_SIGNIFICANT = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_LAST_6H = OriginUnit.DIMENSIONLESS.value, SIUnit.DIMENSIONLESS.value
            CLOUD_COVER_TOTAL = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_EFFECTIVE = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_500_FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_1000_FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BETWEEN_2_TO_7_KM = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            CLOUD_COVER_ABOVE_7_KM = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PRESSURE_AIR_SITE_REDUCED = OriginUnit.PASCAL.value, SIUnit.PASCAL.value
            TEMPERATURE_AIR_MEAN_005 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            RADIATION_GLOBAL = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            VISIBILITY_RANGE = OriginUnit.METER.value, SIUnit.METER.value
            SUNSHINE_DURATION = OriginUnit.SECOND.value, SIUnit.SECOND.value
            PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_FOG_LAST_1H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_6H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_12H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )

    class ICON(DatasetTreeCore):
        class ICON(UnitEnum):
            # https://opendata.dwd.de/weather/lib/MetElementDefinition.xml
            TEMPERATURE_AIR_MEAN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_DEW_POINT_MEAN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MAX_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MIN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            WIND_DIRECTION = (
                OriginUnit.WIND_DIRECTION.value,
                SIUnit.WIND_DIRECTION.value,
            )
            WIND_SPEED = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_1H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_3H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_12H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WEATHER_SIGNIFICANT = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_LAST_6H = OriginUnit.DIMENSIONLESS.value, SIUnit.DIMENSIONLESS.value
            CLOUD_COVER_TOTAL = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_EFFECTIVE = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_500_FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_1000_FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BETWEEN_2_TO_7_KM = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            CLOUD_COVER_ABOVE_7_KM = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PRESSURE_AIR_SITE_REDUCED = OriginUnit.PASCAL.value, SIUnit.PASCAL.value
            TEMPERATURE_AIR_MEAN_005 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            RADIATION_GLOBAL_LAST_3H = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            RADIATION_GLOBAL = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            RADIATION_SKY_LONG_WAVE_LAST_3H = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            VISIBILITY_RANGE = OriginUnit.METER.value, SIUnit.METER.value
            SUNSHINE_DURATION = OriginUnit.SECOND.value, SIUnit.SECOND.value
            PROBABILITY_WIND_GUST_GE_25_KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_40_KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_55_KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_FOG_LAST_1H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_6H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_12H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            TEMPERATURE_AIR_MIN_005_LAST_12H = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MEAN_200_LAST_24H = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            PRECIPITATION_DURATION = OriginUnit.SECOND.value, SIUnit.SECOND.value
            PROBABILITY_DRIZZLE_LAST_1H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_THUNDER_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_SOLID_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_VISIBILITY_BELOW_1000_M = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            ERROR_ABSOLUTE_WIND_SPEED = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            ERROR_ABSOLUTE_WIND_DIRECTION = (
                OriginUnit.WIND_DIRECTION.value,
                SIUnit.WIND_DIRECTION.value,
            )
            ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_200 = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            PRECIPITATION_HEIGHT_LAST_6H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_2_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            SUNSHINE_DURATION_YESTERDAY = OriginUnit.SECOND.value, SIUnit.SECOND.value
            SUNSHINE_DURATION_RELATIVE_LAST_24H = (
                OriginUnit.DIMENSIONLESS.value,
                SIUnit.DIMENSIONLESS.value,
            )
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0_PCT_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30_PCT_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60_PCT_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_RADIATION_GLOBAL_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_3_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_10_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_15_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_25_0_MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_THUNDER_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_SOLID_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_DRIZZLE_LAST_6H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_24H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_WIND_GUST_GE_25_KN_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_40_KN_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_55_KN_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_THUNDER_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_SOLID_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_DRIZZLE_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PRECIPITATION_HEIGHT_LAST_12H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WEATHER_SIGNIFICANT_LAST_3H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0_MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1_0_MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PRECIPITATION_HEIGHT_LAST_24H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            CLOUD_COVER_BELOW_7_KM = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            CLOUD_BASE_CONVECTIVE = OriginUnit.METER.value, SIUnit.METER.value
            PROBABILITY_THUNDER_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            ERROR_ABSOLUTE_PRESSURE_AIR_SITE = (
                OriginUnit.PASCAL.value,
                SIUnit.PASCAL.value,
            )
            SUNSHINE_DURATION_LAST_3H = OriginUnit.SECOND.value, SIUnit.SECOND.value
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )


class DwdDmoStationGroup(Enum):
    SINGLE_STATIONS = "single_stations"
    ALL_STATIONS = "all_stations"


class DwdDmoValues(TimeseriesValues):
    """
    Fetch DWD DMO data.

    Parameters
    ----------
    station_id : List
        - If None, data for all stations_result is returned.
        - If not None, station_ids are a list of station ids for which data is desired.

    parameter: List
        - If None, data for all parameters is returned.
        - If not None, list of parameters, per DMO definition, see
          https://www.dwd.de/DE/leistungen/opendata/help/schluessel_datenformate/kml/mosmix_elemente_pdf.pdf?__blob=publicationFile&v=2
    """  # noqa:B950,E501

    _tz = Timezone.GERMANY
    _data_tz = Timezone.UTC

    def _create_humanized_parameters_mapping(self) -> Dict[str, str]:
        """
        Method for creation of parameter name mappings based on
        self._parameter_base

        :return:
        """
        return {
            parameter.value: parameter.name.lower()
            for parameter in self.sr.stations._parameter_base[self.sr.stations.dmo_type.name][
                self.sr.stations.dmo_type.name
            ]
        }

    def __init__(self, stations_result: StationsResult) -> None:
        """

        :param stations_result:
        """
        super(DwdDmoValues, self).__init__(stations_result=stations_result)

        parameter_base = self.sr.stations._parameter_base
        dataset_accessor = self.sr.stations._dataset_accessor

        parameter_ = []
        for parameter, dataset in self.sr.parameter:
            if parameter == dataset:
                parameter = [par.value for par in parameter_base[dataset_accessor][dataset_accessor]]
                parameter_.extend(parameter)
            else:
                parameter_.append(parameter.value)

        self.kml = KMLReader(
            station_ids=self.sr.station_id.to_list(), parameters=parameter_, settings=self.sr.stations.settings
        )

    def get_dwd_dmo_path(self, dataset: Enum, station_id: Optional[str] = None) -> str:
        path = f"weather/local_forecasts/dmo/{dataset.value}/{self.sr.stations.station_group.value}"
        if self.sr.stations.station_group == DwdDmoStationGroup.ALL_STATIONS:
            return f"{path}/kmz"
        return f"{path}/{station_id}/kmz/"

    @property
    def metadata(self) -> pl.DataFrame:
        """
        Wrapper for dmo metadata

        :return:
        """
        return self.sr.df

    def query(self) -> Iterator[ValuesResult]:
        """
        Replace collect data method as all information is read once from kml file

        :return:
        """
        hpm = self._create_humanized_parameters_mapping()
        self.stations_counter = 0
        self.stations_collected = []

        for df in self._collect_station_parameter():
            df = self._tidy_up_df(df)
            station_id = df.get_column(Columns.STATION_ID.value).take(0).item()
            df = self._organize_df_columns(df, station_id, self.sr.stations.dmo_type)

            if self.sr.humanize:
                df = self._humanize(df, hpm)

            # Filter for dates range if start_date and end_date are defined
            if not df.is_empty() and self.sr.start_date and self.sr.start_date != DwdForecastDate.LATEST:
                df = df.filter(pl.col(Columns.DATE.value).is_between(self.sr.start_date, self.sr.end_date))

            if not self.sr.tidy:
                df = self._tabulate_df(df)

            self.stations_counter += 1
            self.stations_collected.append(station_id)

            yield ValuesResult(stations=self.sr, values=self, df=df)

            if self.stations_counter == self.sr.rank:
                return

    def _collect_station_parameter(self) -> Iterator[pl.DataFrame]:
        """
        Wrapper of read_dmo to collect dmo data (either latest or for
        defined dates)

        :return: pandas DataFrame with data corresponding to station id and parameter
        """
        if self.sr.start_issue == DwdForecastDate.LATEST:
            for df in self.read_dmo(self.sr.stations.start_issue):
                yield df
        else:
            for date in pl.datetime_range(
                self.sr.stations.start_issue, self.sr.stations.end_issue, interval=self.sr.frequency.value, eager=True
            ):
                try:
                    for df in self.read_dmo(date):
                        yield df
                except IndexError as e:
                    log.warning(e)
                    continue

    @staticmethod
    def _tidy_up_df(df: pl.DataFrame) -> pl.DataFrame:
        """

        :param df: pandas DataFrame that is being tidied
        :param dataset: enum of dataset, used for writing dataset in pandas DataFrame
        :return: tidied pandas DataFrame
        """
        df = df.melt(
            id_vars=[
                Columns.STATION_ID.value,
                Columns.DATE.value,
            ],
            variable_name=Columns.PARAMETER.value,
            value_name=Columns.VALUE.value,
        )

        return df.with_columns(pl.lit(value=None, dtype=pl.Float64).alias(Columns.QUALITY.value))

    def read_dmo(self, date: Union[dt.datetime, DwdForecastDate]) -> Iterator[pl.DataFrame]:
        """
        Manage data acquisition for a given date that is used to filter the found files
        on the DMO path of the DWD server.

        :param date: datetime
        :return: pandas DataFrame with gathered information
        """
        for df_forecast in self._read_dmo(date):
            df_forecast = df_forecast.rename(
                mapping={
                    "datetime": Columns.DATE.value,
                }
            )

            yield df_forecast

    def _read_dmo(self, date: Union[DwdForecastDate, dt.datetime]) -> Iterator[pl.DataFrame]:
        """
        Wrapper that either calls read_icon_eu or read_icon depending on
        defined period type

        :param date: datetime
        :return: pandas DataFrame
        """
        if self.sr.stations.dmo_type == DwdDmoType.ICON_EU:
            yield from self.read_icon_eu(date)
        else:
            yield from self.read_icon(date)

    def read_icon_eu(self, date: Union[DwdForecastDate, dt.datetime]) -> Iterator[pl.DataFrame]:
        """
        Reads single icon_eu file with all stations_result and returns every chunk that
        matches with one of the defined station ids.

        :param date: datetime
        :return: pandas DataFrame with data
        """
        if self.sr.stations.station_group == DwdDmoStationGroup.ALL_STATIONS:
            dmo_path = self.get_dwd_dmo_path(DwdDmoDataset.ICON_EU)
            url = urljoin("https://opendata.dwd.de", dmo_path)
            file_url = self.get_url_for_date(url, date)
            self.kml.read(file_url)
            for forecast in self.kml.get_forecasts():
                yield forecast
        else:
            for station_id in self.sr.station_id:
                dmo_path = self.get_dwd_dmo_path(DwdDmoDataset.ICON_EU, station_id=station_id)
                station_url = urljoin("https://opendata.dwd.de", dmo_path)
                try:
                    file_url = self.get_url_for_date(station_url, date)
                except HTTPError:
                    log.warning(f"Files for {station_id} do not exist on the server")
                    continue
                self.kml.read(file_url)
                yield next(self.kml.get_forecasts())

    def read_icon(self, date: Union[DwdForecastDate, dt.datetime]) -> Iterator[pl.DataFrame]:
        """
        Reads multiple icon files with one per each station and returns a
        chunk per file.

        :param date:
        :return:
        """
        if self.sr.stations.station_group == DwdDmoStationGroup.ALL_STATIONS:
            dmo_path = self.get_dwd_dmo_path(DwdDmoDataset.ICON)
            url = urljoin("https://opendata.dwd.de", dmo_path)
            file_url = self.get_url_for_date(url, date)
            self.kml.read(file_url)
            for forecast in self.kml.get_forecasts():
                yield forecast
        else:
            for station_id in self.sr.station_id:
                dmo_path = self.get_dwd_dmo_path(DwdDmoDataset.ICON, station_id=station_id)
                station_url = urljoin("https://opendata.dwd.de", dmo_path)
                try:
                    file_url = self.get_url_for_date(station_url, date)
                except HTTPError:
                    log.warning(f"Files for {station_id} do not exist on the server")
                    continue
                self.kml.read(file_url)
                yield next(self.kml.get_forecasts())

    def get_url_for_date(self, url: str, date: Union[dt.datetime, DwdForecastDate]) -> str:
        """
        Method to get a file url based on the dmo url and the date that is
        used for filtering.

        :param url: dmo path on the dwd server
        :param date: date used for filtering of the available files
        :return: file url based on the filtering
        """
        urls = list_remote_files_fsspec(url, self.sr.stations.settings, CacheExpiry.NO_CACHE)
        df = pl.DataFrame({"url": urls})
        now = dt.datetime.utcnow()
        hours_since_month_start = (now - now.replace(day=1, hour=1, minute=1, second=1)).seconds / 60 / 60
        if hours_since_month_start < 3:
            now = now.replace(month=now.month - 1)
        last_date = (
            df.select(
                pl.col("url")
                .str.split("/")
                .list.last()
                .str.split("_")
                .list.last()
                .map_elements(lambda s: s[:-4])
                .alias("date")
            )
            .with_columns(pl.col("date").cast(int).alias("date_int"))
            .filter(pl.col("date_int").eq(pl.col("date_int").max()))
            .get_column("date")
            .to_list()[0]
        )
        day, hour = last_date[:2], last_date[2:4]
        last_date = f"{now.date().isoformat()[:-3]}-{day} {hour}:00"
        last_date = dt.datetime.fromisoformat(last_date)
        dates = [last_date - dt.timedelta(hours=i * 12) for i in range(len(df))]
        df = df.with_columns(pl.Series(dates[::-1]).alias("date"))
        if date == DwdForecastDate.LATEST:
            date = df.get_column("date").max()
        df = df.filter(pl.col("date").eq(date))
        if df.is_empty():
            raise IndexError(f"Unable to find {date} file within {url}")
        return df.get_column("url").item()


class DwdDmoRequest(TimeseriesRequest):
    """Implementation of sites for dmo sites"""

    _provider = Provider.DWD
    _kind = Kind.FORECAST
    _tz = Timezone.GERMANY
    _dataset_base = DwdDmoDataset
    _parameter_base = DwdDmoParameter
    _unit_base = DwdDmoUnit
    _resolution_type = ResolutionType.FIXED
    _resolution_base = Resolution  # use general Resolution for fixed Resolution
    _period_type = PeriodType.FIXED
    _period_base = DwdDmoPeriod
    _has_datasets = True
    _unique_dataset = True
    _data_range = DataRange.FIXED
    _values = DwdDmoValues

    _url = (
        "https://www.dwd.de/DE/leistungen/opendata/help/schluessel_datenformate/kml/"
        "dmo_stationsliste_txt.asc?__blob=publicationFile&v=1"
    )

    @property
    def _dataset_accessor(self) -> str:
        """
        Implementation for tidying dmo data

        :return:
        """
        return self.dmo_type.name

    @classmethod
    def _setup_resolution_filter(cls, resolution):
        """
        Use ICON_EU and ICON instead of resolution, which is fixed for DMO

        :param resolution:
        :return:
        """
        return (
            [parse_enumeration_from_template(res, intermediate=cls._dataset_base) for res in to_list(resolution)]
            if resolution
            else [*cls._dataset_base]
        )

    _base_columns = [
        Columns.STATION_ID.value,
        Columns.ICAO_ID.value,
        Columns.FROM_DATE.value,
        Columns.TO_DATE.value,
        Columns.HEIGHT.value,
        Columns.LATITUDE.value,
        Columns.LONGITUDE.value,
        Columns.NAME.value,
        Columns.STATE.value,
    ]

    @staticmethod
    def adjust_datetime(datetime_: dt.datetime) -> dt.datetime:
        """
        Adjust datetime to DMO release frequency (9/12 hours). Datetime is floored
        to closest release time e.g. if hour is 14, it will be rounded to 12

        :param datetime_: datetime that is adjusted
        :return: adjusted datetime with floored hour
        """
        adjusted_date = datetime_.replace(minute=0, second=0, microsecond=0)
        delta_hours = adjusted_date.hour % 12
        if delta_hours > 0:
            return adjusted_date.replace(hour=12)
        return adjusted_date

    def __init__(
        self,
        parameter: Optional[List[Union[str, DwdDmoParameter, Parameter]]],
        dmo_type: Union[str, DwdDmoType],
        start_issue: Union[str, dt.datetime, DwdForecastDate] = DwdForecastDate.LATEST,
        end_issue: Optional[Union[str, dt.datetime]] = None,
        start_date: Optional[Union[str, dt.datetime]] = None,
        end_date: Optional[Union[str, dt.datetime]] = None,
        station_group: Optional[Union[str, DwdDmoStationGroup]] = None,
        settings: Optional[Settings] = None,
    ) -> None:
        """

        :param parameter: parameter(s) to be collected
        :param dmo_type: dmo type, either small or large
        :param start_issue: start of issue of dmo which should be caught (DMO run at time XX:YY)
        :param end_issue: end of issue
        :param start_date: start date for filtering returned dataframe
        :param end_date: end date
        """
        self.dmo_type = parse_enumeration_from_template(dmo_type, DwdDmoType)
        self.station_group = (
            parse_enumeration_from_template(station_group, DwdDmoStationGroup) or DwdDmoStationGroup.SINGLE_STATIONS
        )

        super().__init__(
            parameter=parameter,
            start_date=start_date,
            end_date=end_date,
            resolution=Resolution.HOURLY,
            period=Period.FUTURE,
            settings=settings,
        )

        if not start_issue:
            start_issue = DwdForecastDate.LATEST

        try:
            start_issue = parse_enumeration_from_template(start_issue, DwdForecastDate)
        except (InvalidParameter, KeyError, ValueError):
            pass

        if start_issue is not DwdForecastDate.LATEST:
            if not start_issue and not end_issue:
                start_issue = DwdForecastDate.LATEST
            elif not end_issue:
                end_issue = start_issue
            elif not start_issue:
                start_issue = end_issue
            if isinstance(start_issue, str):
                start_issue = dt.datetime.fromisoformat(start_issue)
            start_issue = dt.datetime(start_issue.year, start_issue.month, start_issue.day, start_issue.hour)
            if isinstance(end_issue, str):
                end_issue = dt.datetime.fromisoformat(end_issue)
            end_issue = dt.datetime(end_issue.year, end_issue.month, end_issue.day, end_issue.hour)

            # Shift start date and end date to 0, 12 hour format
            start_issue = self.adjust_datetime(start_issue)
            end_issue = self.adjust_datetime(end_issue)

        # TODO: this should be replaced by the freq property in the main class
        if self.dmo_type == DwdDmoType.ICON_EU:
            self.resolution = Resolution.HOURLY
        else:
            self.resolution = Resolution.HOUR_6

        self.start_issue = start_issue
        self.end_issue = end_issue

    @property
    def issue_start(self):
        """Required for typing"""
        return self.issue_start

    @property
    def issue_end(self):
        """Required for typing"""
        return self.issue_end

    # required patches for stations as data is corrupted for these at least atm
    _station_patches = pl.DataFrame(
        [
            {
                "station_id": "03779",
                "icao_id": "EGRB",
                "name": "LONDON WEATHER CENT.",
                "latitude": "51.31",
                "longitude": "-0.12",
                "height": "5",
            },
            {
                "station_id": "03781",
                "icao_id": "----",
                "name": "KENLEY",
                "latitude": "51.18",
                "longitude": "-0.08",
                "height": "170",
            },
            {
                "station_id": "61226",
                "icao_id": "GAGO",
                "name": "GAO",
                "latitude": "16.16",
                "longitude": "-0.05",
                "height": "265",
            },
            {
                "station_id": "82106",
                "icao_id": "SBUA",
                "name": "SAO GABRIEL CACHOEI",
                "latitude": "-0.13",
                "longitude": "-67.04",
                "height": "90",
            },
            {
                "station_id": "84071",
                "icao_id": "SEQU",
                "name": "QUITO",
                "latitude": "-0.15",
                "longitude": "-78.29",
                "height": "2794",
            },
            {
                "station_id": "F9766",
                "icao_id": "SEQM",
                "name": "QUITO/MARISCAL SUCRE",
                "latitude": "-0.1",
                "longitude": "-78.21",
                "height": "2400",
            },
            {
                "station_id": "P0478",
                "icao_id": "EGLC",
                "name": "LONDON/CITY INTL",
                "latitude": "51.29",
                "longitude": "0.12",
                "height": "5",
            },
        ]
    )

    def _all(self) -> pl.LazyFrame:
        """
        Create meta data DataFrame from available station list

        :return:
        """
        payload = download_file(self._url, self.settings, CacheExpiry.METAINDEX)
        df = pd.read_fwf(
            StringIO(payload.read().decode(encoding="latin-1")),
            skiprows=2,
            skip_blank_lines=True,
            # When inferring colspecs, make sure to look at the whole file. Otherwise,
            # > this messes things up quite often and columns where strings get longer
            # > towards the end of the data may get messed up and truncated.
            # https://github.com/pandas-dev/pandas/issues/21970#issuecomment-480273621
            colspecs="infer",
            infer_nrows=np.Infinity,
            na_values=["----"],
            header=None,
            dtype="str",
        )
        df: pl.DataFrame = pl.from_pandas(df)
        df.columns = [
            Columns.STATION_ID.value,
            Columns.ICAO_ID.value,
            Columns.NAME.value,
            Columns.LATITUDE.value,
            Columns.LONGITUDE.value,
            Columns.HEIGHT.value,
        ]
        df = df.filter(pl.col("station_id").is_in(self._station_patches.get_column("station_id")).not_())
        df = pl.concat([df, self._station_patches])
        df = df.lazy()
        df = df.with_columns(
            pl.col(Columns.LATITUDE.value).str.replace(" ", "").cast(float),
            pl.col(Columns.LONGITUDE.value).str.replace(" ", "").cast(float),
        )
        return df.select([pl.col(col) if col in df.columns else pl.lit(None).alias(col) for col in self._base_columns])
