// Flow Builder - Main JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const canvas = document.getElementById('canvas');
    const configPanel = document.getElementById('config-panel');
    const newAgentModal = document.getElementById('new-agent-modal');
    const connectionTooltip = document.getElementById('connection-tooltip');
    const emptyCanvasHelp = document.getElementById('empty-canvas-help');

    // Mark non-OpenAI or Ollama agents and all tools as "coming soon"
    const markComingSoonFeatures = () => {
        // Apply to agent types (except OpenAI & Ollama)
        const agentCards = document.querySelectorAll('.agent-card');
        agentCards.forEach(card => {
            if (!['openai', 'ollama'].includes(card.getAttribute('data-type'))) {
                card.classList.add('feature-showcase');

                // Add the badge with icon
                const badge = document.createElement('div');
                badge.className = 'feature-badge';
                badge.textContent = 'Under Development';
                card.appendChild(badge);

                // Add a subtle hover title
                card.setAttribute('title', 'This feature is currently in development');
            }
        });

        // Apply to all tools
        const toolCards = document.querySelectorAll('.tool-card');
        toolCards.forEach(card => {
            card.classList.add('feature-showcase');

            // Add the badge with icon
            const badge = document.createElement('div');
            badge.className = 'feature-badge';
            badge.textContent = 'Under Development';
            card.appendChild(badge);

            // Add a subtle hover title
            card.setAttribute('title', 'This feature is currently in development');
        });

        // Style the "New" button for tools with a more elegant approach
        const createToolBtn = document.getElementById('create-tool-btn');
        if (createToolBtn) {
            createToolBtn.style.opacity = '0.7';
            createToolBtn.style.pointerEvents = 'none';
            createToolBtn.style.position = 'relative';
            createToolBtn.style.overflow = 'hidden';
            createToolBtn.setAttribute('title', 'This feature is currently in development');

            // Add a subtle shine effect
            createToolBtn.style.background = 'linear-gradient(45deg, rgba(139, 92, 246, 0.1), rgba(59, 130, 246, 0.05))';
            createToolBtn.style.border = '1px solid rgba(139, 92, 246, 0.2)';
            createToolBtn.style.borderRadius = '4px';
            createToolBtn.style.padding = '2px 8px';

            // Add a small indicator
            const indicator = document.createElement('span');
            indicator.style.fontSize = '8px';
            indicator.style.opacity = '0.8';
            indicator.style.marginLeft = '5px';
            indicator.style.letterSpacing = '0.5px';
            indicator.style.fontWeight = '600';
            indicator.style.display = 'inline-flex';
            indicator.style.alignItems = 'center';
            indicator.style.gap = '3px';

            // Add a small clock icon using SVG
            const iconSpan = document.createElement('span');
            iconSpan.style.display = 'inline-block';
            iconSpan.style.width = '8px';
            iconSpan.style.height = '8px';
            iconSpan.style.marginRight = '2px';
            iconSpan.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="rgba(59, 130, 246, 0.8)" style="width:100%;height:100%">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd" />
            </svg>`;
            iconSpan.style.animation = 'pulseEffect 2s infinite';

            indicator.appendChild(iconSpan);
            indicator.appendChild(document.createTextNode('Coming Soon'));

            // Clear and set new content
            const existingHtml = createToolBtn.innerHTML;
            createToolBtn.innerHTML = existingHtml.split('<span')[0]; // Remove any existing span
            createToolBtn.appendChild(indicator);
        }

        // In the new agent modal, style other options elegantly
        const newAgentTypeSelect = document.getElementById('new-agent-type');
        if (newAgentTypeSelect) {
            Array.from(newAgentTypeSelect.options).forEach(option => {
                if (!['openai', 'ollama'].includes(option.value)) {
                    option.disabled = true;
                    option.style.color = 'rgba(160, 160, 160, 0.6)';
                    option.style.fontStyle = 'italic';
                    option.text = option.text + ' (Under Development)';
                }
            });

            // Add custom styling to the select element
            newAgentTypeSelect.addEventListener('mouseenter', function() {
                const tooltip = document.createElement('div');
                tooltip.style.position = 'absolute';
                tooltip.style.bottom = '-30px';
                tooltip.style.left = '0';
                tooltip.style.fontSize = '10px';
                tooltip.style.color = 'rgba(167, 139, 250, 0.9)';
                tooltip.style.padding = '4px 8px';
                tooltip.style.borderRadius = '4px';
                tooltip.style.background = 'rgba(18, 18, 18, 0.8)';
                tooltip.style.zIndex = '100';
                tooltip.style.whiteSpace = 'nowrap';
                tooltip.textContent = 'Additional agent types are currently under development';
                tooltip.style.transition = 'opacity 0.3s ease';
                tooltip.style.opacity = '0';

                // Remove any existing tooltip
                const existingTooltip = document.querySelector('.agent-type-tooltip');
                if (existingTooltip) existingTooltip.remove();

                tooltip.classList.add('agent-type-tooltip');
                newAgentTypeSelect.parentNode.appendChild(tooltip);

                setTimeout(() => {
                    tooltip.style.opacity = '1';
                }, 50);
            });

            newAgentTypeSelect.addEventListener('mouseleave', function() {
                const tooltip = document.querySelector('.agent-type-tooltip');
                if (tooltip) {
                    tooltip.style.opacity = '0';
                    setTimeout(() => {
                        tooltip.remove();
                    }, 300);
                }
            });
        }
    };

    // Call the function to mark features as coming soon
    markComingSoonFeatures();
    
    // Global variables
    let nodeCounter = 0;
    let nodes = [];
    let connections = [];
    let draggedCard = null;
    let selectedNode = null;
    let connectionStart = null;
    let connectionEnd = null;
    let isDragging = false;
    let dragOffsetX = 0;
    let dragOffsetY = 0;
    let canvasOffset = { x: 0, y: 0 };
    let canvasScale = 1;
    
    // SVG namespace
    const svgNS = "http://www.w3.org/2000/svg";
    
    // Create a node container for proper node positioning (adding first so it's below the connection layer)
    const nodeContainer = document.createElement("div");
    nodeContainer.setAttribute("id", "node-container");
    nodeContainer.style.position = "absolute";
    nodeContainer.style.top = "0";
    nodeContainer.style.left = "0";
    nodeContainer.style.width = "100%";
    nodeContainer.style.height = "100%";
    nodeContainer.style.zIndex = "1";
    canvas.appendChild(nodeContainer);
    
    // Create SVG element for connections (adding second so it's above the node container)
    const svgContainer = document.createElementNS(svgNS, "svg");
    svgContainer.setAttribute("id", "connections-container");
    svgContainer.style.position = "absolute";
    svgContainer.style.top = "0";
    svgContainer.style.left = "0";
    svgContainer.style.width = "100%";
    svgContainer.style.height = "100%";
    svgContainer.style.pointerEvents = "none";
    svgContainer.style.zIndex = "2";
    canvas.appendChild(svgContainer);
    
    // Initialize drag and drop for agent and tool cards
    initDragAndDrop();
    
    // Initialize canvas events
    initCanvasEvents();
    
    // Initialize UI controls
    initUIControls();
    
    // Initialize zoom and pan functionality
    initZoomAndPan();
    
    /**
     * Initialize drag and drop functionality for agent, tool, and IO cards
     */
    function initDragAndDrop() {
        const agentCards = document.querySelectorAll('.agent-card');
        const toolCards = document.querySelectorAll('.tool-card');
        const ioCards = document.querySelectorAll('.io-card');
        
        // Add drag functionality to agent cards
        agentCards.forEach(card => {
            card.addEventListener('dragstart', (e) => {
                draggedCard = {
                    type: 'agent',
                    agentType: card.getAttribute('data-type'),
                    element: card
                };
                
                // Create a ghost image for dragging
                const ghostElement = card.cloneNode(true);
                ghostElement.style.position = 'absolute';
                ghostElement.style.top = '-1000px';
                document.body.appendChild(ghostElement);
                e.dataTransfer.setDragImage(ghostElement, 0, 0);
                
                setTimeout(() => {
                    document.body.removeChild(ghostElement);
                }, 0);
            });
            
            card.addEventListener('dragend', () => {
                draggedCard = null;
            });
        });
        
        // Add drag functionality to tool cards
        toolCards.forEach(card => {
            card.addEventListener('dragstart', (e) => {
                draggedCard = {
                    type: 'tool',
                    toolType: card.getAttribute('data-type'),
                    element: card
                };
                
                // Create a ghost image for dragging
                const ghostElement = card.cloneNode(true);
                ghostElement.style.position = 'absolute';
                ghostElement.style.top = '-1000px';
                document.body.appendChild(ghostElement);
                e.dataTransfer.setDragImage(ghostElement, 0, 0);
                
                setTimeout(() => {
                    document.body.removeChild(ghostElement);
                }, 0);
            });
            
            card.addEventListener('dragend', () => {
                draggedCard = null;
            });
        });
        
        // Add drag functionality to IO cards
        ioCards.forEach(card => {
            card.addEventListener('dragstart', (e) => {
                draggedCard = {
                    type: card.getAttribute('data-type'), // 'input', 'output', or 'router'
                    element: card
                };
                
                // Create a ghost image for dragging
                const ghostElement = card.cloneNode(true);
                ghostElement.style.position = 'absolute';
                ghostElement.style.top = '-1000px';
                document.body.appendChild(ghostElement);
                e.dataTransfer.setDragImage(ghostElement, 0, 0);
                
                setTimeout(() => {
                    document.body.removeChild(ghostElement);
                }, 0);
            });
            
            card.addEventListener('dragend', () => {
                draggedCard = null;
            });
        });
        
        // Add drop functionality to the canvas
        canvas.addEventListener('dragover', (e) => {
            e.preventDefault();
        });
        
        canvas.addEventListener('drop', (e) => {
            e.preventDefault();
            
            if (draggedCard) {
                // Get canvas-relative position
                const rect = canvas.getBoundingClientRect();
                const x = (e.clientX - rect.left - canvasOffset.x) / canvasScale;
                const y = (e.clientY - rect.top - canvasOffset.y) / canvasScale;
                
                // Create a new node at the drop position
                createNode(x, y, draggedCard);
                
                // Remove the help text if it's the first node
                if (nodes.length === 1) {
                    emptyCanvasHelp.style.display = 'none';
                }
            }
        });
    }
    
    /**
     * Initialize canvas events for node selection and connection creation
     */
    function initCanvasEvents() {
        // Handle canvas click to deselect nodes
        canvas.addEventListener('click', (e) => {
            // Only deselect if the canvas itself was clicked, not a node
            if (e.target === canvas || e.target === svgContainer) {
                deselectAllNodes();
                hideConfigPanel();
            }
        });
    }
    
    /**
     * Initialize UI controls like buttons and panels
     */
    function initUIControls() {
        // Config panel
        const closeConfigBtn = document.getElementById('close-config');
        const applyConfigBtn = document.getElementById('apply-config');
        const cancelConfigBtn = document.getElementById('cancel-config');
        
        closeConfigBtn.addEventListener('click', hideConfigPanel);
        cancelConfigBtn.addEventListener('click', hideConfigPanel);
        applyConfigBtn.addEventListener('click', applyNodeConfig);
        
        // New agent modal
        const createAgentBtn = document.getElementById('create-agent-btn');
        const createAgentConfirmBtn = document.getElementById('create-agent-confirm');
        const createToolBtn = document.getElementById('create-tool-btn');
        const modalCloseButtons = document.querySelectorAll('.modal-close');
        const newAgentTypeSelect = document.getElementById('new-agent-type');
        const newAgentConfigContainer = document.getElementById('new-agent-config-container');
        
        createAgentBtn.addEventListener('click', showNewAgentModal);
        createAgentConfirmBtn.addEventListener('click', createNewAgent);
        createToolBtn.addEventListener('click', () => {
            // TODO: Implement create tool functionality
            alert('Create tool functionality will be implemented in the next phase.');
        });
        
        modalCloseButtons.forEach(btn => {
            btn.addEventListener('click', hideModals);
        });
        
        // Agent type change in modal
        newAgentTypeSelect.addEventListener('change', () => {
            updateNewAgentConfigFields(newAgentTypeSelect.value);
        });
        
        // Initialize with the first option
        updateNewAgentConfigFields(newAgentTypeSelect.value);
        
        // Toolbar buttons
        document.getElementById('add-input-btn').addEventListener('click', addInputNode);
        document.getElementById('add-output-btn').addEventListener('click', addOutputNode);
        document.getElementById('undo-btn').addEventListener('click', undoAction);
        document.getElementById('redo-btn').addEventListener('click', redoAction);
        document.getElementById('zoom-in-btn').addEventListener('click', zoomIn);
        document.getElementById('zoom-out-btn').addEventListener('click', zoomOut);
        document.getElementById('zoom-reset-btn').addEventListener('click', resetZoom);
        document.getElementById('delete-btn').addEventListener('click', deleteSelected);
        
        // Utility controls
        document.getElementById('floating-zoom-reset').addEventListener('click', resetZoom);
        
        // Canvas control buttons
        document.getElementById('floating-save-btn').addEventListener('click', saveNetwork);
        document.getElementById('floating-run-btn').addEventListener('click', runNetwork);
    }
    
    /**
     * Initialize zoom and pan functionality for the canvas
     */
    function initZoomAndPan() {
        let isPanning = false;
        let startPoint = { x: 0, y: 0 };
        
        // Handle panning with middle mouse button or spacebar + left mouse button
        canvas.addEventListener('mousedown', (e) => {
            // Middle mouse button (button 1) or spacebar + left button
            if (e.button === 1 || (e.button === 0 && e.getModifierState('Space'))) {
                isPanning = true;
                startPoint = { x: e.clientX, y: e.clientY };
                canvas.style.cursor = 'grabbing';
                e.preventDefault();
            }
        });
        
        window.addEventListener('mousemove', (e) => {
            if (isPanning) {
                const dx = e.clientX - startPoint.x;
                const dy = e.clientY - startPoint.y;
                
                canvasOffset.x += dx;
                canvasOffset.y += dy;
                
                updateCanvasTransform();
                
                startPoint = { x: e.clientX, y: e.clientY };
            }
        });
        
        window.addEventListener('mouseup', () => {
            if (isPanning) {
                isPanning = false;
                canvas.style.cursor = 'default';
            }
        });
        
        // Handle zooming with mouse wheel
        canvas.addEventListener('wheel', (e) => {
            e.preventDefault();
            
            const delta = e.deltaY < 0 ? 0.1 : -0.1;
            const newScale = Math.min(Math.max(canvasScale + delta, 0.3), 2);
            
            // Calculate the position under the mouse (in canvas space)
            const rect = canvas.getBoundingClientRect();
            const mouseX = e.clientX - rect.left;
            const mouseY = e.clientY - rect.top;
            
            // Calculate the point in world space
            const worldX = (mouseX - canvasOffset.x) / canvasScale;
            const worldY = (mouseY - canvasOffset.y) / canvasScale;
            
            // Update the scale
            canvasScale = newScale;
            
            // Calculate the new offset to keep the point under the mouse
            canvasOffset.x = mouseX - worldX * canvasScale;
            canvasOffset.y = mouseY - worldY * canvasScale;
            
            updateCanvasTransform();
        });
    }
    
    /**
     * Update canvas transform based on offset and scale
     */
    function updateCanvasTransform() {
        // Apply transformation only to the node container and SVG container
        const nodeContainerEl = document.getElementById('node-container');
        const svgContainerEl = document.getElementById('connections-container');
        
        nodeContainerEl.style.transform = `translate(${canvasOffset.x}px, ${canvasOffset.y}px) scale(${canvasScale})`;
        svgContainerEl.style.transform = `translate(${canvasOffset.x}px, ${canvasOffset.y}px) scale(${canvasScale})`;
    }
    
    /**
     * Create a new node on the canvas
     * @param {number} x - The x position on the canvas
     * @param {number} y - The y position on the canvas
     * @param {Object} cardData - The data for the card being dropped
     */
    function createNode(x, y, cardData) {
        const nodeId = 'node-' + (++nodeCounter);
        const nodeElement = document.createElement('div');
        nodeElement.id = nodeId;
        nodeElement.classList.add('node');
        
        // Add appropriate class based on type
        if (cardData.type === 'agent') {
            nodeElement.classList.add('agent');
            
            // Get icon and type-specific data
            let icon = 'robot';
            let typeName = 'Agent';
            
            switch (cardData.agentType) {
                case 'openai':
                    typeName = 'OpenAI';
                    break;
                case 'ollama':
                    typeName = 'Ollama';
                    break;
                case 'anthropic':
                    typeName = 'Claude';
                    break;
                case 'bedrock':
                    typeName = 'Bedrock';
                    break;
                case 'custom':
                    typeName = 'Custom';
                    icon = 'code-square';
                    break;
            }
            
            nodeElement.innerHTML = `
                <div class="node-header">
                    <div class="node-title">
                        <i class="bi bi-${icon}"></i>
                        <span>${typeName} Agent</span>
                    </div>
                    <div class="node-actions">
                        <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                        <button class="node-btn delete-node" title="Delete"><i class="bi bi-trash"></i></button>
                        <button class="node-btn" title="Duplicate"><i class="bi bi-copy"></i></button>
                    </div>
                </div>
                <div class="node-content">
                    <div>Configure this agent</div>
                    <div class="node-badge agent-model">Not configured</div>
                </div>
                <div class="node-ports">
                    <div class="input-port">
                        <div class="port port-left" data-port-type="input"></div>
                        <span class="port-label">Input</span>
                    </div>
                    <div class="output-port">
                        <div class="port port-right" data-port-type="output"></div>
                        <span class="port-label">Output</span>
                    </div>
                </div>
            `;
        } else if (cardData.type === 'tool') {
            nodeElement.classList.add('tool');
            
            // Get icon and type-specific data
            let icon = 'tools';
            let typeName = 'Tool';
            
            switch (cardData.toolType) {
                case 'search':
                    typeName = 'Search';
                    icon = 'search';
                    break;
                case 'calculator':
                    typeName = 'Calculator';
                    icon = 'calculator';
                    break;
                case 'database':
                    typeName = 'Database';
                    icon = 'database';
                    break;
                case 'router':
                    typeName = 'Router';
                    icon = 'signpost-split';
                    break;
            }
            
            nodeElement.innerHTML = `
                <div class="node-header">
                    <div class="node-title">
                        <i class="bi bi-${icon}"></i>
                        <span>${typeName} Tool</span>
                    </div>
                    <div class="node-actions">
                        <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                        <button class="node-btn delete-node" title="Delete"><i class="bi bi-trash"></i></button>
                        <button class="node-btn" title="Duplicate"><i class="bi bi-copy"></i></button>
                    </div>
                </div>
                <div class="node-content">
                    <div>Configure this tool</div>
                </div>
                <div class="node-ports">
                    <div class="input-port">
                        <div class="port port-left" data-port-type="input"></div>
                        <span class="port-label">Input</span>
                    </div>
                    <div class="output-port">
                        <div class="port port-right" data-port-type="output"></div>
                        <span class="port-label">Output</span>
                    </div>
                </div>
            `;
        } else if (cardData.type === 'input') {
            nodeElement.classList.add('input');
            
            nodeElement.innerHTML = `
                <div class="node-header">
                    <div class="node-title">
                        <i class="bi bi-box-arrow-in-right"></i>
                        <span>Input</span>
                    </div>
                    <div class="node-actions">
                        <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                        <button class="node-btn delete-node" title="Delete"><i class="bi bi-trash"></i></button>
                    </div>
                </div>
                <div class="node-content">
                    <div>Network input</div>
                </div>
                <div class="node-ports">
                    <div class="output-port">
                        <span class="port-label">Output</span>
                        <div class="port port-right" data-port-type="output"></div>
                    </div>
                </div>
            `;
        } else if (cardData.type === 'output') {
            nodeElement.classList.add('output');

            nodeElement.innerHTML = `
                <div class="node-header">
                    <div class="node-title">
                        <i class="bi bi-box-arrow-right"></i>
                        <span>Output</span>
                    </div>
                    <div class="node-actions">
                        <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                        <button class="node-btn delete-node" title="Delete"><i class="bi bi-trash"></i></button>
                    </div>
                </div>
                <div class="node-content">
                    <div>Network output</div>
                </div>
                <div class="node-ports">
                    <div class="input-port">
                        <div class="port port-left" data-port-type="input"></div>
                        <span class="port-label">Input</span>
                    </div>
                </div>
            `;
        } else if (cardData.type === 'router') {
            nodeElement.classList.add('router');

            // Generate output port HTML for 2 default ports
            let outputPortsHTML = '';
            for (let i = 0; i < 2; i++) {
                outputPortsHTML += `
                    <div class="output-port" data-port-number="${i}">
                        <span class="port-label">Output ${i + 1}</span>
                        <div class="port port-right" data-port-type="output" data-port-number="${i}"></div>
                    </div>
                `;
            }

            nodeElement.innerHTML = `
                <div class="node-header">
                    <div class="node-title">
                        <i class="bi bi-signpost-split"></i>
                        <span>Router</span>
                    </div>
                    <div class="node-actions">
                        <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                        <button class="node-btn delete-node" title="Delete"><i class="bi bi-trash"></i></button>
                    </div>
                </div>
                <div class="node-content">
                    <div>Route messages dynamically</div>
                    <div class="node-badge router-strategy">Keyword Routing</div>
                </div>
                <div class="node-ports">
                    <div class="input-port">
                        <div class="port port-left" data-port-type="input"></div>
                        <span class="port-label">Input</span>
                    </div>
                    <div class="output-ports-container">
                        ${outputPortsHTML}
                    </div>
                </div>
                <div class="add-port-container">
                    <button class="add-port-btn"><i class="bi bi-plus-circle"></i> Add Output</button>
                </div>
            `;

            // Add listener to the "Add Output" button
            nodeElement.addEventListener('DOMNodeInserted', function() {
                const addPortBtn = nodeElement.querySelector('.add-port-btn');
                if (addPortBtn) {
                    addPortBtn.addEventListener('click', function(e) {
                        e.stopPropagation();

                        // Get the current number of output ports
                        const outputPortsContainer = nodeElement.querySelector('.output-ports-container');
                        const currentPorts = outputPortsContainer.querySelectorAll('.output-port').length;

                        // Create a new output port
                        const newPortNumber = currentPorts;
                        const newPortElement = document.createElement('div');
                        newPortElement.className = 'output-port';
                        newPortElement.setAttribute('data-port-number', newPortNumber);
                        newPortElement.innerHTML = `
                            <span class="port-label">Output ${newPortNumber + 1}</span>
                            <div class="port port-right" data-port-type="output" data-port-number="${newPortNumber}"></div>
                        `;

                        // Add the new port to the container
                        outputPortsContainer.appendChild(newPortElement);

                        // Find the node data
                        const nodeData = nodes.find(n => n.id === nodeElement.id);
                        if (nodeData) {
                            // Initialize outputPorts if it doesn't exist
                            if (!nodeData.config.outputPorts) {
                                nodeData.config.outputPorts = 2; // Default starting with 2
                            }

                            // Update the count
                            nodeData.config.outputPorts = currentPorts + 1;

                            // Add port event listeners
                            const portEl = newPortElement.querySelector('.port');
                            if (portEl) {
                                initNodeEvents(nodeElement, nodeData);
                            }
                        }

                        // Update connections
                        updateConnections();
                    });
                }
            });
        }

        // Position the node
        nodeElement.style.left = `${x}px`;
        nodeElement.style.top = `${y}px`;
        
        // Add the node to the node container
        nodeContainer.appendChild(nodeElement);
        
        // Store node data
        // Initialize node data with appropriate defaults
        const nodeData = {
            id: nodeId,
            element: nodeElement,
            type: cardData.type,
            subType: cardData.type === 'agent' ? cardData.agentType :
                   (cardData.type === 'tool' ? cardData.toolType : null),
            position: { x, y },
            config: {}
        };

        // Add type-specific default configuration
        if (cardData.type === 'router') {
            nodeData.config = {
                name: 'Router',
                routingStrategy: 'keyword',
                outputPorts: 2,
                keywordPatterns: [
                    { keyword: '.*', port: 0, originalType: 'regex', originalValue: '.*' },  // Match everything as a default pattern for port 0
                    { keyword: '.*question.*', port: 1, originalType: 'regex', originalValue: '.*question.*' }  // Example pattern for port 1
                ],
                portWeights: { 0: 1.0, 1: 1.0 },
                contentTypeMappings: [
                    { contentType: 'text', port: 0 },
                    { contentType: 'json', port: 1 }
                ]
            };
        }

        nodes.push(nodeData);
        
        // Add event listeners to the node
        initNodeEvents(nodeElement, nodeData);
        
        return nodeData;
    }
    
    /**
     * Initialize events for a node
     * @param {HTMLElement} nodeElement - The DOM element for the node
     * @param {Object} nodeData - The data object for the node
     */
    function initNodeEvents(nodeElement, nodeData) {
        // Make the node draggable with jQuery-like approach
        nodeElement.addEventListener('mousedown', (e) => {
            // Ignore if clicking on a port or button
            if (e.target.classList.contains('port') || 
                e.target.classList.contains('node-btn') ||
                e.target.parentElement.classList.contains('node-btn')) {
                return;
            }
            
            // Get node position relative to canvas
            const canvasRect = canvas.getBoundingClientRect();
            const nodeRect = nodeElement.getBoundingClientRect();
            
            dragOffsetX = e.clientX - nodeRect.left;
            dragOffsetY = e.clientY - nodeRect.top;
            
            selectedNode = nodeData;
            isDragging = true;
            
            // Add selected class
            deselectAllNodes();
            nodeElement.classList.add('selected');
            
            // Apply higher z-index when dragging to be above the connection layer
            nodeElement.style.zIndex = '10';
            
            // Add mousemove and mouseup handlers to document
            const mouseMoveHandler = (moveEvent) => {
                if (isDragging) {
                    // Calculate the new position in canvas coordinates
                    const x = (moveEvent.clientX - canvasRect.left - dragOffsetX) / canvasScale;
                    const y = (moveEvent.clientY - canvasRect.top - dragOffsetY) / canvasScale;
                    
                    // Update node position immediately
                    nodeElement.style.left = `${x}px`;
                    nodeElement.style.top = `${y}px`;
                    nodeData.position = { x, y };
                    
                    // Update connections
                    updateConnections();
                }
            };
            
            const mouseUpHandler = () => {
                isDragging = false;
                nodeElement.style.zIndex = '';
                document.removeEventListener('mousemove', mouseMoveHandler);
                document.removeEventListener('mouseup', mouseUpHandler);
            };
            
            document.addEventListener('mousemove', mouseMoveHandler);
            document.addEventListener('mouseup', mouseUpHandler);
            
            e.stopPropagation();
        });
        
        // Handle node selection
        nodeElement.addEventListener('click', (e) => {
            // Ignore if clicking on a port or button
            if (e.target.classList.contains('port') || 
                e.target.classList.contains('node-btn') ||
                e.target.parentElement.classList.contains('node-btn')) {
                return;
            }
            
            deselectAllNodes();
            nodeElement.classList.add('selected');
            selectedNode = nodeData;
            
            e.stopPropagation();
        });
        
        // Configure button
        const configureBtn = nodeElement.querySelector('.configure-node');
        if (configureBtn) {
            configureBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                deselectAllNodes();
                nodeElement.classList.add('selected');
                selectedNode = nodeData;
                
                showConfigPanel(nodeData);
            });
        }
        
        // Delete button
        const deleteBtn = nodeElement.querySelector('.delete-node');
        if (deleteBtn) {
            deleteBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                // Find all connections to this node
                const relatedConnections = connections.filter(conn => 
                    conn.start.nodeId === nodeData.id || conn.end.nodeId === nodeData.id
                );
                
                // Remove connections
                relatedConnections.forEach(conn => {
                    svgContainer.removeChild(conn.element);
                    connections = connections.filter(c => c.id !== conn.id);
                });
                
                // Remove node
                nodeContainer.removeChild(nodeElement);
                nodes = nodes.filter(n => n.id !== nodeData.id);
                
                // Show help text if no nodes left
                if (nodes.length === 0) {
                    emptyCanvasHelp.style.display = 'flex';
                }
            });
        }
        
        // Add port event listeners for connections
        const ports = nodeElement.querySelectorAll('.port');
        
        ports.forEach(port => {
            // Start connection
            port.addEventListener('mousedown', (e) => {
                e.stopPropagation();
                
                const portType = port.getAttribute('data-port-type');
                const portRect = port.getBoundingClientRect();
                const canvasRect = canvas.getBoundingClientRect();
                
                // Calculate port center position in canvas coordinates
                const startX = (portRect.left + portRect.width / 2 - canvasRect.left - canvasOffset.x) / canvasScale;
                const startY = (portRect.top + portRect.height / 2 - canvasRect.top - canvasOffset.y) / canvasScale;
                
                // Check if this is a router output port (they have port numbers)
                const portNumber = port.getAttribute('data-port-number');

                connectionStart = {
                    nodeId: nodeData.id,
                    port: portType,
                    portNumber: portNumber, // Will be undefined for non-router ports
                    x: startX,
                    y: startY
                };
                
                // Create temporary connection line
                const tempConnection = document.createElementNS(svgNS, 'path');
                tempConnection.setAttribute('id', 'temp-connection');
                tempConnection.setAttribute('class', 'temp-connection');
                svgContainer.appendChild(tempConnection);
                
                // Show connection tooltip
                connectionTooltip.style.opacity = '1';
                connectionTooltip.textContent = 'Click on another port to connect';
                
                e.preventDefault();
            });
            
            // End connection
            port.addEventListener('mouseup', (e) => {
                e.stopPropagation();
                
                if (connectionStart) {
                    const portType = port.getAttribute('data-port-type');
                    const portRect = port.getBoundingClientRect();
                    const canvasRect = canvas.getBoundingClientRect();
                    
                    // Calculate port center position in canvas coordinates
                    const endX = (portRect.left + portRect.width / 2 - canvasRect.left - canvasOffset.x) / canvasScale;
                    const endY = (portRect.top + portRect.height / 2 - canvasRect.top - canvasOffset.y) / canvasScale;
                    
                    // Check if this is a router output port (they have port numbers)
                    const portNumber = port.getAttribute('data-port-number');

                    connectionEnd = {
                        nodeId: nodeData.id,
                        port: portType,
                        portNumber: portNumber, // Will be undefined for non-router ports
                        x: endX,
                        y: endY
                    };
                    
                    // Only create connection if:
                    // 1. Not connecting to the same node
                    // 2. Connecting from output to input
                    if (connectionStart.nodeId !== connectionEnd.nodeId &&
                        ((connectionStart.port === 'output' && connectionEnd.port === 'input') ||
                         (connectionStart.port === 'input' && connectionEnd.port === 'output'))) {
                        
                        // Ensure start is always the output port
                        if (connectionStart.port === 'input') {
                            const temp = connectionStart;
                            connectionStart = connectionEnd;
                            connectionEnd = temp;
                        }
                        
                        createConnection(connectionStart, connectionEnd);
                    }
                    
                    // Remove temporary connection
                    const tempConnection = document.getElementById('temp-connection');
                    if (tempConnection) {
                        svgContainer.removeChild(tempConnection);
                    }
                    
                    // Hide tooltip
                    connectionTooltip.style.opacity = '0';
                    
                    connectionStart = null;
                    connectionEnd = null;
                }
            });
            
            // Hover effect
            port.addEventListener('mouseenter', () => {
                connectionTooltip.style.opacity = '1';
                const rect = port.getBoundingClientRect();
                
                connectionTooltip.style.left = `${rect.left + rect.width / 2}px`;
                connectionTooltip.style.top = `${rect.top - 30}px`;
                
                const portType = port.getAttribute('data-port-type');
                if (connectionStart) {
                    if ((connectionStart.port === 'output' && portType === 'input') ||
                        (connectionStart.port === 'input' && portType === 'output')) {
                        connectionTooltip.textContent = 'Click to connect';
                    } else {
                        connectionTooltip.textContent = 'Invalid connection';
                    }
                } else {
                    connectionTooltip.textContent = portType === 'input' ? 'Input port' : 'Output port';
                }
            });
            
            port.addEventListener('mouseleave', () => {
                if (!connectionStart) {
                    connectionTooltip.style.opacity = '0';
                }
            });
        });
    }
    
    /**
     * Create a connection between two nodes
     * @param {Object} start - The starting port information
     * @param {Object} end - The ending port information
     */
    function createConnection(start, end) {
        const uniqueId = Date.now().toString(36) + Math.random().toString(36).substr(2, 5);
        const connectionId = `connection-${start.nodeId}-${end.nodeId}-${uniqueId}`;
        
        // Check if connection already exists between these exact nodes
        // We only block duplicate connections between the same exact ports/nodes
        const existingConnection = connections.find(conn => 
            (conn.start.nodeId === start.nodeId && conn.end.nodeId === end.nodeId && 
             conn.start.port === start.port && conn.end.port === end.port) || 
            (conn.start.nodeId === end.nodeId && conn.end.nodeId === start.nodeId &&
             conn.start.port === end.port && conn.end.port === start.port)
        );
        
        if (existingConnection) {
            return;
        }
        
        // Create SVG path for the connection
        const path = document.createElementNS(svgNS, 'path');
        path.setAttribute('id', connectionId);
        path.setAttribute('class', 'connection');
        
        // Calculate the path with improved bezier curve control points
        const dx = end.x - start.x;
        const dy = end.y - start.y;
        const distance = Math.sqrt(dx * dx + dy * dy);
        
        // Use dynamic control point distances based on connection length
        const bezierX = Math.min(Math.abs(dx) * 0.5, distance * 0.4);
        const bezierY = Math.min(Math.abs(dy) * 0.2, distance * 0.1);
        
        // Create smoother S-curved path
        const d = `M ${start.x},${start.y} 
                   C ${start.x + bezierX},${start.y + bezierY} 
                     ${end.x - bezierX},${end.y - bezierY} 
                     ${end.x},${end.y}`;
        path.setAttribute('d', d);
        
        svgContainer.appendChild(path);
        
        // Store connection data
        const connectionData = {
            id: connectionId,
            element: path,
            start: {
                nodeId: start.nodeId,
                port: start.port,
                portNumber: start.portNumber, // For router output ports
                x: start.x,
                y: start.y
            },
            end: {
                nodeId: end.nodeId,
                port: end.port,
                portNumber: end.portNumber, // For router output ports
                x: end.x,
                y: end.y
            }
        };
        
        connections.push(connectionData);
        
        // Add hover and click events for better connection interaction
        let deleteBtn = null;

        // Create an invisible hitbox that matches the path but is wider
        // This gives users a larger area to interact with
        const hitbox = document.createElementNS(svgNS, 'path');
        hitbox.setAttribute('id', `hitbox-${connectionId}`);
        hitbox.setAttribute('class', 'connection-hitbox');
        hitbox.setAttribute('d', path.getAttribute('d'));
        svgContainer.insertBefore(hitbox, path);

        // Function to show delete button when hovering the connection
        // Create a variable to track hover intent
        let hoverIntentTimer = null;
        let isButtonVisible = false;

        function showDeleteButton() {
            // Clear any existing timer to prevent race conditions
            if (hoverIntentTimer) {
                clearTimeout(hoverIntentTimer);
            }

            // Don't create delete button if one already exists
            if (document.getElementById(`delete-${connectionId}`)) {
                return;
            }

            // Use a short delay before showing the button to prevent flickering
            // This creates a more stable experience when moving across the line
            hoverIntentTimer = setTimeout(() => {
                // Double-check that we're still hovering before creating the button
                if (!path.matches(':hover') && !hitbox.matches(':hover')) {
                    return;
                }

                // Position delete button at the center of the path
                const rect = path.getBoundingClientRect();
                const centerX = rect.left + (rect.width / 2);
                const centerY = rect.top + (rect.height / 2);

                // Create delete button
                deleteBtn = document.createElement('button');
                deleteBtn.setAttribute('id', `delete-${connectionId}`);
                deleteBtn.classList.add('connection-delete-btn');
                deleteBtn.innerHTML = '<i class="bi bi-trash"></i>';
                deleteBtn.style.position = 'absolute';
                deleteBtn.style.left = `${centerX}px`;
                deleteBtn.style.top = `${centerY}px`;
                deleteBtn.setAttribute('title', 'Delete Connection');

            // Delete action - single click with no confirmation
            deleteBtn.addEventListener('click', (evt) => {
                evt.stopPropagation();

                // Visual fade-out effect
                path.style.opacity = '0';

                // Animate the delete button
                deleteBtn.style.transform = 'translate(-50%, -50%) scale(1.2)';
                deleteBtn.style.opacity = '0';

                // Remove everything after animation
                setTimeout(() => {
                    // Remove the path
                    if (svgContainer.contains(path)) {
                        svgContainer.removeChild(path);
                    }

                    // Remove the hitbox
                    if (svgContainer.contains(hitbox)) {
                        svgContainer.removeChild(hitbox);
                    }

                    // Remove the delete button
                    if (document.body.contains(deleteBtn)) {
                        document.body.removeChild(deleteBtn);
                    }

                    // Update connections array
                    connections = connections.filter(c => c.id !== connectionId);

                    // Show success notification
                    showNotification('Connection deleted', 'success');
                }, 200);
            });

                // Add to DOM with a short delay to prevent flickering
                document.body.appendChild(deleteBtn);

                // Add mouse events to the delete button itself for consistent behavior
                deleteBtn.addEventListener('mouseleave', (evt) => {
                    // When mouse leaves the button, remove it if not over the path or hitbox
                    if (!path.matches(':hover') && !hitbox.matches(':hover')) {
                        deleteBtn.style.opacity = '0';
                        deleteBtn.style.transform = 'translate(-50%, -50%) scale(0.8)';
                        path.classList.remove('selected');

                        setTimeout(() => {
                            if (document.body.contains(deleteBtn)) {
                                document.body.removeChild(deleteBtn);
                            }
                        }, 100);
                    }
                });

                // Add selected class to highlight
                path.classList.add('selected');
            }, 100); // Increased delay for better stability
        }

        // Mouse over event to show delete button
        path.addEventListener('mouseover', showDeleteButton);
        hitbox.addEventListener('mouseover', showDeleteButton);

        // Variable to track mouseout intent - prevents flickering
        let mouseOutTimer = null;

        // Function to handle mouse leaving the connection with improved stability
        function handleMouseOut(e) {
            // Don't remove if mouse moved to any part of our connection or button
            if (e.relatedTarget === deleteBtn ||
                e.relatedTarget === hitbox ||
                e.relatedTarget === path) {
                if (mouseOutTimer) {
                    clearTimeout(mouseOutTimer);
                    mouseOutTimer = null;
                }
                return;
            }

            // Clear any existing hover intent timer
            if (hoverIntentTimer) {
                clearTimeout(hoverIntentTimer);
                hoverIntentTimer = null;
            }

            // Add a short delay before removing the button to prevent flickering
            // when quickly moving across elements
            mouseOutTimer = setTimeout(() => {
                // Double-check that we're really not hovering anything related
                if (path.matches(':hover') ||
                    hitbox.matches(':hover') ||
                    (deleteBtn && deleteBtn.matches(':hover'))) {
                    return;
                }

                const btn = document.getElementById(`delete-${connectionId}`);
                if (!btn) return;

                // Fade-out animation
                btn.style.opacity = '0';
                btn.style.transform = 'translate(-50%, -50%) scale(0.8)';
                path.classList.remove('selected');

                // Remove from DOM after animation
                setTimeout(() => {
                    if (document.body.contains(btn)) {
                        document.body.removeChild(btn);
                    }
                }, 100);
            }, 50); // Short delay for better stability
        }

        // Add mouse out handlers to both path and hitbox
        path.addEventListener('mouseout', handleMouseOut);
        hitbox.addEventListener('mouseout', handleMouseOut);

        // Click event for the connection - reuse the same showDeleteButton function
        path.addEventListener('click', (e) => {
            e.stopPropagation();

            // Same behavior as hover - show the delete button
            showDeleteButton();

            // Highlight the connection
            path.classList.add('selected');

            // Set up a one-time click elsewhere listener to hide the button
            const hideButtonOnClickElsewhere = (evt) => {
                if (evt.target !== path &&
                    evt.target !== hitbox &&
                    evt.target !== deleteBtn) {

                    const btn = document.getElementById(`delete-${connectionId}`);
                    if (btn) {
                        // Fade out nicely
                        btn.style.opacity = '0';
                        setTimeout(() => {
                            if (document.body.contains(btn)) {
                                document.body.removeChild(btn);
                            }
                        }, 200);
                    }

                    // Remove highlight
                    path.classList.remove('selected');

                    // Remove this event listener
                    document.removeEventListener('click', hideButtonOnClickElsewhere);
                }
            };

            // Add the click elsewhere listener after a short delay
            setTimeout(() => {
                document.addEventListener('click', hideButtonOnClickElsewhere);
            }, 10);
        });

        // Also add click handler to the hitbox for consistency
        hitbox.addEventListener('click', (e) => {
            e.stopPropagation();
            showDeleteButton();
        });
        
        return connectionData;
    }
    
    /**
     * Update a temporary connection while dragging
     * @param {number} endX - The current mouse X position
     * @param {number} endY - The current mouse Y position
     */
    function updateTempConnection(endX, endY) {
        if (!connectionStart) return;
        
        const tempConnection = document.getElementById('temp-connection');
        if (!tempConnection) return;
        
        const dx = endX - connectionStart.x;
        const dy = endY - connectionStart.y;
        const distance = Math.sqrt(dx * dx + dy * dy);
        
        // Use dynamic control point distances based on connection length
        const bezierX = Math.min(Math.abs(dx) * 0.5, distance * 0.4);
        const bezierY = Math.min(Math.abs(dy) * 0.2, distance * 0.1);
        
        // Create smoother S-curved path
        const d = `M ${connectionStart.x},${connectionStart.y} 
                   C ${connectionStart.x + bezierX},${connectionStart.y + bezierY} 
                     ${endX - bezierX},${endY - bezierY} 
                     ${endX},${endY}`;
        tempConnection.setAttribute('d', d);
    }
    
    /**
     * Update all connection paths when nodes are moved
     */
    function updateConnections() {
        connections.forEach(conn => {
            // Get the current port positions
            const startNode = nodes.find(n => n.id === conn.start.nodeId);
            const endNode = nodes.find(n => n.id === conn.end.nodeId);
            
            if (startNode && endNode) {
                const startElement = startNode.element.querySelector(`.port[data-port-type="${conn.start.port}"]`);
                const endElement = endNode.element.querySelector(`.port[data-port-type="${conn.end.port}"]`);
                
                if (startElement && endElement) {
                    const startRect = startElement.getBoundingClientRect();
                    const endRect = endElement.getBoundingClientRect();
                    const canvasRect = canvas.getBoundingClientRect();
                    
                    const startX = (startRect.left + startRect.width / 2 - canvasRect.left - canvasOffset.x) / canvasScale;
                    const startY = (startRect.top + startRect.height / 2 - canvasRect.top - canvasOffset.y) / canvasScale;
                    const endX = (endRect.left + endRect.width / 2 - canvasRect.left - canvasOffset.x) / canvasScale;
                    const endY = (endRect.top + endRect.height / 2 - canvasRect.top - canvasOffset.y) / canvasScale;
                    
                    // Update path with improved bezier curve
                    const dx = endX - startX;
                    const dy = endY - startY;
                    const distance = Math.sqrt(dx * dx + dy * dy);
                    
                    // Use dynamic control point distances based on connection length
                    const bezierX = Math.min(Math.abs(dx) * 0.5, distance * 0.4);
                    const bezierY = Math.min(Math.abs(dy) * 0.2, distance * 0.1);
                    
                    // Create smoother S-curved path
                    const d = `M ${startX},${startY} 
                               C ${startX + bezierX},${startY + bezierY} 
                                 ${endX - bezierX},${endY - bezierY} 
                                 ${endX},${endY}`;
                    conn.element.setAttribute('d', d);
                    
                    // Update stored coordinates
                    conn.start.x = startX;
                    conn.start.y = startY;
                    conn.end.x = endX;
                    conn.end.y = endY;
                }
            }
        });
    }
    
    /**
     * Show the configuration panel for a node
     * @param {Object} nodeData - The node to configure
     */
    function showConfigPanel(nodeData) {
        const configPanel = document.getElementById('config-panel');
        const configTitle = document.getElementById('config-title');
        const nodeNameInput = document.getElementById('node-name');

        // Explicitly disable autocomplete on all input fields
        configPanel.querySelectorAll('input, textarea, select').forEach(el => {
            el.setAttribute('autocomplete', 'off');
        });

        // Hide all config sections first
        document.querySelectorAll('.agent-specific-config, .strategy-specific-config').forEach(section => {
            section.style.display = 'none';
        });

        // Set title and current node name
        if (nodeData.type === 'router') {
            configTitle.textContent = 'Configure Router';
            nodeNameInput.value = nodeData.config.name || 'Router';

            // Show router config section
            const routerConfig = document.getElementById('router-config');
            if (routerConfig) {
                routerConfig.style.display = 'block';

                // Set router strategy
                const strategySelect = document.getElementById('router-strategy');
                const strategy = nodeData.config.routingStrategy || 'keyword';
                strategySelect.value = strategy;

                // Show appropriate strategy-specific config
                showRoutingStrategyConfig(strategy, nodeData.config);

                // Set up change listener for strategy select
                strategySelect.onchange = function() {
                    showRoutingStrategyConfig(this.value, nodeData.config);
                };
            }
        } else if (nodeData.type === 'agent') {
            configTitle.textContent = 'Configure Agent';
            nodeNameInput.value = nodeData.config.name || '';

            const configSection = document.getElementById(`${nodeData.subType}-config`);
            if (configSection) {
                configSection.style.display = 'block';
            }

            // Fill in saved config values
            if (nodeData.subType === 'openai') {
                document.getElementById('openai-api-key').value = nodeData.config.apiKey || '';
                document.getElementById('openai-model').value = nodeData.config.model || 'gpt-4o';
                document.getElementById('openai-system-message').value = nodeData.config.systemMessage || 'You are a helpful AI assistant.';
            } else if (nodeData.subType === 'ollama') {
                document.getElementById('ollama-api-url').value = nodeData.config.apiUrl || '';
                document.getElementById('ollama-api-key').value = nodeData.config.apiKey || '';
                document.getElementById('ollama-model').value = nodeData.config.model || 'deepseek-r1:latest';
                document.getElementById('ollama-system-message').value = nodeData.config.systemMessage || 'You are a helpful AI assistant.';
            } else if (nodeData.subType === 'anthropic') {
                document.getElementById('anthropic-api-key').value = nodeData.config.apiKey || '';
                document.getElementById('anthropic-model').value = nodeData.config.model || 'claude-3-opus';
                document.getElementById('anthropic-system-message').value = nodeData.config.systemMessage || 'You are Claude, an AI assistant by Anthropic.';
            } else if (nodeData.subType === 'bedrock') {
                document.getElementById('aws-access-key').value = nodeData.config.accessKey || '';
                document.getElementById('aws-secret-key').value = nodeData.config.secretKey || '';
                document.getElementById('aws-region').value = nodeData.config.region || 'us-east-1';
                document.getElementById('bedrock-model').value = nodeData.config.model || 'anthropic.claude-3-sonnet-20240229-v1:0';
                document.getElementById('bedrock-system-message').value = nodeData.config.systemMessage || 'You are an AI assistant.';
            } else if (nodeData.subType === 'custom') {
                document.getElementById('agent-port').value = nodeData.config.port || '';
                document.getElementById('agent-endpoint').value = nodeData.config.endpoint || '';
                document.getElementById('agent-script').value = nodeData.config.script || '';
            }
        } else {
            configTitle.textContent = 'Configure Tool';
            nodeNameInput.value = nodeData.config.name || '';
            // Tool configuration will be implemented in the next phase
        }
        
        // Show the panel
        configPanel.classList.add('open');
    }
    
    /**
     * Hide the configuration panel
     */
    function hideConfigPanel() {
        const configPanel = document.getElementById('config-panel');
        configPanel.classList.remove('open');
    }

    /**
     * Show configuration options for the selected routing strategy
     * @param {string} strategy - The selected routing strategy
     * @param {Object} config - The current node configuration
     */
    function showRoutingStrategyConfig(strategy, config) {
        // Hide all strategy-specific configs first
        document.querySelectorAll('.strategy-specific-config').forEach(section => {
            section.style.display = 'none';
        });

        // Show the appropriate strategy config section
        switch (strategy) {
            case 'ai':
                const aiConfig = document.getElementById('ai-routing-config');
                if (aiConfig) {
                    aiConfig.style.display = 'block';

                    // Set saved values
                    const promptInput = document.getElementById('ai-router-prompt');
                    if (promptInput) {
                        promptInput.value = config.aiRouterPrompt || 'Examine the message content and route to the appropriate destination based on the output ports available.';
                    }
                }
                break;

            case 'keyword':
                const keywordConfig = document.getElementById('keyword-routing-config');
                if (keywordConfig) {
                    keywordConfig.style.display = 'block';

                    // Populate keyword patterns
                    const patternsContainer = document.getElementById('keyword-patterns-container');
                    if (patternsContainer) {
                        // Clear existing patterns
                        patternsContainer.innerHTML = '';

                        // Add saved patterns or create a default empty one
                        const patterns = config.keywordPatterns || [];
                        if (patterns.length > 0) {
                            patterns.forEach(pattern => {
                                // If this is a pattern saved with our newer format, use the original values
                                if (pattern.originalType && pattern.originalValue) {
                                    addKeywordPatternRow(patternsContainer, pattern.originalValue, pattern.port, pattern.originalType);
                                } else {
                                    // For backwards compatibility with older patterns, guess the pattern type
                                    let patternType = 'contains';
                                    const keyword = pattern.keyword;

                                    // Try to detect the pattern type
                                    if (keyword.startsWith('^') && keyword.endsWith('$')) {
                                        patternType = 'exactMatch';
                                    } else if (keyword.startsWith('^')) {
                                        patternType = 'startsWith';
                                    } else if (keyword.endsWith('$')) {
                                        patternType = 'endsWith';
                                    } else if (keyword.includes('*') || keyword.includes('?') || keyword.includes('[')) {
                                        patternType = 'regex';
                                    }

                                    addKeywordPatternRow(patternsContainer, keyword, pattern.port, patternType);
                                }
                            });
                        } else {
                            // Add default patterns for all output ports
                        let numPorts = 2; // Default minimum

                        // Get the actual number of output ports from the node
                        if (selectedNode && selectedNode.config && selectedNode.config.outputPorts) {
                            numPorts = Math.max(numPorts, selectedNode.config.outputPorts);
                        }

                        // Update the node to have this many output ports
                        if (selectedNode && selectedNode.config) {
                            selectedNode.config.outputPorts = numPorts;

                            // This is a critical function to ensure patterns match output ports
                            // We call this explicitly to ensure consistency
                            ensurePatternsMatchOutputPorts(selectedNode);
                        }

                        // Add a pattern for each output port
                        for (let i = 0; i < numPorts; i++) {
                            const patternType = i % 5 === 0 ? 'regex' :
                                              i % 5 === 1 ? 'contains' :
                                              i % 5 === 2 ? 'startsWith' :
                                              i % 5 === 3 ? 'endsWith' : 'exactMatch';

                            const pattern = i === 0 ? '.*' : // Default catch-all for first port
                                           patternType === 'regex' ? `.*pattern${i}.*` :
                                           patternType === 'contains' ? `keyword${i}` :
                                           patternType === 'startsWith' ? `start${i}` :
                                           patternType === 'endsWith' ? `end${i}` : `exact${i}`;

                            addKeywordPatternRow(patternsContainer, pattern, i, patternType);
                        }
                        }

                        // Set up "Add Pattern" button
                        const addPatternBtn = document.getElementById('add-keyword-pattern');
                        if (addPatternBtn) {
                            addPatternBtn.onclick = function() {
                                if (!selectedNode || selectedNode.type !== 'router') return;

                                // Get the current pattern count to determine the next port number
                                const currentPatterns = patternsContainer.querySelectorAll('.keyword-pattern-row');
                                const nextPortNumber = currentPatterns.length;

                                // Create a new output port
                                addNewOutputPort();

                                // Add the row with a random pattern type for variety
                                const patternTypes = ['contains', 'startsWith', 'endsWith', 'exactMatch', 'regex'];
                                const randomType = patternTypes[Math.floor(Math.random() * patternTypes.length)];
                                addKeywordPatternRow(patternsContainer, '', nextPortNumber, randomType);

                                // Ensure all dropdowns are up-to-date with correct port numbers
                                rebuildAllPortDropdowns();

                                // Add a nice animation effect
                                const newRow = patternsContainer.lastElementChild;
                                newRow.style.transform = 'translateY(10px)';
                                newRow.style.opacity = '0';

                                // Trigger animation
                                setTimeout(() => {
                                    newRow.style.transition = 'all 0.3s ease';
                                    newRow.style.transform = 'translateY(0)';
                                    newRow.style.opacity = '1';

                                    // Focus on the input field
                                    const input = newRow.querySelector('.pattern-input');
                                    if (input) input.focus();
                                }, 10);
                            };
                        }
                    }
                }
                break;

            case 'random':
                const randomConfig = document.getElementById('random-routing-config');
                if (randomConfig) {
                    randomConfig.style.display = 'block';

                    // Populate port weights
                    const weightsContainer = document.getElementById('port-weights-container');
                    if (weightsContainer) {
                        // Clear existing weights
                        weightsContainer.innerHTML = '';

                        // Add port weight rows
                        const weights = config.portWeights || {};
                        const outputPorts = config.outputPorts || 2;

                        for (let i = 0; i < outputPorts; i++) {
                            const weight = weights[i] || 1.0;
                            addPortWeightRow(weightsContainer, i, weight);
                        }
                    }
                }
                break;

            case 'content-type':
                const contentTypeConfig = document.getElementById('content-type-routing-config');
                if (contentTypeConfig) {
                    contentTypeConfig.style.display = 'block';

                    // Populate content type mappings
                    const mappingsContainer = document.getElementById('content-type-mappings-container');
                    if (mappingsContainer) {
                        // Clear existing mappings
                        mappingsContainer.innerHTML = '';

                        // Add saved mappings or create a default empty one
                        const mappings = config.contentTypeMappings || [];
                        if (mappings.length > 0) {
                            mappings.forEach(mapping => {
                                addContentTypeMappingRow(mappingsContainer, mapping.contentType, mapping.port);
                            });
                        } else {
                            addContentTypeMappingRow(mappingsContainer, 'text', 0);
                        }

                        // Set up "Add Mapping" button
                        const addMappingBtn = document.getElementById('add-content-mapping');
                        if (addMappingBtn) {
                            addMappingBtn.onclick = function() {
                                addContentTypeMappingRow(mappingsContainer, '', 0);
                            };
                        }
                    }
                }
                break;
        }
    }

    /**
     * Add a keyword pattern row to the router configuration
     * @param {HTMLElement} container - The container element
     * @param {string} keyword - The keyword to match
     * @param {number} port - The port to route to
     */
    /**
     * Returns an array of output ports that are already selected in existing patterns
     * so they can be disabled in new pattern dropdowns
     */
    function getSelectedPorts() {
        const selectedPorts = [];
        const patterns = document.querySelectorAll('.keyword-pattern-row');

        patterns.forEach(pattern => {
            const select = pattern.querySelector('select.port-select');
            if (select && select.value) {
                selectedPorts.push(parseInt(select.value));
            }
        });

        return selectedPorts;
    }

    /**
     * Refreshes all port dropdowns to correctly show disabled options
     */
    function refreshAllPortDropdowns() {
        const selectedPorts = getSelectedPorts();
        const allSelects = document.querySelectorAll('select.port-select');

        allSelects.forEach(select => {
            // Remember the current value
            const currentValue = select.value;

            // For each option, determine if it should be disabled
            Array.from(select.options).forEach(option => {
                const optionValue = parseInt(option.value);
                option.disabled = optionValue !== parseInt(currentValue) && selectedPorts.includes(optionValue);
            });
        });
    }

    /**
     * Add a new output port to the router node
     * @returns {number} The new port number
     */
    function addNewOutputPort() {
        if (!selectedNode || selectedNode.type !== 'router') return 0;

        // Get the current number of output ports
        let outputPorts = 2; // Default
        if (selectedNode.config && selectedNode.config.outputPorts) {
            outputPorts = selectedNode.config.outputPorts;
        }

        // Update the node data with the new port count
        selectedNode.config.outputPorts = outputPorts + 1;

        // Get the router node element
        const nodeElement = selectedNode.element;

        // Add a new output port visually
        if (nodeElement) {
            const outputPortsContainer = nodeElement.querySelector('.output-ports-container');
            if (outputPortsContainer) {
                const newPortNumber = outputPorts; // 0-based index
                const newPortElement = document.createElement('div');
                newPortElement.className = 'output-port';
                newPortElement.setAttribute('data-port-number', newPortNumber);
                newPortElement.innerHTML = `
                    <span class="port-label">Output ${newPortNumber + 1}</span>
                    <div class="port port-right" data-port-type="output" data-port-number="${newPortNumber}"></div>
                `;

                // Add the new port to the container
                outputPortsContainer.appendChild(newPortElement);

                // Initialize port event listeners
                initNodeEvents(nodeElement, selectedNode);

                // Update connections
                updateConnections();
            }
        }

        // Return the new port number (0-based)
        return outputPorts;
    }

    /**
     * Removes an output port from a router node
     * @param {number} portNumber - The 0-based port number to remove
     * @returns {boolean} - Whether the port was successfully removed
     */
    function removeOutputPort(portNumber) {
        if (!selectedNode || selectedNode.type !== 'router' || !selectedNode.config) return false;

        // Get the current number of output ports
        let outputPorts = 2; // Default minimum
        if (selectedNode.config.outputPorts) {
            outputPorts = selectedNode.config.outputPorts;
        }

        // Don't allow removing if we only have 2 ports (minimum required)
        if (outputPorts <= 2) {
            showNotification('warning', 'Router nodes require at least 2 output ports', 3000);
            return false;
        }

        // Get the router node element
        const nodeElement = selectedNode.element;
        if (!nodeElement) return false;

        // Find the output port element to remove
        const outputPortsContainer = nodeElement.querySelector('.output-ports-container');
        if (!outputPortsContainer) return false;

        // Find the port with the matching port number
        const portElement = outputPortsContainer.querySelector(`.output-port[data-port-number="${portNumber}"]`);
        if (!portElement) return false;

        // Remove the port element
        outputPortsContainer.removeChild(portElement);

        // Update port numbers for all ports that come after the removed one
        const laterPorts = outputPortsContainer.querySelectorAll(`.output-port[data-port-number]`);
        laterPorts.forEach(port => {
            const currentPortNumber = parseInt(port.getAttribute('data-port-number'));
            if (currentPortNumber > portNumber) {
                // Decrement the port number
                const newPortNumber = currentPortNumber - 1;
                port.setAttribute('data-port-number', newPortNumber);

                // Update the label and port element
                const label = port.querySelector('.port-label');
                if (label) {
                    label.textContent = `Output ${newPortNumber + 1}`;
                }

                const portDiv = port.querySelector('.port');
                if (portDiv) {
                    portDiv.setAttribute('data-port-number', newPortNumber);
                }
            }
        });

        // Update the node data with the new port count
        selectedNode.config.outputPorts = outputPorts - 1;

        // Also update pattern-port mapping in the config
        if (selectedNode.config && selectedNode.config.keywordPatterns) {
            // Remove the pattern that points to this port
            const patternIndex = selectedNode.config.keywordPatterns.findIndex(p => p.port === portNumber);
            if (patternIndex !== -1) {
                selectedNode.config.keywordPatterns.splice(patternIndex, 1);
            }

            // Update port numbers for patterns that point to higher port numbers
            selectedNode.config.keywordPatterns.forEach(pattern => {
                if (pattern.port > portNumber) {
                    pattern.port -= 1;
                }
            });
        }

        // Remove any connections to this port
        if (connections) {
            const connectionsToRemove = [];
            connections.forEach(connection => {
                if (connection.sourceNode === selectedNode.id && connection.sourcePort === portNumber) {
                    connectionsToRemove.push(connection);
                }
            });

            // Remove the connections
            connectionsToRemove.forEach(connection => {
                const index = connections.indexOf(connection);
                if (index !== -1) {
                    connections.splice(index, 1);
                }

                // Remove the SVG path if it exists
                if (connection.path) {
                    connection.path.remove();
                }
            });

            // Update connection port numbers for connections after the removed port
            connections.forEach(connection => {
                if (connection.sourceNode === selectedNode.id && connection.sourcePort > portNumber) {
                    connection.sourcePort--;
                }
            });
        }

        return true;
    }

    /**
     * Synchronizes the UI patterns with the node configuration
     * @param {HTMLElement} container - The patterns container
     * @returns {Array} The updated patterns array
     */
    function syncPatternsFromUI(container) {
        if (!container || !selectedNode || !selectedNode.config) return [];

        // Rebuild the patterns array from the UI
        const patterns = [];
        const patternRows = container.querySelectorAll('.keyword-pattern-row');

        // First, update all the ports to match their row index for consistency
        patternRows.forEach((patternRow, rowIndex) => {
            const portSelect = patternRow.querySelector('.port-select');
            const selectedOption = patternRow.querySelector('.selected-option span');

            if (portSelect) {
                // Always assign consecutive port numbers based on row index
                portSelect.value = rowIndex;

                // Update the visual display too
                if (selectedOption) {
                    selectedOption.textContent = `Output ${rowIndex + 1}`;
                }
            }
        });

        // Now build the patterns array with the updated port numbers
        patternRows.forEach((patternRow, rowIndex) => {
            const patternInput = patternRow.querySelector('.pattern-input');
            const portSelect = patternRow.querySelector('.port-select');
            const patternType = patternRow.getAttribute('data-pattern-type') || 'contains';

            if (patternInput && portSelect) {
                patterns.push({
                    keyword: patternInput.value,
                    // Use the row index directly to ensure consistent numbering
                    port: rowIndex,
                    originalType: patternType,
                    originalValue: patternInput.value
                });
            }
        });

        // Update the node's configuration
        selectedNode.config.keywordPatterns = patterns;

        // Update the output port count to match the pattern count
        if (patterns.length > 0) {
            selectedNode.config.outputPorts = Math.max(2, patterns.length);

            // Update the visual output ports in the node
            updateRouterOutputPorts(selectedNode, patterns.length);
        }

        return patterns;
    }

    /**
     * Updates the router node's output ports to match the given count
     * @param {Object} node - The node object to update
     * @param {number} count - The new number of output ports
     */
    function updateRouterOutputPorts(node, count) {
        if (!node || node.type !== 'router' || !node.element) return;

        // Get current output ports
        const nodeElement = node.element;
        const outputPortsContainer = nodeElement.querySelector('.output-ports-container');
        if (!outputPortsContainer) return;

        // Get current port count
        const currentPorts = outputPortsContainer.querySelectorAll('.output-port').length;

        // Make sure we have at least 2 ports
        count = Math.max(2, count);

        // If we already have the right number, do nothing
        if (currentPorts === count) return;

        // If we need to add ports
        if (currentPorts < count) {
            for (let i = currentPorts; i < count; i++) {
                const newPortElement = document.createElement('div');
                newPortElement.className = 'output-port';
                newPortElement.setAttribute('data-port-number', i);
                newPortElement.innerHTML = `
                    <span class="port-label">Output ${i + 1}</span>
                    <div class="port port-right" data-port-type="output" data-port-number="${i}"></div>
                `;
                outputPortsContainer.appendChild(newPortElement);
            }
        }
        // If we need to remove ports
        else if (currentPorts > count) {
            // Remove excess ports starting from the highest port number
            for (let i = currentPorts - 1; i >= count; i--) {
                const portElement = outputPortsContainer.querySelector(`.output-port[data-port-number="${i}"]`);
                if (portElement) {
                    outputPortsContainer.removeChild(portElement);
                }
            }
        }

        // Update the node's config
        node.config.outputPorts = count;
    }

    /**
     * Completely rebuilds all port dropdown selects and their custom UI
     * This ensures port numbers are correct and consecutive
     */
    function rebuildAllPortDropdowns() {
        // Only proceed if we have a selected node
        if (!selectedNode || selectedNode.type !== 'router') return;

        // Get all pattern rows
        const patternRows = document.querySelectorAll('.keyword-pattern-row');

        // Important: Use pattern count instead of outputPorts
        // This ensures we only show options for the actual patterns we have
        const patternCount = patternRows.length;

        // Remove any open custom dropdowns to avoid stale options
        document.querySelectorAll('.port-select-dropdown').forEach(dropdown => {
            dropdown.remove();
        });

        patternRows.forEach((row, rowIndex) => {
            // Get the port select element
            const selectElement = row.querySelector('.port-select');
            if (!selectElement) return;

            // Get the custom selected option element
            const selectedOptionElement = row.querySelector('.selected-option');
            if (!selectedOptionElement) return;

            // Clear and rebuild the select options
            selectElement.innerHTML = '';

            // Only show options for existing patterns
            // This is the key change to fix deleted options still showing
            for (let i = 0; i < patternCount; i++) {
                const option = document.createElement('option');
                option.value = i;
                option.textContent = `Output ${i + 1}`;

                // Only disable if this port is already selected in another row
                const isDisabled = i !== rowIndex && isPortSelectedInAnotherRow(i, rowIndex);
                if (isDisabled) {
                    option.disabled = true;
                }

                selectElement.appendChild(option);
            }

            // Always set the select value to match its row index
            // This ensures consistent port numbering
            selectElement.value = rowIndex;

            // Update the custom UI to show the selected port
            if (selectedOptionElement.querySelector('span')) {
                selectedOptionElement.querySelector('span').textContent = `Output ${rowIndex + 1}`;
            }
        });
    }

    /**
     * Checks if a port is already selected in another pattern row
     * @param {number} port - The port number to check
     * @param {number} currentRowIndex - The index of the current row to exclude from check
     * @returns {boolean} - Whether the port is selected elsewhere
     */
    function isPortSelectedInAnotherRow(port, currentRowIndex) {
        const patternRows = document.querySelectorAll('.keyword-pattern-row');

        for (let i = 0; i < patternRows.length; i++) {
            // Skip checking the current row
            if (i === currentRowIndex) continue;

            const select = patternRows[i].querySelector('.port-select');
            if (select && parseInt(select.value) === port) {
                return true; // This port is used in another row
            }
        }

        return false; // Port is not used in any other row
    }

    /**
     * Ensures that the number of patterns in a router node matches the number of output ports
     * This is critical for maintaining consistent behavior
     */
    function ensurePatternsMatchOutputPorts(node) {
        if (!node || node.type !== 'router' || !node.config) return;

        // Get the current number of output ports
        const outputPorts = Math.max(2, node.config.outputPorts || 2);

        // Get or create the patterns array if it doesn't exist
        if (!node.config.keywordPatterns) {
            node.config.keywordPatterns = [];
        }

        // Ensure we have exactly one pattern per output port
        if (node.config.keywordPatterns.length !== outputPorts) {
            // If we have too few patterns, add new ones
            if (node.config.keywordPatterns.length < outputPorts) {
                for (let i = node.config.keywordPatterns.length; i < outputPorts; i++) {
                    // Create a pattern for each missing port
                    const patternType = i % 5 === 0 ? 'regex' :
                                        i % 5 === 1 ? 'contains' :
                                        i % 5 === 2 ? 'startsWith' :
                                        i % 5 === 3 ? 'endsWith' : 'exactMatch';

                    const pattern = i === 0 ? '.*' : // Default catch-all for first port
                                    patternType === 'regex' ? `.*pattern${i}.*` :
                                    patternType === 'contains' ? `keyword${i}` :
                                    patternType === 'startsWith' ? `start${i}` :
                                    patternType === 'endsWith' ? `end${i}` : `exact${i}`;

                    node.config.keywordPatterns.push({
                        keyword: pattern,
                        port: i,
                        // Adding new format properties
                        originalType: patternType,
                        originalValue: pattern
                    });
                }
            }
            // If we have too many patterns, remove the excess
            else if (node.config.keywordPatterns.length > outputPorts) {
                node.config.keywordPatterns = node.config.keywordPatterns.slice(0, outputPorts);
            }

            // Update the node's output port count to match
            node.config.outputPorts = outputPorts;
        }
    }

    /**
     * Creates a keyword pattern row with pattern type selector
     */
    function addKeywordPatternRow(container, keyword, port, patternType = 'contains') {
        // Check if this is the first pattern and hide the empty state if needed
        const emptyState = document.getElementById('empty-patterns-placeholder');
        if (emptyState && emptyState.style.display !== 'none') {
            emptyState.style.display = 'none';
        }

        // If adding a new pattern with ADD NEW PATTERN button, create a new output port
        if (!keyword && port === 0 && container.querySelectorAll('.keyword-pattern-row').length > 0) {
            port = addNewOutputPort();
        }

        const row = document.createElement('div');
        row.className = 'keyword-pattern-row';
        row.setAttribute('data-pattern-type', patternType);

        // Create unique ID for the row and elements
        const rowId = Date.now() + Math.random().toString(36).substring(2, 9);
        const selectId = `port-select-${rowId}`;
        const inputId = `pattern-input-${rowId}`;
        const typeSelectId = `pattern-type-${rowId}`;

        // Get number of output ports for select options - ensure at least 2
        let outputPorts = 2; // Default
        if (selectedNode && selectedNode.config && selectedNode.config.outputPorts) {
            outputPorts = Math.max(2, selectedNode.config.outputPorts);

            // Update the node config if needed to ensure at least 2 ports
            if (selectedNode.config.outputPorts < 2) {
                selectedNode.config.outputPorts = 2;
            }
        }

        // Get already selected ports to disable them in the dropdown
        const selectedPorts = getSelectedPorts();

        // Create initial selected port text
        const selectedPortText = `Output ${port + 1}`;

        // Set placeholder based on pattern type
        let placeholderText = "";
        switch (patternType) {
            case 'contains':
                placeholderText = "e.g., weather";
                break;
            case 'startsWith':
                placeholderText = "e.g., flight";
                break;
            case 'endsWith':
                placeholderText = "e.g., info";
                break;
            case 'exactMatch':
                placeholderText = "e.g., help me";
                break;
            case 'regex':
                placeholderText = "e.g., .*question.*";
                break;
            default:
                placeholderText = "e.g., weather";
        }

        // If a keyword was passed, use it as value
        const keywordValue = keyword || '';

        row.innerHTML = `
            <div class="pattern-fields-container">
                <select id="${typeSelectId}" class="pattern-type-select" style="height: 40px;">
                    <option value="contains" ${patternType === 'contains' ? 'selected' : ''}>Contains</option>
                    <option value="startsWith" ${patternType === 'startsWith' ? 'selected' : ''}>Starts with</option>
                    <option value="endsWith" ${patternType === 'endsWith' ? 'selected' : ''}>Ends with</option>
                    <option value="exactMatch" ${patternType === 'exactMatch' ? 'selected' : ''}>Exact match</option>
                    <option value="regex" ${patternType === 'regex' ? 'selected' : ''}>Regex</option>
                </select>
                <input id="${inputId}" type="text" class="pattern-input" placeholder="${placeholderText}" value="${keywordValue}" style="height: 40px;">
            </div>
            <div class="custom-select" style="height: 40px;">
                <select id="${selectId}" class="port-select">
                    ${Array(outputPorts).fill().map((_, i) => {
                        const isDisabled = i !== port && selectedPorts.includes(i);
                        return `<option value="${i}" ${port === i ? 'selected' : ''} ${isDisabled ? 'disabled' : ''}>Output ${i+1}</option>`;
                    }).join('')}
                </select>
                <div class="selected-option" data-select-id="${selectId}" style="height: 40px;">
                    <span>${selectedPortText}</span>
                    <i class="bi bi-chevron-down"></i>
                </div>
            </div>
            <button class="remove-pattern-btn" data-row-id="${rowId}" title="Remove Pattern"><i class="bi bi-x-lg"></i></button>
        `;

        // Add row to container
        container.appendChild(row);

        // Update pattern count
        updatePatternCount();

        // Set up pattern type change
        const typeSelect = row.querySelector(`#${typeSelectId}`);
        if (typeSelect) {
            typeSelect.addEventListener('change', function() {
                const newType = this.value;
                row.setAttribute('data-pattern-type', newType);

                // Update the placeholder based on type
                const input = row.querySelector('.pattern-input');
                if (input) {
                    switch (newType) {
                        case 'contains':
                            input.placeholder = "e.g., weather";
                            break;
                        case 'startsWith':
                            input.placeholder = "e.g., flight";
                            break;
                        case 'endsWith':
                            input.placeholder = "e.g., info";
                            break;
                        case 'exactMatch':
                            input.placeholder = "e.g., help me";
                            break;
                        case 'regex':
                            input.placeholder = "e.g., .*question.*";
                            break;
                    }
                }
            });
        }

        // Set up remove button with simplified functionality that directly manages the UI
        const removeBtn = row.querySelector('.remove-pattern-btn');
        if (removeBtn) {
            removeBtn.onclick = function() {
                // Simple direct approach for removing patterns
                // Get the current port number this pattern is routing to
                const portSelect = row.querySelector('.port-select');
                const portNumber = portSelect ? parseInt(portSelect.value) : -1;

                // Only remove if there's more than 1 pattern (always keep at least one pattern)
                if (container.querySelectorAll('.keyword-pattern-row').length > 1) {
                    // Remove pattern row without checking port number constraints
                    // Remember port number for later
                    const portToRemove = portNumber;

                    // Always remove the pattern row
                    container.removeChild(row);
                    updatePatternCount();

                    // Always update the configuration
                    if (selectedNode && selectedNode.config) {
                        // Build patterns from remaining UI rows
                        const patterns = [];
                        container.querySelectorAll('.keyword-pattern-row').forEach(patternRow => {
                            const input = patternRow.querySelector('.pattern-input');
                            const select = patternRow.querySelector('.port-select');
                            const type = patternRow.getAttribute('data-pattern-type') || 'contains';

                            if (input && select) {
                                patterns.push({
                                    keyword: input.value,
                                    port: parseInt(select.value),
                                    originalType: type,
                                    originalValue: input.value
                                });
                            }
                        });

                        // Update config and output port count
                        selectedNode.config.keywordPatterns = patterns;
                        selectedNode.config.outputPorts = Math.max(2, patterns.length);

                        // Find and remove the associated port from the UI
                        const nodeElement = selectedNode.element;
                        if (nodeElement && portToRemove !== -1) {
                            const portsContainer = nodeElement.querySelector('.output-ports-container');
                            if (portsContainer) {
                                // Try to find and remove the port
                                const portElement = portsContainer.querySelector(`.output-port[data-port-number="${portToRemove}"]`);
                                if (portElement) {
                                    portsContainer.removeChild(portElement);
                                }
                            }
                        }
                    }

                    // Update connections
                    if (window.connections && portToRemove !== -1) {
                        // Remove connections to this port
                        window.connections = window.connections.filter(conn => {
                            if (conn.sourceNode === selectedNode.id && conn.sourcePort === portToRemove) {
                                // Remove the visual connection
                                if (conn.path) {
                                    conn.path.remove();
                                }
                                return false; // remove this connection
                            }
                            return true; // keep other connections
                        });
                    }

                    // Fix port numbering in the node UI
                    if (selectedNode && selectedNode.element) {
                        const nodeElement = selectedNode.element;
                        const portsContainer = nodeElement.querySelector('.output-ports-container');
                        if (portsContainer) {
                            // Update port numbers on all port elements to be consecutive
                            const allPorts = Array.from(portsContainer.querySelectorAll('.output-port'));

                            // Sort by current port number
                            allPorts.sort((a, b) => {
                                const aNum = parseInt(a.getAttribute('data-port-number'));
                                const bNum = parseInt(b.getAttribute('data-port-number'));
                                return aNum - bNum;
                            });

                            // Renumber them consecutively
                            allPorts.forEach((port, index) => {
                                // Update port attribute
                                port.setAttribute('data-port-number', index);

                                // Update port label
                                const label = port.querySelector('.port-label');
                                if (label) {
                                    label.textContent = `Output ${index + 1}`;
                                }

                                // Update port dot
                                const portDot = port.querySelector('.port[data-port-type="output"]');
                                if (portDot) {
                                    portDot.setAttribute('data-port-number', index);
                                }
                            });
                        }
                    }

                    // Update patterns in the config to match the renumbered ports
                    if (selectedNode && selectedNode.config && selectedNode.config.keywordPatterns) {
                        // Get the current pattern rows
                        const patternRows = container.querySelectorAll('.keyword-pattern-row');

                        // When removing patterns and ports, we need to renumber the select options
                        // in all pattern rows to ensure consecutive port numbers
                        patternRows.forEach((patternRow, rowIndex) => {
                            const select = patternRow.querySelector('.port-select');
                            if (select) {
                                // Get the current selected option text (e.g., "Output 3")
                                const value = parseInt(select.value);

                                // Update the value to match its row index
                                select.value = rowIndex;

                                // Update the corresponding pattern in the config
                                if (selectedNode.config.keywordPatterns[rowIndex]) {
                                    selectedNode.config.keywordPatterns[rowIndex].port = rowIndex;
                                }
                            }
                        });
                    }

                    // Rebuild the dropdowns completely to ensure correct options
                    rebuildAllPortDropdowns();

                    // Update connections to match the new port numbering
                    if (window.connections) {
                        // Adjust connections to the new port numbers
                        window.connections.forEach(conn => {
                            if (conn.sourceNode === selectedNode.id) {
                                // Find the actual port number from the node's UI
                                const nodeElement = selectedNode.element;
                                if (nodeElement) {
                                    const portElements = nodeElement.querySelectorAll('.port[data-port-type="output"]');
                                    // Set the source port to match the actual port number
                                    const portCount = portElements.length;
                                    if (conn.sourcePort >= portCount) {
                                        conn.sourcePort = portCount - 1;
                                    }
                                }
                            }
                        });
                    }

                    // Update connections after renumbering
                    if (typeof updateConnections === 'function') {
                        updateConnections();
                    }
                } else {
                    // Display a small notification that we need at least one pattern
                    showNotification('warning', 'Router nodes require at least one pattern', 3000);
                }
            };
        }

        // Set up custom select handler
        const selectElement = row.querySelector(`#${selectId}`);
        const selectedOption = row.querySelector('.selected-option');
        if (selectElement && selectedOption) {
            // Create custom dropdown when clicking on the selected option
            selectedOption.addEventListener('click', function(e) {
                e.stopPropagation();

                // Remove any existing dropdowns
                document.querySelectorAll('.port-select-dropdown').forEach(dropdown => {
                    dropdown.remove();
                });

                // Create dropdown
                const dropdown = document.createElement('div');
                dropdown.className = 'port-select-dropdown';

                // Position the dropdown with offset to account for the scrolling and margins
                const rect = selectedOption.getBoundingClientRect();
                dropdown.style.top = (rect.top + rect.height + 4) + 'px';
                dropdown.style.left = rect.left + 'px';
                dropdown.style.width = 'auto';        // Allow auto width for better content display
                dropdown.style.minWidth = '200px';    // Ensure minimum width
                dropdown.style.maxWidth = '250px';    // Cap the maximum width
                dropdown.style.maxHeight = '280px';   // Taller dropdown for more content
                dropdown.style.overflowY = 'auto';    // Scrollable if needed

                // Add dropdown header
                const header = document.createElement('div');
                header.className = 'port-select-dropdown-header';
                header.textContent = 'Select Output';
                dropdown.appendChild(header);

                // Get the number of pattern rows (this is the key change - using pattern count instead of output ports)
                const patternRows = document.querySelectorAll('.keyword-pattern-row');
                const patternCount = patternRows.length;

                // This ensures we only show options for actual patterns that exist
                let effectiveOutputCount = patternCount;

                // Ensure we update the selectElement if needed
                if (effectiveOutputCount > outputPorts) {
                    // Add new options to the select
                    for (let i = outputPorts; i < effectiveOutputCount; i++) {
                        const newOption = document.createElement('option');
                        newOption.value = i;
                        newOption.textContent = `Output ${i+1}`;
                        selectElement.appendChild(newOption);
                    }
                    outputPorts = effectiveOutputCount;
                }

                // Add options - now using the pattern count instead of output ports
                for (let i = 0; i < patternCount; i++) {
                    const option = document.createElement('div');
                    const isDisabled = i !== parseInt(selectElement.value) && selectedPorts.includes(i);
                    option.className = `port-select-option ${isDisabled ? 'disabled' : ''}`;
                    if (selectElement.value == i) {
                        option.classList.add('selected');
                    }

                    option.innerHTML = `
                        <i class="bi bi-check-circle-fill"></i>
                        <div class="port-select-option-label">
                            <span class="port-name">Output ${i+1}</span>
                            <span class="port-desc">Messages matching this pattern will be routed to output port ${i+1}</span>
                        </div>
                    `;

                    if (!isDisabled) {
                        option.addEventListener('click', function() {
                            selectElement.value = i;
                            selectedOption.querySelector('span').textContent = `Output ${i+1}`;
                            dropdown.remove();

                            // Refresh all dropdowns to update disabled states
                            refreshAllPortDropdowns();

                            // Dispatch change event
                            const event = new Event('change');
                            selectElement.dispatchEvent(event);
                        });
                    }

                    dropdown.appendChild(option);
                }

                // Add dropdown to document
                document.body.appendChild(dropdown);

                // Close dropdown when clicking elsewhere
                const closeDropdown = function() {
                    dropdown.remove();
                    document.removeEventListener('click', closeDropdown);
                };

                setTimeout(() => {
                    document.addEventListener('click', closeDropdown);
                }, 0);
            });

            // Update the visual state when the underlying select changes
            selectElement.addEventListener('change', function() {
                const value = parseInt(this.value);
                selectedOption.querySelector('span').textContent = `Output ${value+1}`;
            });
        }
    }

    /**
     * Update the pattern count display in the keyword patterns header
     */
    function updatePatternCount() {
        const container = document.getElementById('keyword-patterns-container');
        const countElement = document.querySelector('.pattern-count');

        if (container && countElement) {
            const count = container.querySelectorAll('.keyword-pattern-row').length;
            countElement.textContent = count === 1 ? '1 pattern' : `${count} patterns`;
        }
    }

    /**
     * Add a port weight row to the router configuration
     * @param {HTMLElement} container - The container element
     * @param {number} portIndex - The port index
     * @param {number} weight - The port weight
     */
    function addPortWeightRow(container, portIndex, weight) {
        const row = document.createElement('div');
        row.className = 'weight-row';

        row.innerHTML = `
            <label>Output ${portIndex + 1} Weight:</label>
            <input type="number" class="form-control weight-input" min="0" step="0.1" value="${weight}" data-port="${portIndex}">
        `;

        container.appendChild(row);
    }

    /**
     * Add a content type mapping row to the router configuration
     * @param {HTMLElement} container - The container element
     * @param {string} contentType - The content type to match
     * @param {number} port - The port to route to
     */
    function addContentTypeMappingRow(container, contentType, port) {
        const row = document.createElement('div');
        row.className = 'mapping-row';

        // Create unique ID for the remove button
        const rowId = Date.now() + Math.random().toString(36).substring(2, 9);

        // Get number of output ports for select options
        let outputPorts = 2; // Default
        if (selectedNode && selectedNode.config && selectedNode.config.outputPorts) {
            outputPorts = selectedNode.config.outputPorts;
        }

        // Create port options
        let portOptions = '';
        for (let i = 0; i < outputPorts; i++) {
            portOptions += `<option value="${i}" ${port === i ? 'selected' : ''}>Output ${i+1}</option>`;
        }

        row.innerHTML = `
            <input type="text" class="form-control content-type-input" placeholder="Content type (e.g., text, json)" value="${contentType || ''}">
            <select class="form-control port-select">
                ${portOptions}
            </select>
            <button class="remove-mapping-btn" data-row-id="${rowId}"><i class="bi bi-x"></i></button>
        `;

        // Add row to container
        container.appendChild(row);

        // Set up remove button
        const removeBtn = row.querySelector('.remove-mapping-btn');
        if (removeBtn) {
            removeBtn.onclick = function() {
                container.removeChild(row);
            };
        }
    }
    
    /**
     * Apply node configuration from the panel
     */
    function applyNodeConfig() {
        if (!selectedNode) return;
        
        // Get common config values
        const nodeName = document.getElementById('node-name').value;
        
        // Validate required fields
        let validationErrors = [];
        
        // Require a name for all nodes
        if (!nodeName || nodeName.trim() === '') {
            validationErrors.push('Node name is required');
        }
        
        // Get and validate type-specific config
        if (selectedNode.type === 'agent') {
            if (selectedNode.subType === 'openai') {
                const apiKey = document.getElementById('openai-api-key').value;
                const model = document.getElementById('openai-model').value;
                const systemMessage = document.getElementById('openai-system-message').value;
                
                if (!apiKey || apiKey.trim() === '') {
                    validationErrors.push('OpenAI API Key is required');
                }
                
                if (!model) {
                    validationErrors.push('Please select a model');
                }
                
                // Store values if validation passes
                if (validationErrors.length === 0) {
                    selectedNode.config.apiKey = apiKey;
                    selectedNode.config.model = model;
                    selectedNode.config.systemMessage = systemMessage;
                }
            } else if (selectedNode.subType === 'ollama') {
                const apiUrl = document.getElementById('ollama-api-url').value;
                const apiKey = document.getElementById('ollama-api-key').value;
                const model = document.getElementById('ollama-model').value;
                const systemMessage = document.getElementById('ollama-system-message').value;
                
                if (!apiUrl || apiUrl.trim() === '') {
                    validationErrors.push('Ollama API url is required');
                }

                if (!apiKey || apiKey.trim() === '') {
                    validationErrors.push('Ollama API Key is required');
                }
                
                if (!model) {
                    validationErrors.push('Please define a model');
                }
                
                // Store values if validation passes
                if (validationErrors.length === 0) {
                    selectedNode.config.apiUrl = apiUrl;
                    selectedNode.config.apiKey = apiKey;
                    selectedNode.config.model = model;
                    selectedNode.config.systemMessage = systemMessage;
                }
            } else if (selectedNode.subType === 'anthropic') {
                const apiKey = document.getElementById('anthropic-api-key').value;
                const model = document.getElementById('anthropic-model').value;
                const systemMessage = document.getElementById('anthropic-system-message').value;
                
                if (!apiKey || apiKey.trim() === '') {
                    validationErrors.push('Anthropic API Key is required');
                }
                
                if (!model) {
                    validationErrors.push('Please select a model');
                }
                
                // Store values if validation passes
                if (validationErrors.length === 0) {
                    selectedNode.config.apiKey = apiKey;
                    selectedNode.config.model = model;
                    selectedNode.config.systemMessage = systemMessage;
                }
            } else if (selectedNode.subType === 'bedrock') {
                const accessKey = document.getElementById('aws-access-key').value;
                const secretKey = document.getElementById('aws-secret-key').value;
                const region = document.getElementById('aws-region').value;
                const model = document.getElementById('bedrock-model').value;
                const systemMessage = document.getElementById('bedrock-system-message').value;
                
                if (!accessKey || accessKey.trim() === '') {
                    validationErrors.push('AWS Access Key is required');
                }
                
                if (!secretKey || secretKey.trim() === '') {
                    validationErrors.push('AWS Secret Key is required');
                }
                
                if (!region) {
                    validationErrors.push('Please select an AWS region');
                }
                
                if (!model) {
                    validationErrors.push('Please select a model');
                }
                
                // Store values if validation passes
                if (validationErrors.length === 0) {
                    selectedNode.config.accessKey = accessKey;
                    selectedNode.config.secretKey = secretKey;
                    selectedNode.config.region = region;
                    selectedNode.config.model = model;
                    selectedNode.config.systemMessage = systemMessage;
                }
            } else if (selectedNode.subType === 'custom') {
                const port = document.getElementById('agent-port').value;
                const endpoint = document.getElementById('agent-endpoint').value;
                const script = document.getElementById('agent-script').value;
                
                if ((!port || port === '') && (!endpoint || endpoint.trim() === '')) {
                    validationErrors.push('Either Port or Endpoint URL is required');
                }
                
                // Store values if validation passes
                if (validationErrors.length === 0) {
                    selectedNode.config.port = port;
                    selectedNode.config.endpoint = endpoint;
                    selectedNode.config.script = script;
                }
            }
        } else {
            // Tool-specific validations will be implemented in the next phase
        }
        
        // Handle router configuration if it's a router node
        if (selectedNode.type === 'router') {
            const routerStrategy = document.getElementById('router-strategy').value;

            // Collect router-specific config based on strategy
            selectedNode.config.routingStrategy = routerStrategy;

            switch (routerStrategy) {
                case 'ai':
                    const aiPrompt = document.getElementById('ai-router-prompt').value;
                    if (!aiPrompt || aiPrompt.trim() === '') {
                        validationErrors.push('AI Router prompt is required');
                    } else {
                        selectedNode.config.aiRouterPrompt = aiPrompt;
                    }
                    break;

                case 'keyword':
                    const patterns = [];
                    document.querySelectorAll('.keyword-pattern-row').forEach(row => {
                        const keywordInput = row.querySelector('.pattern-input');
                        const portSelect = row.querySelector('.port-select');
                        const patternType = row.getAttribute('data-pattern-type') || 'contains';

                        if (keywordInput && portSelect && keywordInput.value.trim() !== '') {
                            // Get the raw keyword value
                            let keywordValue = keywordInput.value.trim();

                            // Convert the pattern based on its type
                            let pattern = keywordValue;

                            // Only auto-convert if not using regex mode
                            if (patternType !== 'regex') {
                                // Escape special regex characters to prevent injection
                                const escapeRegex = (string) => {
                                    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                                };

                                // Convert based on pattern type
                                switch (patternType) {
                                    case 'contains':
                                        pattern = escapeRegex(keywordValue);
                                        break;
                                    case 'startsWith':
                                        pattern = '^' + escapeRegex(keywordValue);
                                        break;
                                    case 'endsWith':
                                        pattern = escapeRegex(keywordValue) + '$';
                                        break;
                                    case 'exactMatch':
                                        pattern = '^' + escapeRegex(keywordValue) + '$';
                                        break;
                                }
                            }

                            // Add the pattern to the list
                            patterns.push({
                                keyword: pattern,
                                port: parseInt(portSelect.value, 10),
                                originalType: patternType,
                                originalValue: keywordValue
                            });
                        }
                    });

                    if (patterns.length === 0) {
                        validationErrors.push('At least one keyword pattern is required');
                    } else {
                        selectedNode.config.keywordPatterns = patterns;

                        // Also update the number of output ports to match the pattern count
                        // This ensures the router always has the right number of ports
                        const patternCount = patterns.length;
                        selectedNode.config.outputPorts = Math.max(2, patternCount);

                        // Update the visual output ports in the node
                        updateRouterOutputPorts(selectedNode, patternCount);
                    }
                    break;

                case 'random':
                    const weights = {};
                    document.querySelectorAll('.weight-input').forEach(input => {
                        const port = parseInt(input.getAttribute('data-port'), 10);
                        const weight = parseFloat(input.value) || 0;
                        weights[port] = weight;
                    });

                    selectedNode.config.portWeights = weights;
                    break;

                case 'content-type':
                    const mappings = [];
                    document.querySelectorAll('.mapping-row').forEach(row => {
                        const contentTypeInput = row.querySelector('.content-type-input');
                        const portSelect = row.querySelector('.port-select');

                        if (contentTypeInput && portSelect && contentTypeInput.value.trim() !== '') {
                            mappings.push({
                                contentType: contentTypeInput.value,
                                port: parseInt(portSelect.value, 10)
                            });
                        }
                    });

                    if (mappings.length === 0) {
                        validationErrors.push('At least one content type mapping is required');
                    } else {
                        selectedNode.config.contentTypeMappings = mappings;
                    }
                    break;
            }
        }

        // If validation fails, show errors and return
        if (validationErrors.length > 0) {
            validationErrors.forEach(error => {
                showNotification(error, 'error');
            });
            return;
        }

        // Apply common config
        selectedNode.config.name = nodeName;

        // Update node display with config info
        if (selectedNode.type === 'router') {
            // Update the strategy badge on the router node
            const nodeElement = document.getElementById(selectedNode.id);
            if (nodeElement) {
                const strategyBadge = nodeElement.querySelector('.router-strategy');
                if (strategyBadge) {
                    const strategyLabels = {
                        'ai': 'AI Routing',
                        'keyword': 'Keyword Routing',
                        'random': 'Random Routing',
                        'content-type': 'Content Type Routing'
                    };
                    strategyBadge.textContent = strategyLabels[selectedNode.config.routingStrategy] || 'Not configured';
                }
            }
        }
        if (selectedNode.type === 'agent') {
            const nodeNameElement = selectedNode.element.querySelector('.node-title span');
            if (nodeNameElement && selectedNode.config.name) {
                nodeNameElement.textContent = selectedNode.config.name;
            }
            
            // Update model badge
            const modelBadge = selectedNode.element.querySelector('.agent-model');
            if (modelBadge) {
                let modelText = 'Not configured';
                
                if (selectedNode.subType === 'openai' && selectedNode.config.model) {
                    modelText = selectedNode.config.model;
                } else if (selectedNode.subType === 'ollama' && selectedNode.config.model) {
                    modelText = selectedNode.config.model;
                } else if (selectedNode.subType === 'anthropic' && selectedNode.config.model) {
                    modelText = selectedNode.config.model;
                } else if (selectedNode.subType === 'bedrock' && selectedNode.config.model) {
                    modelText = selectedNode.config.model.split('.')[1]?.split('-')[0] || selectedNode.config.model;
                } else if (selectedNode.subType === 'custom' && selectedNode.config.port) {
                    modelText = `Port: ${selectedNode.config.port}`;
                } else if (selectedNode.subType === 'custom' && selectedNode.config.endpoint) {
                    modelText = `URL: ${selectedNode.config.endpoint.substring(0, 15)}...`;
                }
                
                modelBadge.textContent = modelText;
            }
            
            // Show success notification
            showNotification(`${selectedNode.config.name} configuration updated`, 'success');

            // Update all connections after router configuration changes
            if (selectedNode.type === 'router') {
                // Ensure connections are redrawn with updated port positions
                updateConnections();
            }
        } else {
            // Tool-specific updates will be implemented in the next phase
        }

        hideConfigPanel();
    }
    
    /**
     * Show the new agent modal
     */
    function showNewAgentModal() {
        newAgentModal.classList.add('open');
    }
    
    /**
     * Hide all modals
     */
    function hideModals() {
        document.querySelectorAll('.modal').forEach(modal => {
            modal.classList.remove('open');
        });
    }
    
    /**
     * Update config fields in the new agent modal based on selected type
     * @param {string} agentType - The selected agent type
     */
    function updateNewAgentConfigFields(agentType) {
        const container = document.getElementById('new-agent-config-container');
        let configHtml = '';
        
        if (agentType === 'openai') {
            configHtml = `
                <div class="form-group">
                    <label for="new-openai-api-key">OpenAI API Key</label>
                    <input type="password" id="new-openai-api-key" class="form-control" placeholder="sk-...">
                </div>
                <div class="form-group">
                    <label for="new-openai-model">Model</label>
                    <select id="new-openai-model" class="form-control">
                        <option value="gpt-4o">GPT-4o</option>
                        <option value="gpt-4-turbo">GPT-4 Turbo</option>
                        <option value="gpt-3.5-turbo">GPT-3.5 Turbo</option>
                    </select>
                </div>
            `;
        } else if (agentType === 'ollama') {
            configHtml = `
                <div class="form-group">
                    <label for="new-ollama-api-url">Ollama API url</label>
                    <input type="text" id="new-ollama-api-url" class="form-control" placeholder="http://localhost:11434">
                </div>
                <div class="form-group">
                    <label for="new-ollama-api-key">Ollama API Key</label>
                    <input type="password" id="new-ollama-api-key" class="form-control" placeholder="sk-...">
                </div>
                <div class="form-group">
                    <label for="new-ollama-model">Model</label>
                    <input type="text" id="new-ollama-api-model" class="form-control" placeholder="deepseek-r1:latest">
                </div>
            `;
        } else if (agentType === 'anthropic') {
            configHtml = `
                <div class="form-group">
                    <label for="new-anthropic-api-key">Anthropic API Key</label>
                    <input type="password" id="new-anthropic-api-key" class="form-control" placeholder="sk_ant-...">
                </div>
                <div class="form-group">
                    <label for="new-anthropic-model">Model</label>
                    <select id="new-anthropic-model" class="form-control">
                        <option value="claude-3-opus">Claude 3 Opus</option>
                        <option value="claude-3-sonnet">Claude 3 Sonnet</option>
                        <option value="claude-3-haiku">Claude 3 Haiku</option>
                    </select>
                </div>
            `;
        } else if (agentType === 'bedrock') {
            configHtml = `
                <div class="form-group">
                    <label for="new-aws-access-key">AWS Access Key</label>
                    <input type="password" id="new-aws-access-key" class="form-control">
                </div>
                <div class="form-group">
                    <label for="new-aws-secret-key">AWS Secret Key</label>
                    <input type="password" id="new-aws-secret-key" class="form-control">
                </div>
                <div class="form-group">
                    <label for="new-aws-region">AWS Region</label>
                    <select id="new-aws-region" class="form-control">
                        <option value="us-east-1">US East (N. Virginia)</option>
                        <option value="us-west-2">US West (Oregon)</option>
                        <option value="eu-west-1">EU (Ireland)</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="new-bedrock-model">Model</label>
                    <select id="new-bedrock-model" class="form-control">
                        <option value="anthropic.claude-3-sonnet-20240229-v1:0">Claude 3 Sonnet</option>
                        <option value="anthropic.claude-3-haiku-20240307-v1:0">Claude 3 Haiku</option>
                        <option value="amazon.titan-text-express-v1">Titan Text</option>
                    </select>
                </div>
            `;
        } else if (agentType === 'custom') {
            configHtml = `
                <div class="form-group">
                    <label for="new-agent-port">Port</label>
                    <input type="number" id="new-agent-port" class="form-control" placeholder="8000" min="1024" max="65535">
                </div>
                <div class="form-group">
                    <label for="new-agent-endpoint">Endpoint URL (optional)</label>
                    <input type="text" id="new-agent-endpoint" class="form-control" placeholder="http://localhost:{port}">
                </div>
                <div class="form-group">
                    <label for="new-agent-script">Script Path (optional)</label>
                    <input type="text" id="new-agent-script" class="form-control" placeholder="/path/to/agent.py">
                </div>
            `;
        }
        
        container.innerHTML = configHtml;
    }
    
    /**
     * Create a new agent from the modal
     */
    function createNewAgent() {
        const agentName = document.getElementById('new-agent-name').value;
        const agentType = document.getElementById('new-agent-type').value;
        
        if (!agentName) {
            alert('Please enter a name for your agent');
            return;
        }
        
        // Get the center of the visible canvas
        const canvasRect = canvas.getBoundingClientRect();
        const x = ((canvasRect.width / 2) - canvasOffset.x) / canvasScale;
        const y = ((canvasRect.height / 2) - canvasOffset.y) / canvasScale;
        
        // Create the node
        const nodeData = createNode(x, y, { 
            type: 'agent', 
            agentType: agentType 
        });
        
        // Configure the node
        nodeData.config.name = agentName;
        
        if (agentType === 'openai') {
            nodeData.config.apiKey = document.getElementById('new-openai-api-key')?.value || '';
            nodeData.config.model = document.getElementById('new-openai-model')?.value || 'gpt-4o';
        } else if (agentType === 'ollama') {
            nodeData.config.apiUrl = document.getElementById('new-ollama-api-url')?.value || '';
            nodeData.config.apiKey = document.getElementById('new-ollama-api-key')?.value || '';
            nodeData.config.model = document.getElementById('new-ollama-model')?.value || 'deepseek-r1:latest';
        } else if (agentType === 'anthropic') {
            nodeData.config.apiKey = document.getElementById('new-anthropic-api-key')?.value || '';
            nodeData.config.model = document.getElementById('new-anthropic-model')?.value || 'claude-3-opus';
        } else if (agentType === 'bedrock') {
            nodeData.config.accessKey = document.getElementById('new-aws-access-key')?.value || '';
            nodeData.config.secretKey = document.getElementById('new-aws-secret-key')?.value || '';
            nodeData.config.region = document.getElementById('new-aws-region')?.value || 'us-east-1';
            nodeData.config.model = document.getElementById('new-bedrock-model')?.value || 'anthropic.claude-3-sonnet-20240229-v1:0';
        } else if (agentType === 'custom') {
            nodeData.config.port = document.getElementById('new-agent-port')?.value || '';
            nodeData.config.endpoint = document.getElementById('new-agent-endpoint')?.value || '';
            nodeData.config.script = document.getElementById('new-agent-script')?.value || '';
        }
        
        // Update node display
        const nodeNameElement = nodeData.element.querySelector('.node-title span');
        if (nodeNameElement) {
            nodeNameElement.textContent = agentName;
        }
        
        // Update model badge
        const modelBadge = nodeData.element.querySelector('.agent-model');
        if (modelBadge) {
            let modelText = 'Not configured';
            
            if (agentType === 'openai' && nodeData.config.model) {
                modelText = nodeData.config.model;
            }if (agentType === 'ollama' && nodeData.config.model) {
                modelText = nodeData.config.model;
            } else if (agentType === 'anthropic' && nodeData.config.model) {
                modelText = nodeData.config.model;
            } else if (agentType === 'bedrock' && nodeData.config.model) {
                modelText = nodeData.config.model.split('.')[1]?.split('-')[0] || nodeData.config.model;
            } else if (agentType === 'custom' && nodeData.config.port) {
                modelText = `Port: ${nodeData.config.port}`;
            }
            
            modelBadge.textContent = modelText;
        }
        
        // Remove the help text if it's the first node
        if (nodes.length === 1) {
            emptyCanvasHelp.style.display = 'none';
        }
        
        hideModals();
    }
    
    /**
     * Add an input node to the canvas
     */
    function addInputNode() {
        // Get the center of the visible canvas
        const canvasRect = canvas.getBoundingClientRect();
        const x = ((canvasRect.width / 3) - canvasOffset.x) / canvasScale;
        const y = ((canvasRect.height / 2) - canvasOffset.y) / canvasScale;
        
        const nodeId = 'node-' + (++nodeCounter);
        const nodeElement = document.createElement('div');
        nodeElement.id = nodeId;
        nodeElement.classList.add('node', 'input');
        
        nodeElement.innerHTML = `
            <div class="node-header">
                <div class="node-title">
                    <i class="bi bi-box-arrow-in-right"></i>
                    <span>Input</span>
                </div>
                <div class="node-actions">
                    <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                </div>
            </div>
            <div class="node-content">
                <div>Network input</div>
            </div>
            <div class="node-ports">
                <div class="output-port">
                    <div class="port port-right" data-port-type="output"></div>
                    <span class="port-label">Output</span>
                </div>
            </div>
        `;
        
        // Position the node
        nodeElement.style.left = `${x}px`;
        nodeElement.style.top = `${y}px`;
        
        // Add the node to the canvas
        canvas.appendChild(nodeElement);
        
        // Store node data
        const nodeData = {
            id: nodeId,
            element: nodeElement,
            type: 'input',
            position: { x, y },
            config: {
                name: 'Input'
            }
        };
        
        nodes.push(nodeData);
        
        // Add event listeners to the node
        initNodeEvents(nodeElement, nodeData);
        
        // Remove the help text if it's the first node
        if (nodes.length === 1) {
            emptyCanvasHelp.style.display = 'none';
        }
        
        return nodeData;
    }
    
    /**
     * Add an output node to the canvas
     */
    function addOutputNode() {
        // Get the center of the visible canvas
        const canvasRect = canvas.getBoundingClientRect();
        const x = ((canvasRect.width * 2/3) - canvasOffset.x) / canvasScale;
        const y = ((canvasRect.height / 2) - canvasOffset.y) / canvasScale;
        
        const nodeId = 'node-' + (++nodeCounter);
        const nodeElement = document.createElement('div');
        nodeElement.id = nodeId;
        nodeElement.classList.add('node', 'output');
        
        nodeElement.innerHTML = `
            <div class="node-header">
                <div class="node-title">
                    <i class="bi bi-box-arrow-right"></i>
                    <span>Output</span>
                </div>
                <div class="node-actions">
                    <button class="node-btn configure-node" title="Configure"><i class="bi bi-gear"></i></button>
                </div>
            </div>
            <div class="node-content">
                <div>Network output</div>
            </div>
            <div class="node-ports">
                <div class="input-port">
                    <div class="port port-left" data-port-type="input"></div>
                    <span class="port-label">Input</span>
                </div>
            </div>
        `;
        
        // Position the node
        nodeElement.style.left = `${x}px`;
        nodeElement.style.top = `${y}px`;
        
        // Add the node to the canvas
        canvas.appendChild(nodeElement);
        
        // Store node data
        const nodeData = {
            id: nodeId,
            element: nodeElement,
            type: 'output',
            position: { x, y },
            config: {
                name: 'Output'
            }
        };
        
        nodes.push(nodeData);
        
        // Add event listeners to the node
        initNodeEvents(nodeElement, nodeData);
        
        // Remove the help text if it's the first node
        if (nodes.length === 1) {
            emptyCanvasHelp.style.display = 'none';
        }
        
        return nodeData;
    }
    
    /**
     * Deselect all nodes and connections
     */
    function deselectAllNodes() {
        nodes.forEach(nodeData => {
            nodeData.element.classList.remove('selected');
        });
        
        connections.forEach(conn => {
            conn.element.classList.remove('selected');
        });
        
        selectedNode = null;
    }
    
    /**
     * Delete selected nodes or connections
     */
    function deleteSelected() {
        if (selectedNode) {
            // Find all connections to this node
            const relatedConnections = connections.filter(conn => 
                conn.start.nodeId === selectedNode.id || conn.end.nodeId === selectedNode.id
            );
            
            // Remove connections
            relatedConnections.forEach(conn => {
                svgContainer.removeChild(conn.element);
                connections = connections.filter(c => c.id !== conn.id);
            });
            
            // Remove node
            nodeContainer.removeChild(selectedNode.element);
            nodes = nodes.filter(n => n.id !== selectedNode.id);
            selectedNode = null;
            
            // Show help text if no nodes left
            if (nodes.length === 0) {
                emptyCanvasHelp.style.display = 'flex';
            }
        } else {
            // Check for selected connection
            const selectedConn = connections.find(conn => conn.element.classList.contains('selected'));
            if (selectedConn) {
                svgContainer.removeChild(selectedConn.element);
                connections = connections.filter(c => c.id !== selectedConn.id);
            }
        }
    }
    
    /**
     * Zoom in on the canvas
     */
    function zoomIn() {
        const newScale = Math.min(canvasScale + 0.1, 2);
        if (newScale !== canvasScale) {
            // Get canvas center
            const rect = canvas.getBoundingClientRect();
            const centerX = rect.width / 2;
            const centerY = rect.height / 2;
            
            // Calculate the point in world space
            const worldX = (centerX - canvasOffset.x) / canvasScale;
            const worldY = (centerY - canvasOffset.y) / canvasScale;
            
            // Update the scale
            canvasScale = newScale;
            
            // Calculate the new offset to keep the center point
            canvasOffset.x = centerX - worldX * canvasScale;
            canvasOffset.y = centerY - worldY * canvasScale;
            
            updateCanvasTransform();
        }
    }
    
    /**
     * Zoom out on the canvas
     */
    function zoomOut() {
        const newScale = Math.max(canvasScale - 0.1, 0.3);
        if (newScale !== canvasScale) {
            // Get canvas center
            const rect = canvas.getBoundingClientRect();
            const centerX = rect.width / 2;
            const centerY = rect.height / 2;
            
            // Calculate the point in world space
            const worldX = (centerX - canvasOffset.x) / canvasScale;
            const worldY = (centerY - canvasOffset.y) / canvasScale;
            
            // Update the scale
            canvasScale = newScale;
            
            // Calculate the new offset to keep the center point
            canvasOffset.x = centerX - worldX * canvasScale;
            canvasOffset.y = centerY - worldY * canvasScale;
            
            updateCanvasTransform();
        }
    }
    
    /**
     * Reset zoom and pan on the canvas
     */
    function resetZoom() {
        canvasScale = 1;
        canvasOffset = { x: 0, y: 0 };
        updateCanvasTransform();
    }
    
    /**
     * Save the current network to the server
     */
    function saveNetwork() {
        const networkData = {
            name: document.getElementById('network-name').textContent,
            nodes: nodes.map(node => ({
                id: node.id,
                type: node.type,
                subType: node.subType,
                position: node.position,
                config: node.config
            })),
            connections: connections.map(conn => ({
                sourceNode: conn.start.nodeId,
                sourcePort: conn.start.port,
                sourcePortNumber: conn.start.portNumber, // For router output ports
                targetNode: conn.end.nodeId,
                targetPort: conn.end.port,
                targetPortNumber: conn.end.portNumber, // For router output ports
                // Also include edge type for router connections
                edgeType: conn.start.portNumber !== undefined ? 'ROUTE_OUTPUT' : 'DATA'
            }))
        };
        
        // TODO: Implement server-side saving
        console.log('Network data to save:', networkData);

        // Show a modern notification instead of an alert
        showNotification('Coming Soon: Network saving functionality will be available in the next phase.', 'info');

        // Add a visual feedback animation to the save button
        const saveBtn = document.getElementById('floating-save-btn');
        saveBtn.classList.add('save-success');

        // Remove the animation class after it completes
        setTimeout(() => {
            saveBtn.classList.remove('save-success');
        }, 1500);
    }
    
    /**
     * Run the current network on the server
     */
    function runNetwork() {
        // Check if there are any nodes first
        if (nodes.length === 0) {
            showNotification('Cannot run an empty network. Please add at least one agent node.', 'error');
            return;
        }

        // Check if all agent nodes are properly configured
        const agentNodes = nodes.filter(node => node.type === 'agent');

        // If there are no agent nodes, show an error
        if (agentNodes.length === 0) {
            showNotification('No agent nodes found. Please add at least one agent to your network.', 'error');
            return;
        }

        // Validate each agent node has necessary configuration
        const unconfiguredAgents = [];
        agentNodes.forEach(agent => {
            // Only check OpenAI & Ollama agents since others are marked as "under development"
            if (['openai', 'ollama'].includes(agent.subType)) {
                // Check if the agent has the required configuration
                if (!agent.config ||
                    !agent.config.apiKey ||
                    !agent.config.model) {
                    if (agent.subType === 'ollama' && !agent.config.apiUrl) {
                        unconfiguredAgents.push(agent);
                    } else {
                        unconfiguredAgents.push(agent);
                    }
                }
            }
        });

        // Show error if any agents are not configured
        if (unconfiguredAgents.length > 0) {
            // First, clear any previous configuration indicators
            document.querySelectorAll('.needs-configuration').forEach(node => {
                node.classList.remove('needs-configuration');
            });

            // Highlight the unconfigured agents
            unconfiguredAgents.forEach(agent => {
                const agentElement = agent.element;
                if (agentElement) {
                    // Find the node content area to add an indicator badge without affecting the node shape
                    // Only update the existing badge if it exists, don't create new elements
                    const nodeContent = agentElement.querySelector('.node-content');
                    if (nodeContent) {
                        // Find the badge if it exists - but don't create one if it doesn't
                        const nodeBadge = nodeContent.querySelector('.node-badge');
                        if (nodeBadge) {
                            // Update the existing badge
                            nodeBadge.textContent = "Not Configured";
                            nodeBadge.classList.add('not-configured-badge');
                        }
                    }

                    // No longer adding the needs-configuration class to avoid distortion

                    // Add a shake animation for emphasis
                    agentElement.classList.add('shake-animation');

                    // Remove the animation class after it completes
                    setTimeout(() => {
                        agentElement.classList.remove('shake-animation');
                    }, 1000);
                }
            });

            showNotification('Some agent nodes need configuration before running. Please configure all highlighted agents.', 'error');
            return;
        }

        // Detect multiple networks on the canvas
        const detectedNetworks = detectMultipleNetworks();

        // Prepare network data for the current state
        const currentNetworkData = {
            name: document.getElementById('network-name').textContent,
            nodes: nodes.map(node => ({
                id: node.id,
                type: node.type,
                subType: node.subType,
                position: node.position,
                config: node.config
            })),
            connections: connections.map(conn => ({
                sourceNode: conn.start.nodeId,
                sourcePort: conn.start.port,
                targetNode: conn.end.nodeId,
                targetPort: conn.end.port
            }))
        };

        // Use the enhanced network execution dialog
        if (typeof window.showNetworkExecutionDialog === 'function') {
            window.showNetworkExecutionDialog(currentNetworkData, detectedNetworks);
        } else {
            // Fallback to old dialog if new one isn't available
            showExecutionDialog(currentNetworkData, detectedNetworks);
        }
    }

    /**
     * Detect multiple independent networks on the canvas
     * @returns {Array} Array of detected networks
     */
    function detectMultipleNetworks() {
        // This function detects separate connected components (networks) on the canvas
        const networks = [];

        // Skip if no nodes or connections
        if (nodes.length === 0 || connections.length === 0) {
            return networks;
        }

        // Create a map of node IDs to track visited status
        const visitedNodes = new Map();
        nodes.forEach(node => visitedNodes.set(node.id, false));

        // Create an adjacency list representation of the graph
        const adjacencyList = new Map();
        nodes.forEach(node => adjacencyList.set(node.id, []));

        // Add connections to the adjacency list (bidirectional)
        connections.forEach(conn => {
            const sourceId = conn.start.nodeId;
            const targetId = conn.end.nodeId;
            adjacencyList.get(sourceId).push(targetId);
            adjacencyList.get(targetId).push(sourceId);
        });

        // Helper function for depth-first search
        function dfs(nodeId, networkNodes) {
            // Mark as visited
            visitedNodes.set(nodeId, true);
            networkNodes.push(nodeId);

            // Visit all adjacent nodes
            adjacencyList.get(nodeId).forEach(adjacentId => {
                if (!visitedNodes.get(adjacentId)) {
                    dfs(adjacentId, networkNodes);
                }
            });
        }

        // Find all connected components (networks)
        for (const node of nodes) {
            if (!visitedNodes.get(node.id)) {
                const networkNodes = [];
                dfs(node.id, networkNodes);

                // Only count as a network if it has at least one input and one output node
                const networkNodeObjects = networkNodes.map(nodeId =>
                    nodes.find(n => n.id === nodeId)
                );

                const hasInput = networkNodeObjects.some(node => node.type === 'input');
                const hasOutput = networkNodeObjects.some(node => node.type === 'output');

                if (hasInput && hasOutput && networkNodes.length >= 3) {
                    // Create a subset of the connections that belong to this network
                    const networkConnections = connections.filter(conn =>
                        networkNodes.includes(conn.start.nodeId) &&
                        networkNodes.includes(conn.end.nodeId)
                    );

                    // Validate this network
                    const isValid = validateNetworkSubset(networkNodeObjects, networkConnections);

                    if (isValid) {
                        // Create network data structure
                        networks.push({
                            name: `Network ${networks.length + 1}`,
                            nodes: networkNodeObjects.map(node => ({
                                id: node.id,
                                type: node.type,
                                subType: node.subType,
                                position: node.position,
                                config: node.config
                            })),
                            connections: networkConnections.map(conn => ({
                                sourceNode: conn.start.nodeId,
                                sourcePort: conn.start.port,
                                targetNode: conn.end.nodeId,
                                targetPort: conn.end.port
                            })),
                            nodeCount: networkNodes.length,
                            connectionCount: networkConnections.length
                        });
                    }
                }
            }
        }

        return networks;
    }

    /**
     * Validate a subset of nodes and connections as a valid network
     * @param {Array} networkNodes - Array of node objects in the network
     * @param {Array} networkConnections - Array of connection objects in the network
     * @returns {boolean} - Whether the network is valid
     */
    function validateNetworkSubset(networkNodes, networkConnections) {
        // Check for input and output nodes
        const inputNodes = networkNodes.filter(node => node.type === 'input');
        const outputNodes = networkNodes.filter(node => node.type === 'output');

        if (inputNodes.length === 0 || outputNodes.length === 0) {
            return false;
        }

        // Check for connections between nodes
        if (networkConnections.length === 0) {
            return false;
        }

        // Check if there's a path from input to output nodes
        // This is a simplified check - a full check would need more graph theory
        const connectedNodeIds = new Set();
        networkConnections.forEach(conn => {
            connectedNodeIds.add(conn.start.nodeId);
            connectedNodeIds.add(conn.end.nodeId);
        });

        // Check if at least one input and one output node are connected
        const inputConnected = inputNodes.some(node => connectedNodeIds.has(node.id));
        const outputConnected = outputNodes.some(node => connectedNodeIds.has(node.id));

        if (!inputConnected || !outputConnected) {
            return false;
        }

        // Check for any disconnected nodes
        for (const node of networkNodes) {
            if (!connectedNodeIds.has(node.id)) {
                return false; // Disconnected node found
            }
        }

        // Check for node configuration
        for (const node of networkNodes) {
            if (node.type === 'agent') {
                const config = node.config || {};
                const agentType = node.subType;

                // Validate agent configuration
                if (agentType === 'openai' && (!config.apiKey || !config.model)) {
                    return false;
                }
                else if (agentType === 'ollama' && (!config.apiUrl || !config.apiKey || !config.model)) {
                    return false;
                }
                else if (agentType === 'anthropic' && (!config.apiKey || !config.model)) {
                    return false;
                }
                else if (agentType === 'bedrock' &&
                        (!config.accessKey || !config.secretKey || !config.region || !config.model)) {
                    return false;
                }
                else if (agentType === 'custom' &&
                        (!config.port && !config.endpoint)) {
                    return false;
                }
            }
        }

        return true;
    }
    
    /**
     * Show the execution dialog for the user to provide input
     * @param {Object} networkData - The prepared network data
     * @param {Array} detectedNetworks - Array of detected networks on the canvas
     */
    function showExecutionDialog(networkData, detectedNetworks) {
        // Check if we have multiple networks
        const hasMultipleNetworks = detectedNetworks && detectedNetworks.length > 0;

        // Check if there's already an execution dialog
        let executionDialog = document.getElementById('execution-dialog');

        // Create the dialog if it doesn't exist
        if (!executionDialog) {
            executionDialog = document.createElement('div');
            executionDialog.id = 'execution-dialog';
            executionDialog.className = 'modal';

            executionDialog.innerHTML = `
                <div class="modal-content execution-dialog-content">
                    <div class="modal-header">
                        <h2>Run Agent Network</h2>
                        <button class="close-btn modal-close"><i class="bi bi-x-lg"></i></button>
                    </div>
                    <div class="modal-body">
                        <div class="execution-tabs">
                            <button class="tab-btn active" data-tab="input">Input</button>
                            <button class="tab-btn" data-tab="networks">Networks</button>
                            <button class="tab-btn" data-tab="execution">Execution</button>
                            <button class="tab-btn" data-tab="output">Output</button>
                        </div>

                        <div class="tab-content" id="input-tab">
                            <div class="form-group">
                                <label for="user-input">Your request:</label>
                                <textarea id="user-input" class="form-control" rows="5" placeholder="Enter your request here..."></textarea>
                            </div>
                            <div class="form-group execution-mode-selector">
                                <label>Execution Mode:</label>
                                <div class="radio-group">
                                    <label class="radio-option">
                                        <input type="radio" name="execution-mode" value="sequential" checked> Sequential
                                    </label>
                                    <label class="radio-option">
                                        <input type="radio" name="execution-mode" value="parallel"> Parallel
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="tab-content hidden" id="networks-tab">
                            <div class="networks-selection">
                                <div class="selection-header">
                                    <h3>Network Selection</h3>
                                    <div class="execution-mode">
                                        <span>Execution Mode:</span>
                                        <div class="toggle-container">
                                            <input type="radio" id="sequential-mode" name="execution-mode" value="sequential" checked>
                                            <label for="sequential-mode">Sequential</label>
                                            <input type="radio" id="parallel-mode" name="execution-mode" value="parallel">
                                            <label for="parallel-mode">Parallel</label>
                                        </div>
                                    </div>
                                </div>

                                <div class="network-list" id="network-selection-list">
                                    <div class="network-item current-network selected">
                                        <div class="network-title">
                                            <i class="bi bi-diagram-3"></i>
                                            <span>Current Network</span>
                                        </div>
                                        <div class="network-details">
                                            <span class="node-count">${nodes.length} nodes</span>
                                            <span class="connection-count">${connections.length} connections</span>
                                        </div>
                                        <div class="network-actions">
                                            <input type="checkbox" class="network-checkbox" id="network-current" checked>
                                            <label for="network-current">Select</label>
                                        </div>
                                    </div>

                                    <!-- Detected networks will be shown here -->
                                    <div id="detected-networks-container">
                                        ${detectedNetworks && detectedNetworks.length > 0 ?
                                            detectedNetworks.map((network, index) => `
                                                <div class="network-item detected-network" data-network-index="${index}">
                                                    <div class="network-title">
                                                        <i class="bi bi-diagram-3"></i>
                                                        <span>${network.name}</span>
                                                    </div>
                                                    <div class="network-details">
                                                        <span class="node-count">${network.nodeCount} nodes</span>
                                                        <span class="connection-count">${network.connectionCount} connections</span>
                                                    </div>
                                                    <div class="network-actions">
                                                        <input type="checkbox" class="network-checkbox" id="network-${index}" checked>
                                                        <label for="network-${index}">Select</label>
                                                    </div>
                                                </div>
                                            `).join('') :
                                            '<div class="no-networks-message">No additional networks detected</div>'
                                        }
                                    </div>

                                    <!-- Saved networks will be loaded here -->
                                    <div id="saved-networks-container"></div>

                                    <!-- Network mode explanation -->
                                    <div class="network-mode-explanation">
                                        <h4>Execution Modes</h4>
                                        <div class="mode-description">
                                            <h5><i class="bi bi-arrow-down"></i> Sequential</h5>
                                            <p>Networks execute one after another. The output of each network is used as input for the next.</p>
                                        </div>
                                        <div class="mode-description">
                                            <h5><i class="bi bi-arrows"></i> Parallel</h5>
                                            <p>All networks execute at the same time with the same input. Results are returned separately.</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="tab-content hidden" id="execution-tab">
                            <div class="execution-status">
                                <div class="status-header">
                                    <h3>Execution Status</h3>
                                    <span class="status-badge pending">Pending</span>
                                    <button id="clear-log-btn" class="secondary-btn" style="margin-left: auto; padding: 4px 10px; font-size: 12px;">
                                        <i class="bi bi-trash"></i> Clear Messages
                                    </button>
                                </div>
                                <div class="execution-log">
                                    <div class="log-container" id="execution-log-container">
                                        <div class="log-entry">Agent network ready to run...</div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="tab-content hidden" id="output-tab">
                            <div class="form-group">
                                <label for="network-output">Result:</label>
                                <div class="output-container" id="network-output-container">
                                    <div class="placeholder-text">Run the network to see results...</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <div class="execution-progress">
                            <div class="progress-bar">
                                <div class="progress-fill" style="width: 0%"></div>
                            </div>
                        </div>
                        <button class="secondary-btn modal-close">Close</button>
                        <button id="run-network-btn" class="primary-btn">Run</button>
                    </div>
                </div>
            `;
            
            document.body.appendChild(executionDialog);
            
            // Initialize tab switching
            const tabButtons = executionDialog.querySelectorAll('.tab-btn');
            const tabContents = executionDialog.querySelectorAll('.tab-content');
            
            tabButtons.forEach(button => {
                button.addEventListener('click', () => {
                    // Remove active class from all buttons
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    
                    // Hide all tab contents
                    tabContents.forEach(content => content.classList.add('hidden'));
                    
                    // Add active class to the clicked button
                    button.classList.add('active');
                    
                    // Show the corresponding tab content
                    const tabName = button.getAttribute('data-tab');
                    const tabContent = document.getElementById(`${tabName}-tab`);
                    tabContent.classList.remove('hidden');
                });
            });
            
            // Handle run button click
            const runButton = executionDialog.querySelector('#run-network-btn');
            runButton.addEventListener('click', () => {
                executeNetworkWithInput(networkData, detectedNetworks);
            });

            // Show Networks tab if multiple networks are detected
            // Add data-mode attributes to mode buttons
            const modeButtons = executionDialog.querySelectorAll('.mode-btn');
            if (modeButtons && modeButtons.length > 0) {
                modeButtons.forEach(btn => {
                    if (btn.textContent.toLowerCase().includes('sequential')) {
                        btn.setAttribute('data-mode', 'sequential');
                    } else if (btn.textContent.toLowerCase().includes('parallel')) {
                        btn.setAttribute('data-mode', 'parallel');
                    }
                });
            }

            if (detectedNetworks && detectedNetworks.length > 0) {
                const networksTab = executionDialog.querySelector('.tab-btn[data-tab="networks"]');
                networksTab.style.display = 'inline-block';

                // Add click handlers for network selection checkboxes and mode buttons
                setTimeout(() => {
                    // Handle checkbox selection
                    const networkCheckboxes = executionDialog.querySelectorAll('.network-checkbox');
                    networkCheckboxes.forEach(checkbox => {
                        checkbox.addEventListener('change', () => {
                            // Update visual selection state when checkbox changes
                            const networkItem = checkbox.closest('.network-item');
                            if (checkbox.checked) {
                                networkItem.classList.add('selected');
                            } else {
                                networkItem.classList.remove('selected');
                            }
                        });
                    });

                    // Handle mode buttons
                    const modeButtons = executionDialog.querySelectorAll('.mode-btn');
                    if (modeButtons.length > 0) {
                        modeButtons.forEach(btn => {
                            btn.addEventListener('click', () => {
                                // Deactivate all buttons
                                modeButtons.forEach(b => b.classList.remove('active'));
                                // Activate clicked button
                                btn.classList.add('active');
                                // Update hidden input with selected mode
                                const mode = btn.getAttribute('data-mode');
                                const hiddenInput = executionDialog.querySelector('input[name="execution-mode"]');
                                if (hiddenInput) {
                                    hiddenInput.value = mode;
                                }
                                // Also update radio buttons in other tabs
                                const radioButtons = executionDialog.querySelectorAll(`input[type="radio"][name="execution-mode"][value="${mode}"]`);
                                radioButtons.forEach(radio => {
                                    radio.checked = true;
                                });
                            });
                        });
                    }
                }, 100);
            } else {
                // Hide Networks tab if no additional networks are detected
                const networksTab = executionDialog.querySelector('.tab-btn[data-tab="networks"]');
                networksTab.style.display = 'none';
            }

            // Handle clear log button click
            const clearLogButton = executionDialog.querySelector('#clear-log-btn');
            clearLogButton.addEventListener('click', () => {
                const logContainer = document.getElementById('execution-log-container');
                // Clear all log entries except the initial one
                logContainer.innerHTML = '<div class="log-entry">Agent network ready to run...</div>';
                // Show notification
                showNotification('Execution logs cleared', 'success');
            });

            // Handle modal close
            const closeButtons = executionDialog.querySelectorAll('.modal-close');
            closeButtons.forEach(button => {
                button.addEventListener('click', () => {
                    executionDialog.classList.remove('open');
                });
            });
        }
        
        // Show the dialog
        executionDialog.classList.add('open');

        // If multiple networks are detected, show the Networks tab by default
        if (hasMultipleNetworks) {
            // Get all tab buttons and contents
            const tabButtons = executionDialog.querySelectorAll('.tab-btn');
            const tabContents = executionDialog.querySelectorAll('.tab-content');

            // Switch to Networks tab
            tabButtons.forEach(btn => btn.classList.remove('active'));
            const networksTab = Array.from(tabButtons).find(btn => btn.getAttribute('data-tab') === 'networks');
            if (networksTab) {
                networksTab.classList.add('active');

                // Show the networks tab content
                tabContents.forEach(content => content.classList.add('hidden'));
                document.getElementById('networks-tab').classList.remove('hidden');

                // Show notification that multiple networks were detected
                showNotification(`${detectedNetworks.length + 1} networks detected on the canvas. Select which ones to run.`, 'info');
            }
        }
    }
    
    /**
     * Execute the network with user input
     * @param {Object} networkData - The prepared network data
     * @param {Array} detectedNetworks - Array of detected networks on the canvas
     */
    function executeNetworkWithInput(networkData, detectedNetworks) {
        const userInput = document.getElementById('user-input').value;

        if (!userInput.trim()) {
            showNotification('Please provide input for the network', 'warning');
            return;
        }

        // Update UI to execution tab
        const executionDialog = document.getElementById('execution-dialog');
        const tabButtons = executionDialog.querySelectorAll('.tab-btn');
        const tabContents = executionDialog.querySelectorAll('.tab-content');

        // Switch to execution tab
        tabButtons.forEach(btn => btn.classList.remove('active'));
        tabButtons[2].classList.add('active'); // Execution tab (index 2 with new Networks tab)

        tabContents.forEach(content => content.classList.add('hidden'));
        document.getElementById('execution-tab').classList.remove('hidden');

        // Update status badge
        const statusBadge = document.querySelector('.status-badge');
        statusBadge.className = 'status-badge running';
        statusBadge.textContent = 'Running';

        // Update progress bar
        const progressFill = document.querySelector('.progress-fill');
        progressFill.style.width = '15%';

        // Get all selected networks for execution
        const networks = getSelectedNetworks(networkData, detectedNetworks);

        // Get execution mode
        const executionMode = document.querySelector('input[name="execution-mode"]:checked').value;

        // If no networks are selected (validation failed), abort
        if (networks.length === 0) {
            // Update status
            statusBadge.className = 'status-badge error';
            statusBadge.textContent = 'Error';

            // Add error log
            addLogEntry('No valid networks selected for execution', 'error');

            // Re-enable run button
            const runButton = document.getElementById('run-network-btn');
            runButton.disabled = false;
            runButton.textContent = 'Run';
            return;
        }

        // Add log entry
        if (networks.length === 1) {
            addLogEntry('Starting network execution...');
        } else {
            addLogEntry(`Starting execution of ${networks.length} networks in ${executionMode} mode...`);
        }
        addLogEntry(`Processing input: "${userInput.substring(0, 40)}${userInput.length > 40 ? '...' : ''}"`);

        // Disable run button during execution
        const runButton = document.getElementById('run-network-btn');
        runButton.disabled = true;
        runButton.textContent = 'Running...';

        // Clear any previous execution highlights
        resetNodeStyles();

        // Prepare the execution data
        let executionData;

        if (networks.length === 1) {
            // Single network execution
            executionData = {
                ...networks[0].data,
                input: userInput
            };
        } else {
            // Multi-network execution
            executionData = {
                networks: networks,
                execution_mode: executionMode,
                input: userInput
            };
        }

        // Send the request to the server
        fetch('/api/workflows/run-network', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(executionData)
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`Network execution failed: ${response.statusText}`);
            }
            return response.json();
        })
        .then(data => {
            // Store execution ID for status polling
            const executionId = data.execution_id;

            if (executionId) {
                // Start polling for updates (for long-running executions)
                addLogEntry(`Execution started with ID: ${executionId}`);
                pollExecutionStatus(executionId);
            } else {
                // Update progress immediately if no execution ID
                progressFill.style.width = '100%';

                // Update status
                statusBadge.className = 'status-badge completed';
                statusBadge.textContent = 'Completed';

                // Add log entries
                if (data.type === 'multi_network_output') {
                    const networkCount = data.networks_count || networks.length;
                    const mode = data.mode || executionMode;
                    const executionTime = data.execution_time ? data.execution_time.toFixed(2) : '?';

                    addLogEntry(`${networkCount} networks executed in ${executionTime} seconds (${mode} mode)`);
                    addLogEntry('Multi-network execution completed successfully', 'success');
                } else {
                    addLogEntry('Network execution completed successfully', 'success');
                }

                showExecutionResult(data, userInput, tabButtons, tabContents, runButton);
            }
        })
        .catch(error => {
            console.error('Execution error:', error);

            // Update status
            statusBadge.className = 'status-badge error';
            statusBadge.textContent = 'Error';

            // Add error log
            addLogEntry(`Error: ${error.message}`, 'error');

            // For development, we'll show a simulated response anyway
            progressFill.style.width = '100%';

            setTimeout(() => {
                tabButtons.forEach(btn => btn.classList.remove('active'));
                tabButtons[3].classList.add('active'); // Output tab (index 3 with new Networks tab)

                tabContents.forEach(content => content.classList.add('hidden'));
                document.getElementById('output-tab').classList.remove('hidden');

                // Update output with error message
                const outputContainer = document.getElementById('network-output-container');
                outputContainer.innerHTML = '';

                const outputContent = document.createElement('div');
                outputContent.className = 'output-content error';

                // For development, we'll simulate a response
                outputContent.textContent = simulateResponse(userInput);

                outputContainer.appendChild(outputContent);

                // Re-enable run button
                runButton.disabled = false;
                runButton.textContent = 'Run Again';
            }, 1000);
        });
    }

    /**
     * Get selected networks for execution
     * @param {Object} currentNetworkData - The current network data
     * @param {Array} detectedNetworks - Array of detected networks on the canvas
     * @returns {Array} Array of selected networks
     */
    function getSelectedNetworks(currentNetworkData, detectedNetworks) {
        const networks = [];
        const executionMode = document.querySelector('input[name="execution-mode"]:checked').value;

        // Check if current network is selected
        const currentNetworkCheckbox = document.getElementById('network-current');
        if (currentNetworkCheckbox && currentNetworkCheckbox.checked) {
            // Validate current network structure
            const validationResult = validateNetwork();
            if (validationResult.valid) {
                networks.push({
                    id: 'current',
                    data: currentNetworkData,
                    mode: executionMode
                });
            } else {
                // Show validation errors
                const currentNetworkItem = document.querySelector('.network-item.current-network');
                if (currentNetworkItem) {
                    currentNetworkItem.classList.add('invalid');
                    showNotification('Current network is invalid. Check connections.', 'warning');
                }
            }
        }

        // Add selected detected networks
        if (detectedNetworks && detectedNetworks.length > 0) {
            detectedNetworks.forEach((network, index) => {
                const networkCheckbox = document.getElementById(`network-${index}`);
                if (networkCheckbox && networkCheckbox.checked) {
                    networks.push({
                        id: `detected-${index}`,
                        data: network,
                        mode: executionMode
                    });
                }
            });
        }

        // In the future we can add saved networks here

        return networks;
    }
    
    /**
     * Reset all node styles to their default state
     */
    function resetNodeStyles() {
        // Remove any execution status classes
        document.querySelectorAll('.node').forEach(nodeEl => {
            nodeEl.classList.remove('node-executing', 'node-completed', 'node-failed', 'node-pending');
        });
    }
    
    /**
     * Poll execution status and update UI
     * @param {string} executionId - The execution ID to poll
     * @param {number} interval - Polling interval in milliseconds
     */
    function pollExecutionStatus(executionId, interval = 1000) {
        let pollCount = 0;
        const maxPolls = 60; // Maximum number of times to poll (1 minute at 1-second intervals)
        let pollTimer = null;
        
        // Get UI elements once
        const statusBadge = document.querySelector('.status-badge');
        const progressFill = document.querySelector('.progress-fill');
        const runButton = document.getElementById('run-network-btn');
        const executionDialog = document.getElementById('execution-dialog');
        const tabButtons = executionDialog.querySelectorAll('.tab-btn');
        const tabContents = executionDialog.querySelectorAll('.tab-content');
        
        // Function to check status
        function checkStatus() {
            pollCount++;
            
            // Fetch status from API
            fetch(`/api/workflows/execution-status/${executionId}`)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`Failed to get execution status: ${response.statusText}`);
                    }
                    return response.json();
                })
                .then(data => {
                    const status = data.status;
                    const nodeStatuses = data.node_statuses || {};
                    
                    // Update node status in UI
                    updateNodeExecutionStatus(nodeStatuses);
                    
                    // Add log entry for active nodes
                    Object.values(nodeStatuses).forEach(node => {
                        if (node.status === 'RUNNING' && !node._logged) {
                            addLogEntry(`Executing node: ${node.name || node.id}`);
                            node._logged = true; // Mark as logged to avoid duplicate entries
                        }
                        else if (node.status === 'COMPLETED' && !node._completed_logged) {
                            addLogEntry(`Completed node: ${node.name || node.id}`);
                            node._completed_logged = true;
                        }
                        else if (node.status === 'FAILED' && !node._failed_logged) {
                            addLogEntry(`Failed node: ${node.name || node.id}`, 'error');
                            addLogEntry(`Error: ${node.error || 'Unknown error'}`, 'error');
                            node._failed_logged = true;
                        }
                    });
                    
                    // Update progress based on completed nodes
                    const totalNodes = Object.keys(nodeStatuses).length;
                    if (totalNodes > 0) {
                        const completedNodes = Object.values(nodeStatuses).filter(
                            n => n.status === 'COMPLETED' || n.status === 'FAILED'
                        ).length;
                        
                        const progress = Math.max(15, Math.min(90, (completedNodes / totalNodes) * 100));
                        progressFill.style.width = `${progress}%`;
                    }
                    
                    // Check if execution is complete
                    if (status === 'COMPLETED') {
                        // Update status
                        statusBadge.className = 'status-badge completed';
                        statusBadge.textContent = 'Completed';
                        
                        // Add log entry
                        addLogEntry('Network execution completed successfully');
                        
                        // Update progress to 100%
                        progressFill.style.width = '100%';
                        
                        // Get final result
                        fetchExecutionResult(executionId, data => {
                            showExecutionResult(data, document.getElementById('user-input').value, 
                                             tabButtons, tabContents, runButton);
                        });
                        
                        // Clear polling timer
                        if (pollTimer) {
                            clearTimeout(pollTimer);
                            pollTimer = null;
                        }
                    }
                    else if (status === 'FAILED') {
                        // Update status
                        statusBadge.className = 'status-badge error';
                        statusBadge.textContent = 'Failed';
                        
                        // Add log entry
                        addLogEntry('Network execution failed', 'error');
                        
                        // Update progress to 100%
                        progressFill.style.width = '100%';
                        
                        // Get final result with error
                        fetchExecutionResult(executionId, data => {
                            showExecutionResult(data, document.getElementById('user-input').value, 
                                             tabButtons, tabContents, runButton);
                        });
                        
                        // Clear polling timer
                        if (pollTimer) {
                            clearTimeout(pollTimer);
                            pollTimer = null;
                        }
                    }
                    else if (pollCount >= maxPolls) {
                        // Timeout case
                        statusBadge.className = 'status-badge warning';
                        statusBadge.textContent = 'Timeout';
                        
                        addLogEntry('Execution polling timed out', 'warning');
                        addLogEntry('The execution may still be running in the background', 'info');
                        
                        // Re-enable run button
                        runButton.disabled = false;
                        runButton.textContent = 'Run Again';
                        
                        // Clear polling timer
                        if (pollTimer) {
                            clearTimeout(pollTimer);
                            pollTimer = null;
                        }
                    }
                    else {
                        // Continue polling
                        pollTimer = setTimeout(checkStatus, interval);
                    }
                })
                .catch(error => {
                    console.error('Polling error:', error);
                    addLogEntry(`Status polling error: ${error.message}`, 'error');
                    
                    // If this was just a temporary error, keep polling
                    if (pollCount < maxPolls) {
                        pollTimer = setTimeout(checkStatus, interval);
                    } else {
                        // Re-enable run button
                        runButton.disabled = false;
                        runButton.textContent = 'Run Again';
                    }
                });
        }
        
        // Start polling
        checkStatus();
    }
    
    /**
     * Update node styles based on execution status
     * @param {Object} nodeStatuses - Map of node statuses
     */
    function updateNodeExecutionStatus(nodeStatuses) {
        // Reset non-active nodes first
        document.querySelectorAll('.node').forEach(nodeEl => {
            const nodeId = nodeEl.id;
            
            // Find matching status by UI node ID
            const matchingStatus = Object.values(nodeStatuses).find(
                s => s.ui_node_id === nodeId
            );
            
            if (!matchingStatus) {
                // No status, clear any execution classes
                nodeEl.classList.remove('node-executing', 'node-completed', 'node-failed');
            } else {
                // Update based on status
                nodeEl.classList.remove('node-executing', 'node-completed', 'node-failed', 'node-pending');
                
                if (matchingStatus.status === 'RUNNING') {
                    nodeEl.classList.add('node-executing');
                } else if (matchingStatus.status === 'COMPLETED') {
                    nodeEl.classList.add('node-completed');
                } else if (matchingStatus.status === 'FAILED') {
                    nodeEl.classList.add('node-failed');
                } else if (matchingStatus.status === 'PENDING') {
                    nodeEl.classList.add('node-pending');
                }
            }
        });
    }
    
    /**
     * Fetch the final execution result
     * @param {string} executionId - The execution ID
     * @param {Function} callback - Callback function to receive result
     */
    function fetchExecutionResult(executionId, callback) {
        fetch(`/api/workflows/execution-status/${executionId}`)
            .then(response => response.json())
            .then(data => {
                callback(data);
            })
            .catch(error => {
                console.error('Error fetching result:', error);
                callback({
                    error: `Failed to fetch execution result: ${error.message}`
                });
            });
    }
    
    /**
     * Show execution result in the output tab
     * @param {Object} data - The execution result data
     * @param {string} userInput - The user input
     * @param {NodeList} tabButtons - The tab buttons
     * @param {NodeList} tabContents - The tab contents
     * @param {HTMLButtonElement} runButton - The run button
     */
    function showExecutionResult(data, userInput, tabButtons, tabContents, runButton) {
        // Switch to output tab
        setTimeout(() => {
            tabButtons.forEach(btn => btn.classList.remove('active'));
            tabButtons[3].classList.add('active'); // Output tab (index 3 with Networks tab)

            tabContents.forEach(content => content.classList.add('hidden'));
            document.getElementById('output-tab').classList.remove('hidden');

            // Update output
            const outputContainer = document.getElementById('network-output-container');
            outputContainer.innerHTML = '';

            // Handle multi-network output
            if (data.type === 'multi_network_output' && data.results && Array.isArray(data.results)) {
                // Show a container for all network results
                const networkResultsContainer = document.createElement('div');
                networkResultsContainer.className = 'multi-network-results';

                // Add execution summary
                const summaryDiv = document.createElement('div');
                summaryDiv.className = 'network-execution-summary';
                summaryDiv.innerHTML = `
                    <h3>Multi-Network Execution Summary</h3>
                    <div class="summary-details">
                        <div><strong>Mode:</strong> ${data.mode} execution</div>
                        <div><strong>Networks:</strong> ${data.networks_count} networks</div>
                        <div><strong>Time:</strong> ${data.execution_time ? data.execution_time.toFixed(2) : '?'} seconds</div>
                    </div>
                `;
                networkResultsContainer.appendChild(summaryDiv);

                // Sort results by network index if available
                const sortedResults = [...data.results].sort((a, b) => {
                    return (a.network_index || 0) - (b.network_index || 0);
                });

                // Add each network result
                sortedResults.forEach((networkResult, index) => {
                    const resultItem = document.createElement('div');
                    resultItem.className = 'network-result-item';

                    // Create header
                    const headerDiv = document.createElement('div');
                    headerDiv.className = 'network-result-header';

                    // Add title
                    const titleDiv = document.createElement('div');
                    titleDiv.className = 'network-result-title';
                    titleDiv.innerHTML = `
                        <i class="bi bi-diagram-3"></i>
                        <span>Network ${networkResult.network_index !== undefined ? networkResult.network_index + 1 : index + 1}</span>
                    `;
                    headerDiv.appendChild(titleDiv);

                    // Add execution time if available
                    if (networkResult.execution_time) {
                        const timeDiv = document.createElement('div');
                        timeDiv.className = 'network-result-time';
                        timeDiv.textContent = `${networkResult.execution_time.toFixed(2)}s`;
                        headerDiv.appendChild(timeDiv);
                    }

                    resultItem.appendChild(headerDiv);

                    // Add content or error
                    if (networkResult.error) {
                        // Show error message
                        const errorDiv = document.createElement('div');
                        errorDiv.className = 'network-result-error';
                        errorDiv.textContent = networkResult.error;
                        resultItem.appendChild(errorDiv);
                    } else if (networkResult.result) {
                        // Create content div
                        const contentDiv = document.createElement('div');
                        contentDiv.className = 'network-result-content';

                        // Format the result content
                        const formattedResult = formatResultContent(networkResult.result);
                        contentDiv.appendChild(formattedResult);

                        resultItem.appendChild(contentDiv);
                    }

                    networkResultsContainer.appendChild(resultItem);
                });

                // Add to output container
                outputContainer.appendChild(networkResultsContainer);

                // If in sequential mode, also show the final result
                if (data.mode === 'sequential' && data.result) {
                    const finalResultDiv = document.createElement('div');
                    finalResultDiv.className = 'final-result';
                    finalResultDiv.innerHTML = '<h3>Final Output</h3>';

                    const contentDiv = document.createElement('div');
                    contentDiv.className = 'network-result-content';

                    // Format the final result
                    const formattedResult = formatResultContent(data.result);
                    contentDiv.appendChild(formattedResult);

                    finalResultDiv.appendChild(contentDiv);
                    outputContainer.appendChild(finalResultDiv);
                }
            }
            // Handle error case
            else if (data.error) {
                const outputContent = document.createElement('div');
                outputContent.className = 'output-content error';
                outputContent.textContent = `Error: ${data.error}`;
                outputContainer.appendChild(outputContent);
            }
            // Handle regular single network output
            else if (data.result !== undefined) {
                const outputContent = document.createElement('div');
                outputContent.className = 'output-content';

                // Format the result based on type
                const formattedContent = formatResultContent(data);
                outputContent.appendChild(formattedContent);

                outputContainer.appendChild(outputContent);
            }
            // Fallback - use simulated response for development
            else {
                const outputContent = document.createElement('div');
                outputContent.className = 'output-content';
                outputContent.textContent = simulateResponse(userInput);
                outputContainer.appendChild(outputContent);
            }

            // Re-enable run button
            runButton.disabled = false;
            runButton.textContent = 'Run Again';
        }, 1000);
    }

    /**
     * Format result content based on its type
     * @param {Object|string} data - The result data to format
     * @returns {HTMLElement} - Formatted HTML element with the content
     */
    function formatResultContent(data) {
        const container = document.createElement('div');

        // If data is just a string, treat it as text
        if (typeof data === 'string') {
            container.className = 'formatted-content text';

            // Replace newlines with <br> elements for better display
            const lines = data.split('\n');
            for (let i = 0; i < lines.length; i++) {
                const line = document.createTextNode(lines[i]);
                container.appendChild(line);
                if (i < lines.length - 1) {
                    container.appendChild(document.createElement('br'));
                }
            }
            return container;
        }

        // Get the result data and type
        let resultContent = data.result || data;
        const resultType = data.type || '';
        const resultFormat = data.format || '';

        // Format based on type
        if (resultType === 'json' || resultFormat === 'json' ||
            (typeof resultContent === 'object' && resultContent !== null)) {
            // Display as formatted JSON
            container.className = 'formatted-content json';
            try {
                const formattedJson = typeof resultContent === 'string'
                    ? JSON.stringify(JSON.parse(resultContent), null, 2)
                    : JSON.stringify(resultContent, null, 2);

                // Use a pre element for code formatting
                const pre = document.createElement('pre');
                pre.textContent = formattedJson;
                container.appendChild(pre);
            } catch (e) {
                // Fall back to string representation if JSON parsing fails
                container.textContent = typeof resultContent === 'string'
                    ? resultContent
                    : JSON.stringify(resultContent);
            }
        }
        else if (resultType === 'markdown' || resultFormat === 'markdown') {
            // For markdown, we'd ideally use a markdown renderer
            // For now, we'll use a simple pre element with a markdown class
            container.className = 'formatted-content markdown';

            // Use a pre element to preserve formatting
            const pre = document.createElement('pre');
            pre.textContent = resultContent;
            container.appendChild(pre);
        }
        else if (resultType === 'html' || resultFormat === 'html') {
            // For HTML content, use innerHTML
            container.className = 'formatted-content html';
            container.innerHTML = resultContent;
        }
        else {
            // Default to text
            container.className = 'formatted-content text';

            // Check if result is a string or object
            if (typeof resultContent === 'string') {
                // Replace newlines with <br> elements for better display
                const lines = resultContent.split('\n');
                for (let i = 0; i < lines.length; i++) {
                    const line = document.createTextNode(lines[i]);
                    container.appendChild(line);
                    if (i < lines.length - 1) {
                        container.appendChild(document.createElement('br'));
                    }
                }
            } else {
                // For objects, display as formatted JSON
                const pre = document.createElement('pre');
                try {
                    pre.textContent = JSON.stringify(resultContent, null, 2);
                } catch (e) {
                    pre.textContent = String(resultContent);
                }
                container.appendChild(pre);
            }
        }

        return container;
    }
    
    /**
     * Add a log entry to the execution log
     * @param {string} message - The log message
     * @param {string} type - The type of log entry (info, warning, error)
     */
    function addLogEntry(message, type = 'info') {
        const logContainer = document.getElementById('execution-log-container');
        const logEntry = document.createElement('div');
        logEntry.className = `log-entry ${type}`;
        
        const timestamp = new Date().toLocaleTimeString();
        logEntry.innerHTML = `<span class="log-time">[${timestamp}]</span> ${message}`;
        
        logContainer.appendChild(logEntry);
        logContainer.scrollTop = logContainer.scrollHeight;
        
        // Update progress bar
        const progressFill = document.querySelector('.progress-fill');
        const currentWidth = parseInt(progressFill.style.width) || 15;
        const newWidth = Math.min(currentWidth + Math.random() * 20, 90);
        progressFill.style.width = `${newWidth}%`;
    }
    
    /**
     * Show a notification message
     * @param {string} message - The notification message
     * @param {string} type - The type of notification (info, success, warning, error)
     */
    function showNotification(message, type = 'info', duration = 4000) {
        // Check if notification container exists
        let notificationContainer = document.getElementById('notification-container');

        if (!notificationContainer) {
            notificationContainer = document.createElement('div');
            notificationContainer.id = 'notification-container';
            document.body.appendChild(notificationContainer);
        }

        // Create notification
        const notification = document.createElement('div');
        notification.className = `notification ${type}`;

        const iconMap = {
            info: 'info-circle-fill',
            success: 'check-circle-fill',
            warning: 'exclamation-triangle-fill',
            error: 'x-circle-fill'
        };

        notification.innerHTML = `
            <div class="notification-icon">
                <i class="bi bi-${iconMap[type]}"></i>
            </div>
            <div class="notification-content">
                <p>${message}</p>
            </div>
            <button class="notification-close">
                <i class="bi bi-x"></i>
            </button>
        `;
        
        // Add to container
        notificationContainer.appendChild(notification);
        
        // Add close button functionality
        const closeButton = notification.querySelector('.notification-close');
        closeButton.addEventListener('click', () => {
            notification.classList.add('closing');
            setTimeout(() => {
                notificationContainer.removeChild(notification);
            }, 300);
        });
        
        // Auto-dismiss after a delay (use shorter time for success/info, longer for warnings, no auto-dismiss for errors)
        const dismissDelay = type === 'error' ? 0 : // Don't auto-dismiss errors
                            type === 'warning' ? 7000 : // Longer time for warnings
                            type === 'success' ? 3000 : 4000; // Shorter for success/info

        if (dismissDelay > 0) {
            setTimeout(() => {
                if (notification.parentNode === notificationContainer) {
                    notification.classList.add('closing');
                    setTimeout(() => {
                        if (notification.parentNode === notificationContainer) {
                            notificationContainer.removeChild(notification);
                        }
                    }, 300);
                }
            }, dismissDelay);
        }
    }
    
    /**
     * Validate the network before running
     * @returns {Object} - Validation result with valid flag and errors array
     */
    function validateNetwork() {
        const errors = [];
        
        // Check for input and output nodes
        const inputNodes = nodes.filter(node => node.type === 'input');
        const outputNodes = nodes.filter(node => node.type === 'output');
        
        if (inputNodes.length === 0) {
            errors.push('Please add an input node to your network');
        }
        
        if (outputNodes.length === 0) {
            errors.push('Please add an output node to your network');
        }
        
        // Check for connections between nodes
        if (connections.length === 0) {
            errors.push('Please connect nodes in your network');
        }
        
        // Validate agent configurations
        for (const node of nodes) {
            if (node.type === 'agent') {
                const agentType = node.subType;
                const config = node.config || {};
                
                // Check that the node has a name
                if (!config.name || config.name.trim() === '') {
                    errors.push(`Agent node #${node.id.split('-')[1]} needs a name`);
                }
                
                // Validate based on agent type
                if (agentType === 'openai') {
                    if (!config.apiKey || config.apiKey.trim() === '') {
                        errors.push(`OpenAI agent ${config.name || 'Unnamed'} is missing API key`);
                    }
                    if (!config.model) {
                        errors.push(`OpenAI agent ${config.name || 'Unnamed'} is missing model selection`);
                    }
                }
                else if (agentType === 'ollama') {
                    if (!config.apiUrl || config.apiUrl.trim() === '') {
                        errors.push(`Ollama agent ${config.name || 'Unnamed'} is missing API url`);
                    }
                    if (!config.apiKey || config.apiKey.trim() === '') {
                        errors.push(`Ollama agent ${config.name || 'Unnamed'} is missing API key`);
                    }
                    if (!config.model) {
                        errors.push(`Ollama agent ${config.name || 'Unnamed'} is missing model definition`);
                    }
                }
                else if (agentType === 'anthropic') {
                    if (!config.apiKey || config.apiKey.trim() === '') {
                        errors.push(`Claude agent ${config.name || 'Unnamed'} is missing API key`);
                    }
                    if (!config.model) {
                        errors.push(`Claude agent ${config.name || 'Unnamed'} is missing model selection`);
                    }
                }
                else if (agentType === 'bedrock') {
                    if (!config.accessKey || config.accessKey.trim() === '' || 
                        !config.secretKey || config.secretKey.trim() === '') {
                        errors.push(`Bedrock agent ${config.name || 'Unnamed'} is missing AWS credentials`);
                    }
                    if (!config.region) {
                        errors.push(`Bedrock agent ${config.name || 'Unnamed'} is missing AWS region`);
                    }
                    if (!config.model) {
                        errors.push(`Bedrock agent ${config.name || 'Unnamed'} is missing model selection`);
                    }
                }
                else if (agentType === 'custom') {
                    if ((!config.port || config.port === '') && 
                        (!config.endpoint || config.endpoint.trim() === '')) {
                        errors.push(`Custom agent ${config.name || 'Unnamed'} needs either a port or endpoint URL`);
                    }
                }
            }
        }
        
        // Check for isolated nodes (no connections)
        const connectedNodeIds = new Set();
        connections.forEach(conn => {
            connectedNodeIds.add(conn.start.nodeId);
            connectedNodeIds.add(conn.end.nodeId);
        });
        
        for (const node of nodes) {
            if (!connectedNodeIds.has(node.id)) {
                errors.push(`Node ${node.config?.name || node.type} is not connected to any other node`);
            }
        }
        
        // Check if the network forms a complete path from input to output
        // This is a simplified check - a full check would need graph traversal
        const inputNodeIds = inputNodes.map(node => node.id);
        const outputNodeIds = outputNodes.map(node => node.id);
        
        // Check if at least one input node is connected
        let inputConnected = false;
        for (const inputId of inputNodeIds) {
            if (connectedNodeIds.has(inputId)) {
                inputConnected = true;
                break;
            }
        }
        
        if (!inputConnected && inputNodeIds.length > 0) {
            errors.push('Input node is not connected to the network');
        }
        
        // Check if at least one output node is connected
        let outputConnected = false;
        for (const outputId of outputNodeIds) {
            if (connectedNodeIds.has(outputId)) {
                outputConnected = true;
                break;
            }
        }
        
        if (!outputConnected && outputNodeIds.length > 0) {
            errors.push('Output node is not connected to the network');
        }
        
        return {
            valid: errors.length === 0,
            errors: errors
        };
    }
    
    /**
     * Simulate a response for development purposes
     * @param {string} input - The user input
     * @returns {string} - The simulated response
     */
    function simulateResponse(input) {
        const responses = [
            `I've processed your request: "${input.substring(0, 30)}${input.length > 30 ? '...' : ''}"\n\nBased on my analysis, I can provide the following information:\n\n1. The request appears to be about ${input.split(' ').slice(0, 3).join(' ')}...\n2. I've identified key concepts that relate to your question\n3. My recommendation is to focus on the primary aspects mentioned in your request`,
            
            `Thank you for your query about "${input.substring(0, 20)}${input.length > 20 ? '...' : ''}"\n\nHere's what I found:\n• The main topic involves ${input.split(' ').slice(-3).join(' ')}\n• There are several factors to consider\n• Based on available information, the most effective approach would be to analyze further`,
            
            `I've analyzed your request regarding "${input.substring(0, 25)}${input.length > 25 ? '...' : ''}"\n\nMy findings:\n1. This appears to be related to ${input.split(' ')[0]} technology\n2. Current trends suggest a growing interest in this area\n3. For best results, consider exploring advanced techniques\n\nWould you like me to elaborate on any specific aspect?`
        ];
        
        // Get a "random" response based on the input length
        const index = input.length % responses.length;
        return responses[index];
    }
    
    /**
     * Undo the last action
     */
    function undoAction() {
        // TODO: Implement undo functionality
        alert('Undo functionality will be implemented in the next phase.');
    }
    
    /**
     * Redo the last undone action
     */
    function redoAction() {
        // TODO: Implement redo functionality
        alert('Redo functionality will be implemented in the next phase.');
    }
    
    // Handle window mousemove and mouseup events for connection creation only
    window.addEventListener('mousemove', (e) => {
        // Node dragging is now handled in the node-specific event handler
        
        // Handle connection creation
        if (connectionStart) {
            const canvasRect = canvas.getBoundingClientRect();
            const mouseX = (e.clientX - canvasRect.left - canvasOffset.x) / canvasScale;
            const mouseY = (e.clientY - canvasRect.top - canvasOffset.y) / canvasScale;
            
            updateTempConnection(mouseX, mouseY);
            
            // Update tooltip position
            connectionTooltip.style.left = `${e.clientX}px`;
            connectionTooltip.style.top = `${e.clientY - 30}px`;
        }
    });
    
    window.addEventListener('mouseup', () => {
        // End node dragging
        if (isDragging) {
            isDragging = false;
        }
        
        // End connection creation if not completed
        if (connectionStart && !connectionEnd) {
            const tempConnection = document.getElementById('temp-connection');
            if (tempConnection) {
                svgContainer.removeChild(tempConnection);
            }
            
            connectionStart = null;
            connectionTooltip.style.opacity = '0';
        }
    });
});