"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../../lib");
const physical_name_generator_1 = require("../../lib/private/physical-name-generator");
class TestResource extends lib_1.Resource {
}
module.exports = nodeunit.testCase({
    generatePhysicalName: {
        'generates correct physical names'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
            const testResourceA = new TestResource(stack, 'A');
            const testResourceB = new TestResource(testResourceA, 'B');
            test.equal(physical_name_generator_1.generatePhysicalName(testResourceA), 'teststackteststackaa164c141d59b37c1b663');
            test.equal(physical_name_generator_1.generatePhysicalName(testResourceB), 'teststackteststackab27595cd34d8188283a1f');
            test.done();
        },
        'generates different names in different accounts'(test) {
            const appA = new lib_1.App();
            const stackA = new lib_1.Stack(appA, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
            const resourceA = new TestResource(stackA, 'Resource');
            const appB = new lib_1.App();
            const stackB = new lib_1.Stack(appB, 'TestStack', { env: { account: '012345678913', region: 'bermuda-triangle-1' } });
            const resourceB = new TestResource(stackB, 'Resource');
            test.notEqual(physical_name_generator_1.generatePhysicalName(resourceA), physical_name_generator_1.generatePhysicalName(resourceB));
            test.done();
        },
        'generates different names in different regions'(test) {
            const appA = new lib_1.App();
            const stackA = new lib_1.Stack(appA, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
            const resourceA = new TestResource(stackA, 'Resource');
            const appB = new lib_1.App();
            const stackB = new lib_1.Stack(appB, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-2' } });
            const resourceB = new TestResource(stackB, 'Resource');
            test.notEqual(physical_name_generator_1.generatePhysicalName(resourceA), physical_name_generator_1.generatePhysicalName(resourceB));
            test.done();
        },
        'fails when the region is an unresolved token'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912', region: lib_1.Aws.REGION } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the region is un-resolved or missing/);
            test.done();
        },
        'fails when the region is not provided'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912' } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the region is un-resolved or missing/);
            test.done();
        },
        'fails when the account is an unresolved token'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: lib_1.Aws.ACCOUNT_ID, region: 'bermuda-triangle-1' } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the account is un-resolved or missing/);
            test.done();
        },
        'fails when the account is not provided'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { region: 'bermuda-triangle-1' } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the account is un-resolved or missing/);
            test.done();
        },
    },
    GeneratedWhenNeededMarker: {
        'is correctly recognized'(test) {
            const marker = new physical_name_generator_1.GeneratedWhenNeededMarker();
            const asString = lib_1.Token.asString(marker);
            test.ok(physical_name_generator_1.isGeneratedWhenNeededMarker(asString));
            test.done();
        },
        'throws when resolved'(test) {
            const marker = new physical_name_generator_1.GeneratedWhenNeededMarker();
            const asString = lib_1.Token.asString(marker);
            test.throws(() => new lib_1.Stack().resolve(asString), /Use "this.physicalName" instead/);
            test.done();
        },
    },
    isGeneratedWhenNeededMarker: {
        'correctly response for other tokens'(test) {
            test.ok(!physical_name_generator_1.isGeneratedWhenNeededMarker('this is not even a token!'));
            test.ok(!physical_name_generator_1.isGeneratedWhenNeededMarker(lib_1.Lazy.stringValue({ produce: () => 'Bazinga!' })));
            test.done();
        }
    },
});
//# sourceMappingURL=data:application/json;base64,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