"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const util_1 = require("./util");
module.exports = {
    'addFileAsset correctly sets metadata and creates S3 parameters'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        stack.addFileAsset({
            fileName: 'file-name',
            packaging: lib_1.FileAssetPackaging.ZIP_DIRECTORY,
            sourceHash: 'source-hash'
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxapi.ASSET_METADATA);
        test.equal(assetMetadata && assetMetadata.data.path, 'file-name');
        test.equal(assetMetadata && assetMetadata.data.id, 'source-hash');
        test.equal(assetMetadata && assetMetadata.data.packaging, lib_1.FileAssetPackaging.ZIP_DIRECTORY);
        test.equal(assetMetadata && assetMetadata.data.sourceHash, 'source-hash');
        test.deepEqual(util_1.toCloudFormation(stack), {
            Parameters: {
                AssetParameterssourcehashS3BucketE6E91E3E: {
                    Type: 'String',
                    Description: 'S3 bucket for asset "source-hash"'
                },
                AssetParameterssourcehashS3VersionKeyAC4157C3: {
                    Type: 'String',
                    Description: 'S3 key for asset version "source-hash"'
                },
                AssetParameterssourcehashArtifactHashADBAE418: {
                    Type: 'String',
                    Description: 'Artifact hash for asset "source-hash"'
                }
            }
        });
        test.done();
    },
    'addDockerImageAsset correctly sets metadata'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        stack.addDockerImageAsset({
            sourceHash: 'source-hash',
            directoryName: 'directory-name',
            repositoryName: 'repository-name'
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxapi.ASSET_METADATA);
        test.equal(assetMetadata && assetMetadata.data.packaging, 'container-image');
        test.equal(assetMetadata && assetMetadata.data.path, 'directory-name');
        test.equal(assetMetadata && assetMetadata.data.sourceHash, 'source-hash');
        test.equal(assetMetadata && assetMetadata.data.repositoryName, 'repository-name');
        test.equal(assetMetadata && assetMetadata.data.imageTag, 'source-hash');
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
    'addDockerImageAsset uses the default repository name'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        stack.addDockerImageAsset({
            sourceHash: 'source-hash',
            directoryName: 'directory-name',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxapi.ASSET_METADATA);
        test.equal(assetMetadata && assetMetadata.data.packaging, 'container-image');
        test.equal(assetMetadata && assetMetadata.data.path, 'directory-name');
        test.equal(assetMetadata && assetMetadata.data.sourceHash, 'source-hash');
        test.equal(assetMetadata && assetMetadata.data.repositoryName, 'aws-cdk/assets');
        test.equal(assetMetadata && assetMetadata.data.imageTag, 'source-hash');
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
    'addDockerImageAsset supports overriding repository name through a context key as a workaround until we have API for that'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.node.setContext('assets-ecr-repository-name', 'my-custom-repo-name');
        // WHEN
        stack.addDockerImageAsset({
            sourceHash: 'source-hash',
            directoryName: 'directory-name',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxapi.ASSET_METADATA);
        test.equal(assetMetadata && assetMetadata.data.packaging, 'container-image');
        test.equal(assetMetadata && assetMetadata.data.path, 'directory-name');
        test.equal(assetMetadata && assetMetadata.data.sourceHash, 'source-hash');
        test.equal(assetMetadata && assetMetadata.data.repositoryName, 'my-custom-repo-name');
        test.equal(assetMetadata && assetMetadata.data.imageTag, 'source-hash');
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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