"use strict";
/**
 * Constructs compatibility layer.
 *
 * This file includes types that shadow types in the "constructs" module in
 * order to allow backwards-compatiblity in the AWS CDK v1.0 release line.
 *
 * There are pretty ugly hacks here, which mostly involve downcasting types to
 * adhere to legacy AWS CDK APIs.
 *
 * This file, in its entirety, is expected to be removed in v2.0.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const constructs = require("constructs");
const ORIGINAL_CONSTRUCT_NODE_SYMBOL = Symbol.for('@aws-cdk/core.ConstructNode');
const CONSTRUCT_SYMBOL = Symbol.for('@aws-cdk/core.Construct');
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct extends constructs.Construct {
    constructor(scope, id) {
        super(scope, id, {
            nodeFactory: {
                createNode: (h, s, i) => new ConstructNode(h, s, i)._actualNode
            }
        });
        Object.defineProperty(this, CONSTRUCT_SYMBOL, { value: true });
        this.node = ConstructNode._unwrap(constructs.Node.of(this));
        const disableTrace = this.node.tryGetContext(cxapi.DISABLE_METADATA_STACK_TRACE);
        if (disableTrace) {
            this.node.setContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA, true);
        }
    }
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x) {
        return typeof x === 'object' && x !== null && CONSTRUCT_SYMBOL in x;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     */
    onValidate() {
        return this.validate();
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    onPrepare() {
        this.prepare();
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    onSynthesize(session) {
        this.synthesize({
            outdir: session.outdir,
            assembly: session.assembly
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     */
    validate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    prepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    synthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
/**
 * Represents the construct node in the scope tree.
 */
class ConstructNode {
    constructor(host, scope, id) {
        this._actualNode = new constructs.Node(host, scope, id);
        // store a back reference on _actualNode so we can our ConstructNode from it
        Object.defineProperty(this._actualNode, ORIGINAL_CONSTRUCT_NODE_SYMBOL, {
            value: this,
            configurable: false,
            enumerable: false
        });
    }
    /**
     * Returns the wrapping `@aws-cdk/core.ConstructNode` instance from a `constructs.ConstructNode`.
     *
     * @internal
     */
    static _unwrap(c) {
        const x = c[ORIGINAL_CONSTRUCT_NODE_SYMBOL];
        if (!x) {
            throw new Error(`invalid ConstructNode type`);
        }
        return x;
    }
    /**
     * Synthesizes a CloudAssembly from a construct tree.
     * @param root The root of the construct tree.
     * @param options Synthesis options.
     */
    static synth(root, options = {}) {
        const builder = new cxapi.CloudAssemblyBuilder(options.outdir);
        root._actualNode.synthesize({
            outdir: builder.outdir,
            skipValidation: options.skipValidation,
            sessionContext: {
                assembly: builder
            }
        });
        return builder.buildAssembly(options);
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     * @param node The root node
     */
    static prepare(node) {
        return node._actualNode.prepare();
    }
    /**
     * Invokes "validate" on all constructs in the tree (depth-first, pre-order) and returns
     * the list of all errors. An empty list indicates that there are no errors.
     *
     * @param node The root node
     */
    static validate(node) {
        return node._actualNode.validate().map(e => ({ source: e.source, message: e.message }));
    }
    /**
     * Returns the scope in which this construct is defined.
     *
     * The value is `undefined` at the root of the construct scope tree.
     */
    get scope() {
        return this._actualNode.scope;
    }
    /**
     * The id of this construct within the current scope.
     *
     * This is a a scope-unique id. To obtain an app-unique id for this construct, use `uniqueId`.
     */
    get id() { return this._actualNode.id; }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() { return this._actualNode.path; }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() { return this._actualNode.uniqueId; }
    /**
     * Return a direct child by id, or undefined
     *
     * @param id Identifier of direct child
     * @returns the child if found, or undefined
     */
    tryFindChild(id) { return this._actualNode.tryFindChild(id); }
    /**
     * Return a direct child by id
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child
     * @returns Child with the given id.
     */
    findChild(id) { return this._actualNode.findChild(id); }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     *
     * @throws if there is more than one child
     * @returns a construct or undefined if there is no default child
     */
    get defaultChild() { return this._actualNode.defaultChild; }
    /**
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     */
    set defaultChild(value) { this._actualNode.defaultChild = value; }
    /**
     * All direct children of this construct.
     */
    get children() { return this._actualNode.children; }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PREORDER) { return this._actualNode.findAll(order); }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) { this._actualNode.setContext(key, value); }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or `undefined` if there is no context value for thie key.
     */
    tryGetContext(key) { return this._actualNode.tryGetContext(key); }
    /**
     * An immutable array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() { return this._actualNode.metadata; }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, fromFunction) { this._actualNode.addMetadata(type, data, fromFunction); }
    /**
     * Adds a { "info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     */
    addInfo(message) {
        this._actualNode.addMetadata(cxapi.INFO_METADATA_KEY, message);
    }
    /**
     * Adds a { "warning": <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     */
    addWarning(message) {
        this._actualNode.addMetadata(cxapi.WARNING_METADATA_KEY, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this._actualNode.addMetadata(cxapi.ERROR_METADATA_KEY, message);
    }
    /**
     * Applies the aspect to this Constructs node
     */
    applyAspect(aspect) { this._actualNode.applyAspect(aspect); }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     */
    get scopes() { return this._actualNode.scopes; }
    /**
     * @returns The root of the construct tree.
     */
    get root() { return this._actualNode.root; }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() { return this._actualNode.locked; }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) { this._actualNode.addDependency(...dependencies); }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    get dependencies() { return this._actualNode.dependencies; }
    /**
     * Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) { return this._actualNode.tryRemoveChild(childName); }
}
exports.ConstructNode = ConstructNode;
/**
 * Separator used to delimit construct path components.
 */
ConstructNode.PATH_SEP = '/';
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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