"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x) { return resolve(x, { ...options, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return options.resolver.resolveString(fragments, makeContext()[0]);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) !== 'string') {
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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