"""Default class for joints, i.e. points in space at a specific timestamp. The methods in this class are mainly
handled by the methods in the class Pose, but some of them can be directly accessed."""
import numpy as np
from numpy import cos, sin, deg2rad
from krajjat.classes.exceptions import InvalidParameterValueException


class Joint(object):
    """Creates a Joint instance, with a joint label and x, y, and z coordinates.

    .. versionadded:: 2.0

    Parameters
    ----------
    joint_label: str, optional
        The label of the joint (e.g. ``"Head"``).
    x: float, optional
        The position of the joint on the x-axis (in meters).
    y: float, optional
        The position of the joint on the y-axis (in meters).
    z: float, optional
        The position of the joint on the z-axis (in meters).

    Attributes
    ----------
    joint_label: str
        The label of the joint (e.g. ``"Head"``).
    x: float
        The position of the joint on the x-axis (in meters).
    y: float
        The position of the joint on the y-axis (in meters).
    z: float
        The position of the joint on the z-axis (in meters).
    _velocity_over_threshold: bool
        Defines if the velocity of this joint, compared to the previous joint, has been found to be over threshold
        defined in the parameters :meth:`.Sequence.correct_jitter`.
    _interpolated: bool
        Defines if the coordinates of this joint have been modified by :meth:`.Sequence.correct_zeros`.
    _dejittered: bool
        Defines if the coordinates of this joint have been modified by :meth:`.Sequence.correct_jitter`.
    _rereferenced: bool
        Defines if the coordinates of this joint have been modified by :meth:`.Sequence.re_reference`.
    _randomized: bool
        Defines if the coordinates of this joint have been randomly generated by (typically, by
        :meth:`.Sequence.randomize`).

    Example
    -------
    >>> j = Joint("Head", 4.8, 15.16, 23.42)
    """

    def __init__(self, joint_label=None, x=None, y=None, z=None):

        self.joint_label = joint_label
        self.x = None
        self.y = None
        self.z = None
        self.set_position(x, y, z)

        self._velocity_over_threshold = False
        self._interpolated = False
        self._dejittered = False
        self._rereferenced = False
        self._randomized = False

    # === Setter functions ===

    def set_joint_label(self, joint_label):
        """Sets the :attr:`joint_label` attribute of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        joint_label: str
            The label of the joint (e.g. ``"Head"``).

        Example
        -------
        >>> joint = Joint()
        >>> joint.set_joint_label("Hand")
        >>> print(joint.joint_label)
        "Hand"
        """
        self.joint_label = joint_label

    def set_x(self, x):
        """Sets the :attr:`x` coordinate of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        x: float|None
            The position of the joint on the x-axis.

        Example
        -------
        >>> joint = Joint("Head", 0, 0, 0)
        >>> joint.set_x(4)
        >>> print(joint.get_position())
        (4, 0, 0)
        """
        if x is not None:
            self.x = float(x)
        else:
            self.x = None

    def set_y(self, y):
        """Sets the :attr:`y` coordinate of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        y: float|None
            The position of the joint on the y-axis.

        Example
        -------
        >>> joint = Joint("Head", 0, 0, 0)
        >>> joint.set_y(8)
        >>> print(joint.get_position())
        (0, 8, 0)
        """
        if y is not None:
            self.y = float(y)
        else:
            self.y = None

    def set_z(self, z):
        """Sets the :attr:`z` coordinate of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        z: float|None
            The position of the joint on the z-axis.

        Example
        -------
        >>> joint = Joint("Head", 0, 0, 0)
        >>> joint.set_z(15)
        >>> print(joint.get_position())
        (0, 0, 15)
        """
        if z is not None:
            self.z = float(z)
        else:
            self.z = None

    def set_coordinate(self, axis, value):
        """Sets the coordinate on the specified axis (x, y, or z).

        .. versionadded:: 2.0

        Parameters
        ----------
        axis: str
            The axis (``"x"``, ``"y"``, or ``"z"``) on which to set the coordinate.
        value: float|None
            The value of the coordinate.

        Example
        -------
        >>> joint = Joint("Head", 0, 0, 0)
        >>> joint.set_coordinate("x", 1)
        >>> joint.x
        1
        """
        if axis == "x":
            self.set_x(value)
        elif axis == "y":
            self.set_y(value)
        elif axis == "z":
            self.set_z(value)
        else:
            raise InvalidParameterValueException("axis", axis, ["x", "y", "z"])

    def set_position(self, x, y, z):
        """Sets the :attr:`x`, :attr:`y` and :attr:`z` coordinates of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        x: float|None
            The position of the joint on the x-axis.
        y: float|None
            The position of the joint on the y-axis.
        z: float|None
            The position of the joint on the z-axis.

        Example
        -------
        >>> joint = Joint()
        >>> joint.set_position(4, 8, 15)
        """

        self.set_x(x)
        self.set_y(y)
        self.set_z(z)

    def set_to_zero(self):
        """Sets the joints coordinates to (0, 0, 0).

        .. versionadded:: 2.0

        Example
        -------
        >>> joint = Joint()
        >>> joint.set_to_zero()
        >>> print(joint.get_position())
        (0, 0, 0)
        """
        self.x = 0.0
        self.y = 0.0
        self.z = 0.0

    def set_to_none(self):
        """Sets the joints coordinates to (None, None, None).

        .. versionadded:: 2.0

        Example
        -------
        >>> joint = Joint()
        >>> joint.set_to_none()
        >>> print(joint.get_position())
        (None, None, None)
        """
        self.x = None
        self.y = None
        self.z = None

    # === Getter functions ===

    def get_joint_label(self):
        """Returns the :attr:`joint_label` attribute.

        .. versionadded:: 2.0

        Returns
        -------
        str
            The label of the joint (e.g. ``"Head"``).

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.get_joint_label()
        "Head"
        """
        return self.joint_label

    def get_x(self):
        """Returns the :attr:`x` coordinate of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The x coordinate of the joint (in meters).

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.get_x()
        1
        """
        return self.x

    def get_y(self):
        """Returns the :attr:`y` coordinate of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The y coordinate of the joint (in meters).

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.get_y()
        2
        """
        return self.y

    def get_z(self):
        """Returns the :attr:`z` coordinate of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The z coordinate of the joint (in meters).

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.get_z()
        3
        """
        return self.z

    def get_coordinate(self, axis):
        """Returns the coordinate on the specified axis (x, y, or z).

        .. versionadded:: 2.0

        Parameters
        ----------
        axis: str
            The axis (``"x"``, ``"y"``, or ``"z"``) from which to get the coordinate.

        Returns
        -------
        float
            The x, y or z coordinate of the joint (in meters).

        Raises
        ------
        InvalidParameterValueException
            If the axis is neither ``"x"``, ``"y"`` nor ``"z"``.

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.get_coordinate("x")
        1
        >>> joint.get_coordinate("z")
        3
        """
        if axis == "x":
            return self.x
        elif axis == "y":
            return self.y
        elif axis == "z":
            return self.z
        else:
            raise InvalidParameterValueException("axis", axis, ["x", "y", "z"])

    def get_position(self):
        """Returns a list containing the :attr:`x`, :attr:`y` and :attr:`z` coordinates of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The x coordinate of the joint (in meters).

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.get_position()
        (1, 2, 3)
        """
        return self.x, self.y, self.z

    # === Predicate functions ===

    def has_velocity_over_threshold(self):
        """Returns the value of the attribute :attr:`_has_velocity_over_threshold`, which will be `True` if the
        distance travelled by this joint over time, compared to the previous joint, has been found to be over threshold
        defined in the parameters of :meth:`Sequence.correct_jitter()`. This value allows to show the joint in color
        when displaying the sequence in the function :meth:`display_functions.compare_sequences()`.

        .. versionadded:: 2.0

        Returns
        -------
        bool
            Value of the attribute :attr:`_has_velocity_over_threshold`.
        """
        return self._velocity_over_threshold

    def is_corrected(self):
        """Returns `True` if at least one of the values of the attributes :attr:`_zero_corrected`,
        :attr:`_dejittered`, or :attr:`_rereferenced` is `True`. These parameters will be `True` if they have been
        modified by either :meth:`Sequence.correct_zeros()`, :meth:`Sequence.correct_jitter()`, or
        :meth:`.Sequence.re_reference()` respectively.

        .. versionadded:: 2.0

        Returns
        -------
        bool
            Value of the attribute :attr:`_is_corrected`.
        """
        return self._interpolated or self._dejittered or self._rereferenced

    def is_randomized(self):
        """Returns the value of the attribute :attr:`_is_randomized`, which will be `True` if the coordinates of this
        joint have been randomly generated by :meth:`.Sequence.randomize()`.

        .. versionadded:: 2.0

        Returns
        -------
        bool
            Value of the attribute :attr:`_is_randomized`.
        """
        return self._randomized

    def is_zero(self):
        """Returns `True` if the coordinates of the joint are (0, 0, 0).

        ... versionadded:: 2.0

        Returns
        -------
        bool
            `True` if the coordinates of the joint are (0, 0, 0).

        Examples
        --------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.is_zero()
        False
        >>> joint = Joint("Head", 0, 0, 0)
        >>> joint.is_zero()
        True
        >>> joint = Joint()
        >>> joint.is_zero()  # Joints are None
        False
        """
        if self.x == 0 and self.y == 0 and self.z == 0:
            return True
        return False

    def is_none(self):
        """Returns `True` if the coordinates of the joint are (None, None, None).

        ... versionadded:: 2.0

        Returns
        -------
        bool
            `True` if the coordinates of the joint are (None, None, None).

        Examples
        --------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint.is_none()
        False
        >>> joint = Joint("Head", 0, 0, 0)
        >>> joint.is_none()
        False
        >>> joint = Joint()
        >>> joint.is_none()
        True
        """
        if self.x is None and self.y is None and self.z is None:
            return True
        return False

    # === Other methods ===
    def copy(self):
        """Returns a deep copy of itself.

        .. versionadded:: 2.0

        Returns
        -------
        Joint
            A deep copy of the Joint instance.
        """

        j = Joint(self.joint_label, self.x, self.y, self.z)
        j._velocity_over_threshold = self._velocity_over_threshold
        j._dejittered = self._dejittered
        j._interpolated = self._interpolated
        j._rereferenced = self._rereferenced
        j._randomized = self._randomized
        return j

    def rotate(self, yaw=0, pitch=0, roll=0):
        """Returns converted coordinates given three rotations: yaw, pitch and roll.

        .. versionadded:: 2.0

        Warning
        -------
        This function is experimental as of version 2.0.

        Parameters
        ----------
        yaw: float, optional
            The angle of yaw, or rotation on the z-axis, in degrees (default: 0).
        pitch: float, optional
            The angle of pitch, or rotation on the y-axis, in degrees (default: 0).
        roll: float, optional
            The angle of roll, or rotation on the x-axis, in degrees (default: 0).

        Returns
        -------
        float
            The converted x coordinate.
        float
            The converted y coordinate.
        float
            The converted z coordinate.

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 0)
        >>> joint.rotate(90, 0, 0)
        (-2, 1, 0)
        """

        # Convert to degrees
        yaw = deg2rad(yaw)
        pitch = deg2rad(pitch)
        roll = deg2rad(roll)

        x = self.x * cos(yaw) * cos(pitch) + \
            self.y * (cos(yaw) * sin(pitch) * sin(roll) - sin(yaw) * cos(roll)) + \
            self.z * (cos(yaw) * sin(pitch) * cos(roll) - sin(yaw) * sin(roll))
        y = self.x * sin(yaw) * cos(pitch) + \
            self.y * (sin(yaw) * sin(pitch) * sin(roll) + cos(yaw) * cos(roll)) + \
            self.z * (sin(yaw) * sin(pitch) * sin(roll) * cos(yaw) * sin(roll))
        z = self.x * -sin(pitch) + \
            self.y * cos(pitch) * sin(roll) + \
            self.z * cos(pitch) * cos(roll)

        return x, y, z

    # === Private methods ===
    def _randomize_coordinates_keep_movement(self, joint_pose_0, joint_random, verbosity=1):
        """Returns a joint with modified coordinates while keeping the relative position compared to the position of the
        joint at the beginning of the sequence. This function is used by `.Sequence.randomize()`.

        .. versionadded:: 2.0

        Parameters
        ----------
        joint_pose_0: Joint
            The joint from the first pose of the sequence, with the same :attr:`joint_label`.
        joint_random: Joint
            A joint with random coordinates, generated by :meth:`.Sequence.randomize()`
        verbosity: int, optional
            Sets how much feedback the code will provide in the console output:

            • *0: Silent mode.* The code won’t provide any feedback, apart from error messages.
            • *1: Normal mode* (default). The code will provide essential feedback such as progression markers and
              current steps.
            • *2: Chatty mode.* The code will provide all possible information on the events happening. Note that this
              may clutter the output and slow down the execution.

        Returns
        -------
        Joint
            The original joint with added random coordinates.
        """

        if verbosity > 1:
            print(f"Coordinates before randomization: {self.x}, {self.y}, {self.z}")
        x = joint_random.x + (self.x - joint_pose_0.x)
        y = joint_random.y + (self.y - joint_pose_0.y)
        z = joint_random.z + (self.z - joint_pose_0.z)
        j = Joint(self.joint_label, x, y, z)
        j._randomized = True
        if verbosity > 1:
            print(f"Coordinates after randomization: {self.x}, {self.y}, {self.z}")

        return j

    # === Magic methods ===

    def __repr__(self):
        """Returns a string containing the joint label, the x, y ans z coordinates, and adds information if one or more
        of the private attributes :attr:`_has_velocity_over_threshold`, :attr:`_is_corrected` or :attr:`_is_randomized`
        are `True`.

        Returns
        -------
        str
            A formatted string of the information contained in all the attributes of the object.

        Examples
        --------
        >>> joint = Joint("Head", 4.8, 15.16, 23.42)
        >>> print(joint)
        Head: (4.8, 15.16, 23.42)
        """
        txt = str(self.joint_label) + ": (" + str(self.x) + ", " + str(self.y) + ", " + str(self.z) + ")"
        append_txt = []
        if self._velocity_over_threshold:
            append_txt.append("Over threshold")
        if self.is_corrected():
            append_txt.append("Corrected")
            corrected_txt = []
            if self._dejittered:
                corrected_txt.append("Dejittered")
            if self._interpolated:
                corrected_txt.append("Interpolated")
            if self._rereferenced:
                corrected_txt.append("Rereferenced")
            if len(corrected_txt) != 0:
                append_txt[-1] += " (" + ", ".join(corrected_txt) + ")"
        if self._randomized:
            append_txt.append("Randomized")
        if len(append_txt) != 0:
            txt += " · " + " · ".join(append_txt)
        return txt

    def __eq__(self, other):
        """Returns `True` if the attributes :attr:`x`, :attr:`y`, :attr:`z` and :attr:`joint_label` are identical
        between the two :class:`Joint` objects (with a precision of 5 decimal places).

        .. versionadded:: 2.0

        Parameters
        ----------
        other: Joint
            Another :class:`Joint` object.

        Examples
        --------
        >>> joint_1 = Joint("Head", 1, 2, 3)
        >>> joint_2 = Joint("Head", 1, 2, 3)
        >>> joint_1 == joint_2
        True
        >>> joint_3 = Joint("Neck", 1, 2, 3)
        >>> joint_1 == joint_3
        False
        >>> joint_4 = Joint("Head", 1, 2, 4)
        >>> joint_1 == joint_4
        False
        >>> joint_5 = joint_1.copy()
        >>> joint_1 == joint_5
        True
        """
        if self.x is not None or other.x is not None:
            if (self.x is None and other.x is not None) or (self.x is not None and other.x is None) or \
                not np.isclose(self.x, other.x, equal_nan=True):
                return False

        if self.y is not None or other.y is not None:
            if (self.y is None and other.y is not None) or (self.y is not None and other.y is None) or \
                not np.isclose(self.y, other.y, equal_nan=True):
                return False

        if self.z is not None or other.z is not None:
            if (self.z is None and other.z is not None) or (self.z is not None and other.z is None) or \
                not np.isclose(self.z, other.z, equal_nan=True):
                return False

        if not self.joint_label == other.joint_label:
            return False

        return True

    def __getitem__(self, key):
        """Returns the value of the coordinate :attr:`x`, :attr:`y` or :attr:`z` depending on the value of `key`. This
        function is a wrapper for :meth:`get_coordinate()`.

        .. versionadded:: 2.0

        Parameters
        ----------
        key: str
            A coordinate, either ``"x"``, ``"y"`` or ``"z"``.

        Returns
        -------
        float|None
            The value of the coordinate.

        Raises
        ------
        InvalidParameterValueException
            If the key is neither ``"x"``, ``"y"`` nor ``"z"``.

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint["x"]
        1
        """
        return self.get_coordinate(key)

    def __setitem__(self, key, value):
        """Sets the value of the coordinate :attr:`x`, :attr:`y` or :attr:`z`. This function is a wrapper for
        :meth:`set_position()`.

        .. versionadded:: 2.0

        Parameters
        ----------
        key: str
            The coordinate, either ``"x"``, ``"y"`` or ``"z"``.
        value: float|None
            The value of the coordinate.

        Raises
        ------
        InvalidParameterValueException
            If the key is neither ``"x"``, ``"y"`` nor ``"z"``.

        Example
        -------
        >>> joint = Joint("Head", 1, 2, 3)
        >>> joint["x"] = 4
        >>> joint.get_x()
        4
        """
        self.set_coordinate(key, value)
