"""Code generation for parser modules from grammar rules.

This module provides functions to generate Python source code for a parser
from a set of grammar rules. It emits parser functions, rule closures, and
trivia parsing logic for whitespace and comments.
"""

from importlib.metadata import version

from pest.grammar.codegen.builder import Builder
from pest.grammar.rule import BuiltInRule
from pest.grammar.rule import Rule

VERSION = version("python-pest")

PRELUDE = f"""\
\"\"\"This file was generated by Python Pest version {VERSION}.

Usage:

    from module_name import Rule
    from module_name import parse

    parse_tree = parse(Rule.START_RULE, "input text")

Running this module directly exposes a basic command line interface for
debugging grammars.
    
    python -m module_name --help
\"\"\"

from __future__ import annotations

from enum import StrEnum
from typing import TYPE_CHECKING

import regex as re

from pest.exceptions import PestParsingError
from pest.pairs import Pair
from pest.pairs import Pairs
from pest.state import ParserState
from pest.state import RuleFrame

if TYPE_CHECKING:
    from collections.abc import Callable

# ruff: noqa: D103 N802 N816 N806 PLR0912 PLR0915 PLR2004
"""


def generate_module(rules: dict[str, Rule]) -> str:
    """Generate the complete Python source code for a parser module.

    Args:
        rules: A dictionary mapping rule names to Rule objects.

    Returns:
        The generated Python source code as a string, including all parser
        functions and trivia parsing logic.
    """
    generated_rules = "\n\n".join(
        generate_rule(name, rules)
        for name, rule in rules.items()
        if not isinstance(rule, BuiltInRule) or name == "EOI"
    )
    return "\n\n".join(
        [
            PRELUDE,
            generate_rule_enum(rules),
            generated_rules,
            generate_parse_trivia(rules),
            generate_rule_map(rules),
            generate_parse_entry_point(),
            generate_cli(),
        ]
    )


def generate_rule(name: str, rules: dict[str, Rule]) -> str:
    """Generate the full parser function for a single grammar rule.

    Returns the source of a top-level assignment:
        parse_<rule> = _parse_<rule>()

    The generated closure includes:
      - rule-local constants (regexes, tables, etc.)
      - a RuleFrame instance for the rule
      - the inner parser function implementing the rule body

    Args:
        name: The name of the rule to generate.
        rules: A dictionary mapping rule names to Rule objects.

    Returns:
        The generated Python source code for the rule as a string.
    """
    rule = rules[name]
    inner_gen = Builder(rules)
    pairs_var = "pairs"
    rule.generate(inner_gen, "matched", pairs_var)

    gen = Builder()
    func_name = f"parse_{rule.name}"

    gen.writeln(f"def _{func_name}() -> Callable[[ParserState, list[Pair]], bool]:")
    with gen.block():
        # Emit rule-scoped constants (regexes, tables, etc.)
        if inner_gen.rule_constants:
            for const_name, expr in inner_gen.rule_constants:
                gen.writeln(f"{const_name} = {expr}")
            gen.writeln("")  # spacer after constants

        # Each closure has its own RuleFrame
        gen.writeln(f"rule_frame = RuleFrame({rule.name!r}, {rule.modifier})")
        gen.writeln("")

        # Inline the inner function body generated earlier
        for line in inner_gen.lines:
            gen.writeln(line)
        gen.writeln("")

        # Expose the inner parser function
        gen.writeln("return inner")
        gen.writeln("")

    # At module scope, instantiate the closure
    gen.writeln(f"{func_name} = _{func_name}()")

    return gen.render()


def generate_parse_trivia(rules: dict[str, Rule]) -> str:
    """Generate a `parse_trivia` function that parses implicit rules.

    The generated function parses whitespace and comments according to the
    presence of `WHITESPACE`, `COMMENT`, or the optimized `SKIP` rule in
    the grammar. If none are present, the function is a no-op.

    Args:
        rules: A dictionary mapping rule names to Rule objects.

    Returns:
        The generated Python source code for trivia parsing as a string.
    """
    has_skip = "SKIP" in rules
    has_ws = "WHITESPACE" in rules
    has_comment = "COMMENT" in rules

    gen = Builder()
    gen.writeln("def parse_trivia(state: ParserState, pairs: list[Pair]) -> bool:")
    with gen.block():
        if not (has_skip or has_ws or has_comment):
            # Nothing to do
            gen.writeln("return True")
            return gen.render()

        gen.writeln("if state.atomic_depth > 0:")
        with gen.block():
            gen.writeln("return True")

        if has_skip:
            gen.writeln("return parse_SKIP(state, pairs)")
            return gen.render()

        gen.writeln("with state.suppress_failures():")
        with gen.block():
            gen.writeln("while True:")
            with gen.block():
                gen.writeln("state.checkpoint()")
                gen.writeln("matched = False")

                if has_ws:
                    gen.writeln("matched = parse_WHITESPACE(state, pairs)")
                    gen.writeln("if matched:")
                    with gen.block():
                        gen.writeln("state.ok()")
                        gen.writeln("continue")
                    gen.writeln("state.restore()")

                if has_comment:
                    gen.writeln("state.checkpoint()")
                    gen.writeln("matched = parse_COMMENT(state, pairs)")
                    gen.writeln("if matched:")
                    with gen.block():
                        gen.writeln("state.ok()")
                    gen.writeln("else:")
                    with gen.block():
                        gen.writeln("state.restore()")

                gen.writeln("if not matched:")
                with gen.block():
                    gen.writeln("break")

        gen.writeln("return True")

    return gen.render()


def generate_rule_enum(rules: dict[str, Rule]) -> str:
    """Generate a `StrEnum` class definition with one item per rule."""
    gen = Builder()
    gen.writeln("class Rule(StrEnum):")
    with gen.block():
        gen.writeln('"""Grammar rules."""')
        for name, rule in rules.items():
            if not isinstance(rule, BuiltInRule) or rule.name == "EOI":
                gen.writeln(f"{name.upper()} = {name!r}")

    return gen.render()


def generate_rule_map(rules: dict[str, Rule]) -> str:
    """Generate a dictionary mapping rule names to `parse_*` callables."""
    gen = Builder()
    gen.writeln("_RULE_MAP: dict[str, Callable[[ParserState, list[Pair]], bool]] = {")
    with gen.block():
        for name, rule in rules.items():
            if not isinstance(rule, BuiltInRule) or rule.name == "EOI":
                gen.writeln(f"{name!r}: parse_{name},")
    gen.writeln("}")
    return gen.render()


_PARSE_DOC = '''\
    """Parse the given `text` starting from the specified `start_rule`.

    Args:
        start_rule: The name of the rule to start parsing from.
        text: The input string to parse.
        start_pos: The position in the input string to start parsing from (default: 0).

    Returns:
        Pairs: The parse tree as a `Pairs` object.

    Raises:
        KeyError: If `start_rule` is not a valid rule name.
        PestParsingError: If the input `text` cannot be parsed according to the grammar.
    """'''


def generate_parse_entry_point() -> str:
    """Generate a `parse` function."""
    gen = Builder()
    gen.writeln(
        "def parse(start_rule: str, text: str, *, start_pos: int = 0) -> Pairs:"
    )

    gen.writeln(_PARSE_DOC)

    with gen.block():
        gen.writeln("state = ParserState(text, start_pos)")
        gen.writeln("pairs: list[Pair] = []")
        gen.writeln("matched = _RULE_MAP[start_rule](state, pairs)")

        gen.writeln("if matched:")
        with gen.block():
            gen.writeln("return Pairs(pairs)")

        gen.writeln("raise PestParsingError(state)")

    gen.writeln("\nclass Parser:")
    with gen.block():
        gen.writeln('"""A class wrapping `parse()` in `Parser.parse()`."""')
        gen.writeln(
            "def parse("
            "self, start_rule: str, text: str, *, start_pos: int = 0"
            ") -> Pairs:"
        )
        with gen.block():
            gen.writeln(
                '"""Parse the given `text` starting from the specified `start_rule`."""'
            )
            gen.writeln("return parse(start_rule, text, start_pos=start_pos)")

    return gen.render()


CLI = """\
def main() -> None:
    parser = argparse.ArgumentParser(
        prog="parser",
        description="A parser generated by Python pest.",
    )

    parser.add_argument(
        "rule",
        metavar="RULE",
        help="The name of the rule to start parsing from.",
    )

    group = parser.add_mutually_exclusive_group(required=True)

    group.add_argument(
        "-i",
        "--input",
        metavar="INPUT",
        help="Input text to parse.",
    )

    group.add_argument(
        "-f",
        "--file",
        metavar="FILE",
        type=argparse.FileType("r"),
        help="Path to a file containing text to parse, or `-` to read from stdin.",
    )

    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Display verbose output",
    )

    args = parser.parse_args()
    text = args.input or args.file.read()

    print(  # noqa: T201
        parse(args.rule, text).dumps(compact=not args.verbose)
    )

if __name__ == "__main__":
    import argparse
    main()
"""


def generate_cli() -> str:
    """Generate a command line interface for testing the generated parser."""
    return CLI
