"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
    }) {
        super(scope, id);
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: props.validationSettings,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.0.1-alpha.1" };
//# sourceMappingURL=data:application/json;base64,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