"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('IAM policy document', () => {
    test('the Permission class is a programming model for iam', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addActions('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResources('myQueue');
        p.addResources('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${core_1.Token.asString({ account: 'account' })}name`);
        p.addAccountCondition('12221121221');
        expect(stack.resolve(p.toStatementJson())).toEqual({ Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: { AWS: { 'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']] } },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } } });
    });
    test('the PolicyDocument class is a dom for iam policy documents', () => {
        const stack = new core_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addActions('sqs:SendMessage');
        p1.addNotResources('arn:aws:sqs:us-east-1:123456789012:forbidden_queue');
        const p2 = new lib_1.PolicyStatement();
        p2.effect = lib_1.Effect.DENY;
        p2.addActions('cloudformation:CreateStack');
        const p3 = new lib_1.PolicyStatement();
        p3.effect = lib_1.Effect.ALLOW;
        p3.addNotActions('cloudformation:UpdateTerminationProtection');
        const p4 = new lib_1.PolicyStatement();
        p4.effect = lib_1.Effect.DENY;
        p4.addNotPrincipals(new lib_1.CanonicalUserPrincipal('OnlyAuthorizedUser'));
        doc.addStatements(p1);
        doc.addStatements(p2);
        doc.addStatements(p3);
        doc.addStatements(p4);
        expect(stack.resolve(doc)).toEqual({
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', NotResource: 'arn:aws:sqs:us-east-1:123456789012:forbidden_queue' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' },
                { Effect: 'Allow', NotAction: 'cloudformation:UpdateTerminationProtection' },
                { Effect: 'Deny', NotPrincipal: { CanonicalUser: 'OnlyAuthorizedUser' } }]
        });
    });
    test('Cannot combine Actions and NotActions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                actions: ['abc:def'],
                notActions: ['abc:def'],
            });
        }).toThrow(/Cannot add 'NotActions' to policy statement if 'Actions' have been added/);
    });
    test('Throws with invalid actions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                actions: ['service:action', '*', 'service:acti*', 'in:val:id'],
            });
        }).toThrow(/Action 'in:val:id' is invalid/);
    });
    test('Throws with invalid not actions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                notActions: ['service:action', '*', 'service:acti*', 'in:val:id'],
            });
        }).toThrow(/Action 'in:val:id' is invalid/);
    });
    test('Cannot combine Resources and NotResources', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                resources: ['abc'],
                notResources: ['def'],
            });
        }).toThrow(/Cannot add 'NotResources' to policy statement if 'Resources' have been added/);
    });
    test('Cannot add NotPrincipals when Principals exist', () => {
        const stmt = new lib_1.PolicyStatement({
            principals: [new lib_1.CanonicalUserPrincipal('abc')],
        });
        expect(() => {
            stmt.addNotPrincipals(new lib_1.CanonicalUserPrincipal('def'));
        }).toThrow(/Cannot add 'NotPrincipals' to policy statement if 'Principals' have been added/);
    });
    test('Cannot add Principals when NotPrincipals exist', () => {
        const stmt = new lib_1.PolicyStatement({
            notPrincipals: [new lib_1.CanonicalUserPrincipal('abc')],
        });
        expect(() => {
            stmt.addPrincipals(new lib_1.CanonicalUserPrincipal('def'));
        }).toThrow(/Cannot add 'Principals' to policy statement if 'NotPrincipals' have been added/);
    });
    test('Permission allows specifying multiple actions upon construction', () => {
        const stack = new core_1.Stack();
        const perm = new lib_1.PolicyStatement();
        perm.addResources('MyResource');
        perm.addActions('Action1', 'Action2', 'Action3');
        expect(stack.resolve(perm.toStatementJson())).toEqual({
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource'
        });
    });
    test('PolicyDoc resolves to undefined if there are no permissions', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        expect(stack.resolve(p)).toBeUndefined();
    });
    test('canonicalUserPrincipal adds a principal to a policy with the passed canonical user id', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = 'averysuperduperlongstringfor';
        p.addPrincipals(new lib_1.CanonicalUserPrincipal(canoncialUser));
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                CanonicalUser: canoncialUser,
            },
        });
    });
    test('addAccountRootPrincipal adds a principal with the current account root', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::',
                            { Ref: 'AWS::AccountId' },
                            ':root',
                        ],
                    ],
                },
            },
        });
    });
    test('addFederatedPrincipal adds a Federated principal with the passed value', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal('com.amazon.cognito', { StringEquals: { key: 'value' } });
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                Federated: 'com.amazon.cognito',
            },
            Condition: {
                StringEquals: { key: 'value' },
            },
        });
    });
    test('addAwsAccountPrincipal can be used multiple times', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] },
                ],
            },
        });
    });
    describe('hasResource', () => {
        test('false if there are no resources', () => {
            expect(new lib_1.PolicyStatement().hasResource).toEqual(false);
        });
        test('true if there is one resource', () => {
            expect(new lib_1.PolicyStatement({ resources: ['one-resource'] }).hasResource).toEqual(true);
        });
        test('true for multiple resources', () => {
            const p = new lib_1.PolicyStatement();
            p.addResources('r1');
            p.addResources('r2');
            expect(p.hasResource).toEqual(true);
        });
    });
    describe('hasPrincipal', () => {
        test('false if there is no principal', () => {
            expect(new lib_1.PolicyStatement().hasPrincipal).toEqual(false);
        });
        test('true if there is a principal', () => {
            const p = new lib_1.PolicyStatement();
            p.addArnPrincipal('bla');
            expect(p.hasPrincipal).toEqual(true);
        });
        test('true if there is a notPrincipal', () => {
            const p = new lib_1.PolicyStatement();
            p.addNotPrincipals(new lib_1.CanonicalUserPrincipal('test'));
            expect(p.hasPrincipal).toEqual(true);
        });
    });
    test('statementCount returns the number of statement in the policy document', () => {
        const p = new lib_1.PolicyDocument();
        expect(p.statementCount).toEqual(0);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'] }));
        expect(p.statementCount).toEqual(1);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['service:action2'] }));
        expect(p.statementCount).toEqual(2);
    });
    describe('{ AWS: "*" } principal', () => {
        test('is represented as `Anyone`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.Anyone()] }));
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' },
                ],
                Version: '2012-10-17',
            });
        });
        test('is represented as `AnyPrincipal`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.AnyPrincipal()] }));
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' },
                ],
                Version: '2012-10-17',
            });
        });
        test('is represented as `addAnyPrincipal`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const s = new lib_1.PolicyStatement();
            s.addAnyPrincipal();
            p.addStatements(s);
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' },
                ],
                Version: '2012-10-17',
            });
        });
    });
    test('addResources() will not break a list-encoded Token', () => {
        const stack = new core_1.Stack();
        const statement = new lib_1.PolicyStatement();
        statement.addActions(...core_1.Lazy.listValue({ produce: () => ['a', 'b', 'c'] }));
        statement.addResources(...core_1.Lazy.listValue({ produce: () => ['x', 'y', 'z'] }));
        expect(stack.resolve(statement.toStatementJson())).toEqual({
            Effect: 'Allow',
            Action: ['a', 'b', 'c'],
            Resource: ['x', 'y', 'z'],
        });
    });
    test('addCanonicalUserPrincipal can be used to add cannonical user principals', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        const s1 = new lib_1.PolicyStatement();
        s1.addCanonicalUserPrincipal('cannonical-user-1');
        const s2 = new lib_1.PolicyStatement();
        s2.addPrincipals(new lib_1.CanonicalUserPrincipal('cannonical-user-2'));
        p.addStatements(s1);
        p.addStatements(s2);
        expect(stack.resolve(p)).toEqual({
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } },
            ],
            Version: '2012-10-17',
        });
    });
    test('addPrincipal correctly merges array in', () => {
        const stack = new core_1.Stack();
        const arrayPrincipal = {
            get grantPrincipal() { return this; },
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: new lib_1.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
            addToPolicy() { return false; },
            addToPrincipalPolicy() { return { statementAdded: false }; },
        };
        const s = new lib_1.PolicyStatement();
        s.addAccountRootPrincipal();
        s.addPrincipals(arrayPrincipal);
        expect(stack.resolve(s.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar',
                ],
            },
        });
    });
    // https://github.com/aws/aws-cdk/issues/1201
    test('policy statements with multiple principal types can be created using multiple addPrincipal calls', () => {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addArnPrincipal('349494949494');
        s.addServicePrincipal('test.service');
        s.addResources('resource');
        s.addActions('action');
        expect(stack.resolve(s.toStatementJson())).toEqual({
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource',
        });
    });
    describe('Service principals', () => {
        test('regional service principals resolve appropriately', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com');
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' },
            });
        });
        test('regional service principals resolve appropriately (with user-set region)', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' },
            });
        });
        test('obscure service principals resolve to the user-provided value', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('test.service-principal.dev');
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' },
            });
        });
    });
    describe('CompositePrincipal can be used to represent a principal that has multiple types', () => {
        test('with a single principal', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(stack.resolve(statement.toStatementJson())).toEqual({ Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
        });
        test('conditions are not allowed on individual principals of a composite', () => {
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('i:am'));
            expect(() => p.addPrincipals(new lib_1.FederatedPrincipal('federated', { StringEquals: { 'aws:some-key': 'some-value' } })))
                .toThrow(/Components of a CompositePrincipal must not have conditions/);
        });
        test('principals and conditions are a big nice merge', () => {
            const stack = new core_1.Stack();
            // add via ctor
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('i:am:an:arn'), new lib_1.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_1.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            // add via policy statement
            statement.addArnPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: '123' });
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Condition: {
                    cond2: { boom: '123' },
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                },
            });
        });
        test('cannot mix types of assumeRoleAction in a single composite', () => {
            // GIVEN
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            expect(() => p.addPrincipals(new lib_1.FederatedPrincipal('fed', {}, 'sts:Boom')))
                .toThrow(/Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
        });
    });
    describe('PrincipalWithConditions can be used to add a principal with conditions', () => {
        test('includes conditions from both the wrapped principal and the wrapper', () => {
            const stack = new core_1.Stack();
            const principalOpts = {
                conditions: {
                    BinaryEquals: {
                        'principal-key': 'SGV5LCBmcmllbmQh',
                    },
                },
            };
            const p = new lib_1.ServicePrincipal('s3.amazonaws.com', principalOpts)
                .withConditions({ StringEquals: { 'wrapper-key': ['val-1', 'val-2'] } });
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Condition: {
                    BinaryEquals: { 'principal-key': 'SGV5LCBmcmllbmQh' },
                    StringEquals: { 'wrapper-key': ['val-1', 'val-2'] },
                },
                Effect: 'Allow',
                Principal: {
                    Service: 's3.amazonaws.com',
                },
            });
        });
        test('conditions from addCondition are merged with those from the principal', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.AccountPrincipal('012345678900').withConditions({ StringEquals: { key: 'val' } });
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            statement.addCondition('Null', { 'banned-key': 'true' });
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Effect: 'Allow',
                Principal: { AWS: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::012345678900:root']] } },
                Condition: { StringEquals: { key: 'val' }, Null: { 'banned-key': 'true' } },
            });
        });
        test('adding conditions via `withConditions` does not affect the original principal', () => {
            const originalPrincipal = new lib_1.ArnPrincipal('iam:an:arn');
            const principalWithConditions = originalPrincipal.withConditions({ StringEquals: { key: 'val' } });
            expect(originalPrincipal.policyFragment.conditions).toEqual({});
            expect(principalWithConditions.policyFragment.conditions).toEqual({ StringEquals: { key: 'val' } });
        });
        test('conditions are merged when operators conflict', () => {
            const p = new lib_1.FederatedPrincipal('fed', {
                OperatorOne: { 'fed-key': 'fed-val' },
                OperatorTwo: { 'fed-key': 'fed-val' },
                OperatorThree: { 'fed-key': 'fed-val' },
            }).withConditions({
                OperatorTwo: { 'with-key': 'with-val' },
                OperatorThree: { 'with-key': 'with-val' },
            });
            const statement = new lib_1.PolicyStatement();
            statement.addCondition('OperatorThree', { 'add-key': 'add-val' });
            statement.addPrincipals(p);
            expect(statement.toStatementJson()).toEqual({
                Effect: 'Allow',
                Principal: { Federated: 'fed' },
                Condition: {
                    OperatorOne: { 'fed-key': 'fed-val' },
                    OperatorTwo: { 'fed-key': 'fed-val', 'with-key': 'with-val' },
                    OperatorThree: { 'fed-key': 'fed-val', 'with-key': 'with-val', 'add-key': 'add-val' },
                },
            });
        });
        test('tokens can be used in conditions', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const statement = new lib_1.PolicyStatement();
            // WHEN
            const p = new lib_1.ArnPrincipal('arn:of:principal').withConditions({
                StringEquals: core_1.Lazy.anyValue({ produce: () => ({ goo: 'zar' }) }),
            });
            statement.addPrincipals(p);
            // THEN
            const resolved = stack.resolve(statement.toStatementJson());
            expect(resolved).toEqual({
                Condition: {
                    StringEquals: {
                        goo: 'zar',
                    },
                },
                Effect: 'Allow',
                Principal: {
                    AWS: 'arn:of:principal',
                },
            });
        });
        test('conditions cannot be merged if they include tokens', () => {
            const p = new lib_1.FederatedPrincipal('fed', {
                StringEquals: { foo: 'bar' },
            }).withConditions({
                StringEquals: core_1.Lazy.anyValue({ produce: () => ({ goo: 'zar' }) }),
            });
            const statement = new lib_1.PolicyStatement();
            expect(() => statement.addPrincipals(p)).toThrow(/multiple "StringEquals" conditions cannot be merged if one of them contains an unresolved token/);
        });
        test('values passed to `withConditions` overwrite values from the wrapped principal ' +
            'when keys conflict within an operator', () => {
            const p = new lib_1.FederatedPrincipal('fed', {
                Operator: { key: 'p-val' },
            }).withConditions({
                Operator: { key: 'with-val' },
            });
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(statement.toStatementJson()).toEqual({
                Effect: 'Allow',
                Principal: { Federated: 'fed' },
                Condition: {
                    Operator: { key: 'with-val' },
                },
            });
        });
    });
    describe('duplicate statements', () => {
        test('without tokens', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement = new lib_1.PolicyStatement();
            statement.addResources('resource1', 'resource2');
            statement.addActions('action1', 'action2');
            statement.addServicePrincipal('service');
            statement.addConditions({
                a: {
                    b: 'c',
                },
                d: {
                    e: 'f',
                },
            });
            // WHEN
            p.addStatements(statement);
            p.addStatements(statement);
            p.addStatements(statement);
            // THEN
            expect(stack.resolve(p).Statement).toHaveLength(1);
        });
        test('with tokens', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement1 = new lib_1.PolicyStatement();
            statement1.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement1.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            const statement2 = new lib_1.PolicyStatement();
            statement2.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement2.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            // WHEN
            p.addStatements(statement1);
            p.addStatements(statement2);
            // THEN
            expect(stack.resolve(p).Statement).toHaveLength(1);
        });
    });
    test('autoAssignSids enables auto-assignment of a unique SID for each statement', () => {
        // GIVEN
        const doc = new lib_1.PolicyDocument({
            assignSids: true,
        });
        // WHEN
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action2'], resources: ['resource2'] }));
        // THEN
        const stack = new core_1.Stack();
        expect(stack.resolve(doc)).toEqual({
            Version: '2012-10-17',
            Statement: [
                { Action: 'service:action1', Effect: 'Allow', Resource: 'resource1', Sid: '0' },
                { Action: 'service:action2', Effect: 'Allow', Resource: 'resource2', Sid: '1' },
            ],
        });
    });
    test('constructor args are equivalent to mutating in-place', () => {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addActions('service:action1', 'service:action2');
        s.addAllResources();
        s.addArnPrincipal('arn');
        s.addCondition('key', { equals: 'value' });
        const doc1 = new lib_1.PolicyDocument();
        doc1.addStatements(s);
        const doc2 = new lib_1.PolicyDocument();
        doc2.addStatements(new lib_1.PolicyStatement({
            actions: ['service:action1', 'service:action2'],
            resources: ['*'],
            principals: [new lib_1.ArnPrincipal('arn')],
            conditions: {
                key: { equals: 'value' },
            },
        }));
        expect(stack.resolve(doc1)).toEqual(stack.resolve(doc2));
    });
    describe('fromJson', () => {
        test("throws error when Statement isn't an array", () => {
            expect(() => {
                lib_1.PolicyDocument.fromJson({
                    Statement: 'asdf',
                });
            }).toThrow(/Statement must be an array/);
        });
    });
    test('adding another condition with the same operator does not delete the original', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addCondition('StringEquals', { 'kms:ViaService': 'service' });
        p.addAccountCondition('12221121221');
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Condition: { StringEquals: { 'kms:ViaService': 'service', 'sts:ExternalId': '12221121221' } },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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