"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyType = exports.Policy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
class PolicyBase extends core_1.Resource {
}
class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope, id, policyId) {
        return Policy.fromPolicyAttributes(scope, id, { policyId });
    }
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope, id, attrs) {
        class Import extends PolicyBase {
            constructor(policyType, policyId) {
                super(scope, id);
                this.policyType = policyType;
                this.policyId = policyId;
            }
        }
        const policyType = attrs.policyType ?? PolicyType.STATIC;
        return new Import(policyType, attrs.policyId);
    }
    /**
     * Create a policy based on a file containing a cedar policy. Best practice would be
     * for the file name to end in `.cedar` but this is not required. Policy is parsed for valid
     * syntax but not validated against schema. In order to validate against schema, use
     * `PolicyStore.addPoliciesFromPath()`
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param props A `StaticPolicyFromFileProps` object.
     */
    static fromFile(scope, id, props) {
        const policyFileContents = fs.readFileSync(props.path).toString();
        (0, cedar_helpers_1.checkParsePolicy)(policyFileContents);
        return new Policy(scope, id, {
            definition: {
                static: {
                    statement: policyFileContents,
                    description: props.description || `${props.path} - Created by CDK`,
                },
            },
            policyStore: props.policyStore,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        // validations
        if (props.definition.static && props.definition.templateLinked) {
            throw new Error('Policy can either be static or templateLinked');
        }
        let definition;
        if (props.definition.static) {
            (0, cedar_helpers_1.checkParsePolicy)(props.definition.static.statement);
            definition = {
                static: {
                    ...props.definition.static,
                    statement: props.definition.static.statement,
                },
            };
        }
        else if (props.definition.templateLinked) {
            definition = {
                templateLinked: {
                    policyTemplateId: props.definition.templateLinked.policyTemplate.policyTemplateId,
                    principal: props.definition.templateLinked.principal,
                    resource: props.definition.templateLinked.resource,
                },
            };
        }
        else {
            throw new Error('Policy must either be static or templateLinked');
        }
        // resource
        this.policy = new aws_verifiedpermissions_1.CfnPolicy(this, id, {
            definition: definition,
            policyStoreId: props.policyStore.policyStoreId,
        });
        // assign construct props
        this.policyId = this.policy.attrPolicyId;
        this.policyType = props.definition.static
            ? PolicyType.STATIC
            : PolicyType.TEMPLATELINKED;
        this.definition = props.definition;
        this.policyStoreId = props.policyStore.policyStoreId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@cdklabs/cdk-verified-permissions.Policy", version: "0.0.4" };
/**
 * PolicyType options
 */
var PolicyType;
(function (PolicyType) {
    PolicyType["STATIC"] = "Static";
    PolicyType["TEMPLATELINKED"] = "TemplateLinked";
})(PolicyType || (exports.PolicyType = PolicyType = {}));
//# sourceMappingURL=data:application/json;base64,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